#include "dismissed_car_watcher.h"

#include <drive/backend/database/drive_api.h>
#include <drive/backend/fines/config.h>

#include <rtline/library/json/cast.h>
#include <rtline/util/algorithm/ptr.h>

#include <util/generic/algorithm.h>

TString TRTDismissedCarWatcher::GetTypeName() {
    return "fines_dismissed_car_watcher";
}

TString TRTDismissedCarWatcherState::GetTypeName() {
    return TRTDismissedCarWatcher::GetTypeName();
}

IRTBackgroundProcess::TFactory::TRegistrator<TRTDismissedCarWatcher> TRTDismissedCarWatcher::Registrator(TRTDismissedCarWatcher::GetTypeName());
TRTHistoryWatcherState::TFactory::TRegistrator<TRTDismissedCarWatcherState> TRTDismissedCarWatcherState::Registrator(TRTDismissedCarWatcherState::GetTypeName());

TRTDismissedCarWatcher::TRTDismissedCarWatcher()
    : TBase()
    , TagEventsHandler(TEventsHandler(NEntityTagsManager::EEntityType::Car))
{
    InitNotifyHandlers();
}

bool TRTDismissedCarWatcher::DoStart(const TRTBackgroundProcessContainer& container) {
    if (!TBase::DoStart(container)) {
        return false;
    }
    if (!NotifyHandlers) {
        return false;
    }
    NotifyHandlers->SetSignalName(GetRTProcessName());
    return true;
}

void TRTDismissedCarWatcher::InitNotifyHandlers() {
    if (!NotifyHandlers) {
        NotifyHandlers = MakeAtomicShared<TNotifyHandlers>(GetType());
    }
}

TExpectedState TRTDismissedCarWatcher::DoExecute(TAtomicSharedPtr<IRTBackgroundProcessState> state, const TExecutionContext& context) const {
    const NDrive::IServer& server = context.GetServerAs<NDrive::IServer>();

    auto nativeState = std::dynamic_pointer_cast<TState>(state);
    const ui64 lastProcessedEventId = (!!nativeState) ? nativeState->GetLastEventId() : std::numeric_limits<ui64>::max();

    if (TagEventsHandler.Fetch(server, lastProcessedEventId, DataActuality)) {
        HandleHistoryEvents(server);
    }

    return BuildState((RetryOnError) ? TagEventsHandler.GetLastProcessedEventId() : TagEventsHandler.GetLastProducedEventId());
}

TAtomicSharedPtr<TRTDismissedCarWatcher::TState> TRTDismissedCarWatcher::BuildState(const ui64 lastEventId) const {
    auto state = MakeAtomicShared<TState>();
    state->SetLastEventId(lastEventId);
    return state;
}

bool TRTDismissedCarWatcher::HandleHistoryEvents(const NDrive::IServer& server) const {
    while (auto eventPtr = TagEventsHandler.Next(server)) {
        TMessagesCollector errors;

        auto status = HandleEvent(server, eventPtr, errors);

        if (errors.HasMessages()) {
            ERROR_LOG << errors.GetStringReport() << Endl;
        }

        NotifyHandlers->Handle(status, server);

        if (status == EHandlingStatus::InvalidSettingValue || status == EHandlingStatus::UpsertError) {
            return false;
        }
    }
    return true;
}

TRTDismissedCarWatcher::EHandlingStatus TRTDismissedCarWatcher::HandleEvent(const NDrive::IServer& server, TAtomicSharedPtr<TCarTagHistoryEvent> eventPtr, TMessagesCollector& errors) const {
    return MarkCarDismissed(server, eventPtr->GetObjectId(), errors);
}

TRTDismissedCarWatcher::EHandlingStatus TRTDismissedCarWatcher::MarkCarDismissed(const NDrive::IServer& server, const TString& carId, TMessagesCollector& /* errors */) const {
    NDrive::NFine::TFineConstructorConfig::TDismissedCarInfo carInfo;
    carInfo.SetSince(ModelingNow());
    {
        carInfo.SetId(carId);

        auto gCars = server.GetDriveAPI()->GetCarsData();
        auto carFetchResult = gCars->FetchInfo(carId, TInstant::Zero());
        auto carPtr = carFetchResult.GetResultPtr(carId);
        if (!!carPtr) {
            carInfo.SetSts(carPtr->GetRegistrationID());
            carInfo.SetVin(carPtr->GetVin());
        }
    }

    const TString settingKey = JoinSeq(".", { NDrive::NFine::TFineConstructorConfig::SettingPrefix, "auto_dismissed_charge_cars" });
    NJson::TJsonValue settingValue = server.GetSettings().GetJsonValue(settingKey);

    {
        TVector<NDrive::NFine::TFineConstructorConfig::TDismissedCarInfo> existingCarInfoCollection;
        if (settingValue.IsDefined() && !NJson::TryFromJson(settingValue, existingCarInfoCollection)) {
            return EHandlingStatus::InvalidSettingValue;
        }

        const bool alreadyMarked = AnyOf(existingCarInfoCollection.cbegin(), existingCarInfoCollection.cend(), [&carInfo](const auto& existingCarInfo) { return existingCarInfo.HasEqualKeyField(carInfo); });
        if (alreadyMarked) {
            return EHandlingStatus::AlreadyMarked;
        }
    }

    settingValue.AppendValue(NJson::ToJson(carInfo));

    if (!server.GetSettings().SetValue(settingKey, settingValue.GetStringRobust(), GetRobotUserId())) {
        return EHandlingStatus::UpsertError;
    }

    return EHandlingStatus::MarkSuccessful;
}

NDrive::TScheme TRTDismissedCarWatcher::DoGetScheme(const IServerBase& server) const {
    auto scheme = TBase::DoGetScheme(server);

    const NDrive::IServer& impl = server.GetAsSafe<NDrive::IServer>();
    scheme.Add<TFSVariants>("tag_names", "Имена тегов")
          .SetVariants(NContainer::Keys(impl.GetDriveDatabase().GetTagsManager().GetTagsMeta().GetRegisteredTags()))
          .SetMultiSelect(true);
    scheme.Add<TFSBoolean>("retry_on_error", "Повторять обработку тега в случае ошибки").SetDefault(true);
    scheme.Add<TFSStructure>("notify_handlers", "Настройки нотификаторов").SetStructure(TNotifyHandlers::GetScheme(impl));

    return scheme;
}

bool TRTDismissedCarWatcher::DoDeserializeFromJson(const NJson::TJsonValue& data) {
    if (!TJsonProcessor::ReadContainer(data, "tag_names", TagNames) || !InitTagEventsHandler(TagNames) || TagEventsHandler.Empty()) {
        return false;
    }
    if (!TJsonProcessor::Read(data, "retry_on_error", RetryOnError)) {
        return false;
    }
    if (!!NotifyHandlers && !NotifyHandlers->DeserializeFromJson(data["notify_handlers"])) {
        return false;
    }
    return TBase::DoDeserializeFromJson(data);
}

NJson::TJsonValue TRTDismissedCarWatcher::DoSerializeToJson() const {
    auto result = TBase::DoSerializeToJson();
    TJsonProcessor::WriteContainerArray(result, "tag_names", TagNames);
    TJsonProcessor::Write(result, "retry_on_error", RetryOnError);
    if (!!NotifyHandlers) {
        TJsonProcessor::WriteSerializable(result, "notify_handlers", *NotifyHandlers);
    }
    return result;
}

bool TRTDismissedCarWatcher::InitTagEventsHandler(const TSet<TString>& tagNames) {
    TagEventsHandler.Clear();
    TagEventsHandler.AddFilter(MakeAtomicShared<TagNameEventsFilter>(EObjectHistoryAction::Add, tagNames));
    return TagEventsHandler.AreFiltersValid();
}
