#include "element_car_watcher.h"


namespace {
    struct TMessagePack {
        TString SuccessTitle;
        TString FailTitle;
        TVector<TString> Messages;

        TMessagePack(TString&& success, TString&& fail, TVector<TString>&& msg)
            : SuccessTitle(success)
            , FailTitle(fail)
            , Messages(msg)
        {
        }
    };

    TMaybe<TMap<TString, TString>> GetLessors(const TSet<TString>& ids, const NDrive::IServer& server, NDrive::TEntitySession& tx) {
        auto attachmentsMap = server.GetDriveAPI()->GetCarAttachmentAssignments().GetAttachmentOfType(ids, EDocumentAttachmentType::CarRegistryDocument, tx);
        if (!attachmentsMap) {
            return {};
        }
        TMap<TString, TString> result;
        for (auto&& [carId, attachments] : *attachmentsMap) {
            if (attachments.empty()) {
                continue;
            }
            if (auto regDocument = dynamic_cast<const TCarRegistryDocument*>(attachments.front().Get())) {
                result.emplace(carId, regDocument->GetLessor());
            }
        }
        return result;
    }
}

TRTElementCarWatcherProcess::TFactory::TRegistrator<TRTElementCarWatcherProcess> TRTElementCarWatcherProcess::Registrator(TRTElementCarWatcherProcess::GetTypeName());

NDrive::TScheme TRTElementCarWatcherProcess::DoGetScheme(const IServerBase& server) const {
    NDrive::TScheme scheme = TBase::DoGetScheme(server);
    scheme.Add<TFSVariants>("notifier", "Способ нотификации").SetVariants(server.GetNotifierNames());
    scheme.Add<TFSBoolean>("dry_run").SetDefault(true);
    scheme.Add<TFSBoolean>("deactivate_absent", "Деактивировать отсутствующие машины").SetDefault(false);
    {
        NDrive::TScheme item;
        item.Add<TFSVariants>("department", "Отделение").SetVariants(GetEnumAllValues<NDrive::EDepartment>()).SetRequired(true);
        item.Add<TFSArray>("lessors", "Отделение").SetElement<TFSString>();
        scheme.Add<TFSArray>("departments", "Отделения").SetElement(item);
    }
    scheme.Add<TFSDuration>("timeout", "Время ожидания ответа от Element").SetDefault(Timeout);
    return scheme;
}

bool TRTElementCarWatcherProcess::DoDeserializeFromJson(const NJson::TJsonValue& jsonInfo) {
    for (auto&& item : jsonInfo["departments"].GetArray()) {
        NDrive::EDepartment department = NDrive::Unknown;
        TSet<TString> lessors;
        if (!NJson::ParseField(item["department"], NJson::Stringify(department), /* required = */ true)
            || !NJson::ParseField(item["lessors"], lessors, /* required = */ true))
        {
            return false;
        }
        for (auto&& lessor : lessors) {
            LessorsMap.emplace(lessor, department);
        }
    }
    return LessorsMap
        && NJson::ParseField(jsonInfo["notifier"], NotifierName, /* required = */ false)
        && NJson::ParseField(jsonInfo["dry_run"], DryRun, /* required = */ true)
        && NJson::ParseField(jsonInfo["deactivate_absent"], DeactivateAbsent, /* required = */ true)
        && NJson::ParseField(jsonInfo["timeout"], Timeout, /* required = */ false)
        && TBase::DoDeserializeFromJson(jsonInfo);
}

NJson::TJsonValue TRTElementCarWatcherProcess::DoSerializeToJson() const {
    NJson::TJsonValue result = TBase::DoSerializeToJson();
    NJson::InsertNonNull(result, "notifier", NotifierName);
    NJson::InsertField(result, "dry_run", DryRun);
    NJson::InsertField(result, "deactivate_absent", DeactivateAbsent);
    NJson::TJsonValue departments = NJson::JSON_ARRAY;
    TMap<NDrive::EDepartment, TSet<TString>> departmentsMap;
    for (auto&& [lessor, department] : LessorsMap) {
        departmentsMap[department].insert(lessor);
    }
    for (const auto& [department, lessors] : departmentsMap) {
        departments.AppendValue(NJson::TMapBuilder
            ("department", ToString(department))
            ("lessors", NJson::ToJson(lessors))
        );
    }
    NJson::InsertField(result, "departments", departments);
    NJson::InsertField(result, "timeout", NJson::Hr(Timeout));
    return result;
}

TExpectedState TRTElementCarWatcherProcess::DoExecuteFiltered(TAtomicSharedPtr<IRTBackgroundProcessState> /* state */, const NDrive::IServer& server, TTagsModificationContext& context) const {
    if (!server.GetDriveAPI()->HasElementFineClient()) {
        return nullptr;
    }
    const auto& carsManager = *Yensured(server.GetDriveAPI()->GetCarsData());
    const auto& client = server.GetDriveAPI()->GetElementFinesClient();
    TMap<TString, NDrive::TElementCarData> elementMap;
    {
        auto elementCarsFuture = client.GetCarsData();
        if (!elementCarsFuture.Wait(client.GetConfig().GetRequestTimeout())) {
            return MakeUnexpected<TString>("Fail to collect cars from element");
        }
        auto elementCars = elementCarsFuture.GetValue();
        for (auto&& car : elementCars) {
            TString vin = car.GetVin();
            elementMap.emplace(std::move(vin), std::move(car));
        }
    }
    TVector<NDrive::TElementCarData> carsToRegister;
    TVector<NDrive::TElementCarData> carsToActivate;
    {
        TMap<TString, TString> lessors;
        {
            auto tx = carsManager.GetHistoryManager().BuildTx<NSQL::ReadOnly>();
            auto map = GetLessors(context.GetFilteredCarIds(), server, tx);
            if (!map) {
                return MakeUnexpected<TString>("Fail to fetch car attachments to registration: " + tx.GetStringReport());
            }
            lessors = std::move(*map);
        }
        TSet<TString> skippedCars;
        TSet<TString> skippedCarsByLessor;
        TSet<TString> unknownLessors;
        for (auto&& [id, carData] : context.GetFetchedCarsData()) {
            if (auto it = elementMap.find(carData.GetVin()); it != elementMap.end()) {
                if (!it->second.GetActivityDef(true)) {
                    carsToActivate.emplace_back(it->second);
                    carsToActivate.back().SetActivity(true);
                }
                elementMap.erase(it);
                continue;
            }
            auto lessor = lessors.FindPtr(id);
            if (!lessor) {
                skippedCarsByLessor.insert(id);
                continue;
            }
            if (auto department = LessorsMap.FindPtr(*lessor)) {
                TString sts = ToString(carData.GetRegistrationID());
                if (sts.size() < 4 || !carData.GetVin()) {
                    skippedCars.insert(id);
                    continue;
                }
                NDrive::TElementCarData item;
                item.SetNumber(carData.GetNumber());
                item.SetStsSeries(sts.substr(0, 4));
                item.SetStsNumber(sts.substr(4));
                item.SetVin(carData.GetVin());
                item.SetModel(carData.GetModel());
                item.SetActivity(true);
                item.SetDepartment(*department);
                carsToRegister.emplace_back(std::move(item));
            } else {
                skippedCarsByLessor.insert(id);
                unknownLessors.insert(*lessor);
            }
        }
        if (unknownLessors) {
            if (NotifierName) {
                NDrive::INotifier::MultiLinesNotify(server.GetNotifier(GetNotifierName()), "Не настроенные лизингодатели: " + ToString(unknownLessors.size()), MakeVector(unknownLessors));
            } else {
                NDrive::TEventLog::Log("ElementUnknownLessors", NJson::TMapBuilder("lessors", NJson::ToJson(unknownLessors)));
            }
        }
        if (skippedCarsByLessor) {
            if (NotifierName) {
                NDrive::INotifier::MultiLinesNotify(server.GetNotifier(GetNotifierName()), "Машины с неизвестными лизингодателями: " + ToString(skippedCarsByLessor.size()), MakeVector(skippedCarsByLessor));
            } else {
                NDrive::TEventLog::Log("ElementSkippedByLessor", NJson::TMapBuilder("cars", NJson::ToJson(skippedCarsByLessor)));
            }
        }
        if (skippedCars) {
            if (NotifierName) {
                NDrive::INotifier::MultiLinesNotify(server.GetNotifier(GetNotifierName()), "Машины с неполными данными: " + ToString(skippedCars.size()), MakeVector(skippedCars));
            } else {
                NDrive::TEventLog::Log("ElementSkipped", NJson::TMapBuilder("cars", NJson::ToJson(skippedCars)));
            }
        }
    }
    TVector<NThreading::TFuture<TVector<NDrive::TElementCarData>>> futures;
    TVector<TMessagePack> notifications;
    if (carsToRegister) {
        TVector<TString> carMessages;
        Transform(carsToRegister.begin(), carsToRegister.end(), std::back_inserter(carMessages), [](const auto& car) { return car.GetVin(); });
        notifications.emplace_back("Добавлено: " + ToString(carsToRegister.size()), "Ошибка при добавлении", std::move(carMessages));
        if (!DryRun) {
            futures.emplace_back(client.AddCars(carsToRegister));
        }
    }
    if (carsToActivate) {
        TVector<TString> carMessages;
        Transform(carsToActivate.begin(), carsToActivate.end(), std::back_inserter(carMessages), [](const auto& car) { return car.GetVin(); });
        notifications.emplace_back("Активировано: " + ToString(carsToActivate.size()), "Ошибка при активации", std::move(carMessages));
        if (!DryRun) {
            futures.emplace_back(client.ChangeCarsActivity(carsToActivate));
        }
    }
    if (elementMap && DeactivateAbsent) {
        TVector<TString> carMessages;
        if (!DryRun) {
            TVector<NDrive::TElementCarData> carsToUpdate;
            for (auto&& [vin, car] : elementMap) {
                carMessages.emplace_back(std::move(vin));
                car.SetActivity(false);
                carsToUpdate.emplace_back(std::move(car));
            }
            futures.emplace_back(client.ChangeCarsActivity(carsToUpdate));
        } else {
            carMessages = MakeVector(NContainer::Keys(elementMap));
        }
        notifications.emplace_back("Деактивировано: " + ToString(carMessages.size()), "Ошибка при деактивации", std::move(carMessages));
    }
    if (futures) {
        auto waiter = NThreading::WaitAll(futures);
        if (!waiter.Wait(Timeout)) {
            NDrive::TEventLog::Log("ElementCarWatcherTimeout", NJson::TMapBuilder("timeout", NJson::ToJson(Timeout))("futures", futures.size()));
        }
        for (size_t i = 0; i < futures.size(); ++i) {
            auto& future = futures[i];
            auto& notification = notifications[i];
            const TString& title = future.HasValue() ? notification.SuccessTitle : notification.FailTitle;
            const auto& msg = future.HasValue() ? notification.Messages : TVector<TString>{ NThreading::GetExceptionMessage(future) };
            if (NotifierName) {
                NDrive::INotifier::MultiLinesNotify(server.GetNotifier(GetNotifierName()), title, msg);
            } else {
                NDrive::TEventLog::Log("ElementCarWatcherAction", NJson::TMapBuilder(title, NJson::ToJson(msg)));
            }
        }
    } else if (notifications) {
        for (auto&& notification : notifications) {
            if (NotifierName) {
                NDrive::INotifier::MultiLinesNotify(server.GetNotifier(GetNotifierName()), notification.SuccessTitle, notification.Messages);
            } else {
                NDrive::TEventLog::Log("ElementCarWatcherAction", NJson::TMapBuilder(notification.SuccessTitle, NJson::ToJson(notification.Messages)));
            }
        }
    }
    return MakeAtomicShared<IRTBackgroundProcessState>();
}
