#include "element_photo_processor.h"

#include <drive/backend/fines/constructors/fine_attachment_constructor.h>
#include <drive/backend/fines/manager.h>

#include <drive/library/cpp/element/client.h>


const TString TRTElementFinePhotoCollector::TypeName = "fines_element_photo_processor";

IRTRegularBackgroundProcess::TFactory::TRegistrator<TRTElementFinePhotoCollector> TRTElementFinePhotoCollector::Registrator(TRTElementFinePhotoCollector::TypeName);
TRTInstantWatcherState::TFactory::TRegistrator<TRTElementFinePhotoCollectorState> TRTElementFinePhotoCollectorState::Registrator(TRTElementFinePhotoCollector::TypeName);


TString TRTElementFinePhotoCollectorState::GetType() const {
    return TRTElementFinePhotoCollector::TypeName;
}


size_t TElementFinePhotoCollectorContext::GetCollectedTotal() const {
    return Photos.size();
}


TRTElementFinePhotoCollector::TRTElementFinePhotoCollector()
    : TBase(nullptr)
{
    InitNotifyHandlers();
}

TString TRTElementFinePhotoCollector::GetType() const {
    return TypeName;
}

TExpectedState TRTElementFinePhotoCollector::DoExecute(TAtomicSharedPtr<IRTBackgroundProcessState> state, const TExecutionContext& context) const {
    const auto& server = context.GetServerAs<NDrive::IServer>();
    if (!server.GetDriveAPI() || !server.GetDriveAPI()->HasFinesManager() || !server.GetDriveAPI()->HasElementFineClient()) {
        ERROR_LOG << "Fines manager or Element client is undefined" << Endl;
        return nullptr;
    }
    return TBase::DoExecute(state, context);
}

NDrive::NFine::TFineFilterGroup TRTElementFinePhotoCollector::GetFilters(const NDrive::IServer& server, const TFineIdToEventPtrMapping& fineIdToEventPtrMapping) const {
    NDrive::NFine::TFineFilterGroup filters = TBase::GetFilters(server, fineIdToEventPtrMapping);
    filters.Append(MakeAtomicShared<NDrive::NFine::TFineExplicitPhotoFilter>());
    return filters;
}

TAtomicSharedPtr<TRTElementFinePhotoCollector::IState> TRTElementFinePhotoCollector::BuildState(const ui64 lastEventId) const {
    auto state = MakeAtomicShared<TState>();
    state->SetLastEventId(lastEventId);
    return state;;
}

TRTElementFinePhotoCollector::IBaseContext::TPtr TRTElementFinePhotoCollector::ConstructContext(const NDrive::IServer& server, const NDrive::NFine::TFineAttachmentConstructor& fineAttachmentConstructor, const TString& fineId, const TString& rulingNumber, TEventsHandler::TEventPtr relatedEventPtr) const {
    return MakeAtomicShared<TContext>(server, fineAttachmentConstructor, fineId, rulingNumber, relatedEventPtr);
}

bool TRTElementFinePhotoCollector::FilterFine(const IBaseContext::TPtr context, TMessagesCollector& errors) const {
    if (!TBase::FilterFine(context, errors)) {
        return false;
    }
    auto existingFinePhotosPtr = context->GetFineAttachmentConstructor().GetKnownPhotos().FindPtr(context->GetFineId());
    return !existingFinePhotosPtr || existingFinePhotosPtr->empty();
}

bool TRTElementFinePhotoCollector::CollectExternalAttachments(IBaseContext::TPtr context, TMessagesCollector& errors) const {
    auto derivedContextPtr = VerifyDynamicCast<TContext*>(context.Get());
    const auto& client = context->GetServer().GetDriveAPI()->GetElementFinesClient();
    auto future = client.GetViolationPhotos(context->GetRulingNumber());
    const auto timeout = client.GetConfig().GetRequestTimeout();
    if (!future.Wait(timeout)) {
        errors.AddMessage(__LOCATION__, "Request timeouted");
        return false;
    }
    if (!future.HasValue()) {
        errors.AddMessage(__LOCATION__, "Request failed with exception: " + NThreading::GetExceptionMessage(future));
        return false;
    }
    derivedContextPtr->SetPhotos(future.ExtractValue());
    return true;
}

bool TRTElementFinePhotoCollector::ConstructNativeFineAttachments(IBaseContext::TPtr context, TMessagesCollector& errors) const {
    auto derivedContextPtr = VerifyDynamicCast<TContext*>(context.Get());
    for (auto&& photo : derivedContextPtr->MutablePhotos()) {
        NDrive::NFine::TAutocodeFineAttachmentEntry attachmentEntry;
        if (!context->GetFineAttachmentConstructor().ConstructAutocodeFineAttachmentEntry(context->GetFineId(), photo, attachmentEntry, errors)) {
            return false;
        }
        context->MutableAttachmentEntries().push_back(std::move(attachmentEntry));
    }
    return true;
}
