#include "element_processor.h"

#include <drive/backend/database/transaction/assert.h>

#include <drive/library/cpp/element/client.h>


TElementFinesProcessingContext::TElementFinesProcessingContext(const NDrive::IServer& server
    , const NDrive::NFine::TFineConstructor& fineConstructor
    , const TPrefechedFinesContainer& prefechedFines
    , TVector<TExternalFine>&& externalFines
)
    : TBase(server, fineConstructor, prefechedFines)
    , ExternalFines(std::move(externalFines))
{
}


const TString TRTElementFinesCollector::TypeName = "fines_element_processor";
const ui32 TRTElementFinesCollector::DefaultMaxRequestFinesCount = 1000000;


IRTRegularBackgroundProcess::TFactory::TRegistrator<TRTElementFinesCollector> TRTElementFinesCollector::Registrator(TRTElementFinesCollector::TypeName);
TRTInstantWatcherState::TFactory::TRegistrator<TRTElementFinesCollectorState> TRTElementFinesCollectorState::Registrator(TRTElementFinesCollector::TypeName);

TString TRTElementFinesCollectorState::GetType() const {
    return TRTElementFinesCollector::TypeName;
}

TRTElementFinesCollector::TRTElementFinesCollector()
    : TBase()
{
    InitNotifyHandlers();
}

TString TRTElementFinesCollector::GetType() const {
    return TypeName;
}

NDrive::NFine::ESourceType TRTElementFinesCollector::GetSourceType() const {
    return NDrive::NFine::ESourceType::Element;
}

NDrive::TScheme TRTElementFinesCollector::DoGetScheme(const IServerBase& server) const {
    auto scheme = TBase::DoGetScheme(server);
    scheme.Add<TFSNumeric>("min_ruling_date", "Дата, с которой забираются штрафы; все если не задано").SetVisual(TFSNumeric::EVisualType::DateTime);
    scheme.Add<TFSArray>("sts_to_check", "STS автомобилей для проверки; все если не задано").SetElement<TFSString>();
    scheme.Add<TFSBoolean>("exclude_sts", "Для всех STS, кроме указанных").SetDefault(false);
    scheme.Add<TFSNumeric>("max_request_fines_count", "Ограничение на кол-во единовременно запрашиваемых штрафов; без ограничений если ноль").SetMin(0).SetMax(DefaultMaxRequestFinesCount).SetDefault(0);
    scheme.Add<TFSVariants>("penalty_check_policy", "Способ сбора штрафов, по умолчанию все").SetVariants(GetEnumAllValues<EPenaltyCheckPolicy>()).SetDefault(ToString(EPenaltyCheckPolicy::All));
    scheme.Add<TFSBoolean>("is_update_allowed", "Разрешено ли дообновление информации о штрафах").SetDefault(false);
    scheme.Add<TFSBoolean>("unknown_article_only", "Дообновлять только с неизвестной статьей нарушения").SetDefault(false);
    return scheme;
}

bool TRTElementFinesCollector::DoDeserializeFromJson(const NJson::TJsonValue& data) {
    return TBase::DoDeserializeFromJson(data)
        && NJson::ParseField(data, "min_ruling_date", MinRulingDate, /* required = */ false)
        && NJson::ParseField(data, "max_request_fines_count", MaxRequestFinesCount, /* required = */ false)
        && NJson::ParseField(data, "sts_to_check", StsToCheck, /* required = */ false)
        && NJson::ParseField(data, "exclude_sts", ExcludeSts, /* required = */ false)
        && NJson::ParseField(data, "penalty_check_policy", NJson::Stringify(PenaltyCheckPolicy), /* required = */ false)
        && NJson::ParseField(data, "is_update_allowed", UpdateAllowedFlag, /* required = */ false)
        && NJson::ParseField(data, "unknown_article_only", UpdateUnknownArticleOnly, /* required = */ false);
}

NJson::TJsonValue TRTElementFinesCollector::DoSerializeToJson() const {
    NJson::TJsonValue result = TBase::DoSerializeToJson();
    result.InsertValue("min_ruling_date", MinRulingDate.Seconds());
    result.InsertValue("max_request_fines_count", MaxRequestFinesCount);
    NJson::InsertNonNull(result, "sts_to_check", StsToCheck);
    result.InsertValue("exclude_sts", ExcludeSts);
    result.InsertValue("penalty_check_policy", ToString(PenaltyCheckPolicy));
    result.InsertValue("is_update_allowed", UpdateAllowedFlag);
    result.InsertValue("unknown_article_only", UpdateUnknownArticleOnly);
    return result;
}

bool IsRequestSizeUnlimited(const ui32 maxRequestFinesCount) {
    return (maxRequestFinesCount == 0 || maxRequestFinesCount == TRTElementFinesCollector::DefaultMaxRequestFinesCount);
}

TMaybe<TVector<NDrive::TElementFine>> CollectFines(const NDrive::IServer& server, const ui64 offset, const ui64 limit, const TInstant sinceRulingDate, const TInstant lastUpdate, TMessagesCollector& errors) {
    const auto& client = server.GetDriveAPI()->GetElementFinesClient();
    NDrive::TElementFinesClient::TFinesRequestParams params;
    params.Offset = offset;
    params.Limit = limit;
    params.Since = sinceRulingDate;
    params.LastUpdate = lastUpdate;
    auto future = client.GetFines(params);
    const auto timeout = client.GetConfig().GetRequestTimeout();
    if (!future.Wait(timeout)) {
        errors.AddMessage(__LOCATION__, "Request timeouted");
        return {};
    }
    if (!future.HasValue()) {
        errors.AddMessage(__LOCATION__, "Request failed with exception: " + NThreading::GetExceptionMessage(future));
        return {};
    }
    return std::move(future.GetValue());
}

TExpectedState TRTElementFinesCollector::DoExecute(TAtomicSharedPtr<IRTBackgroundProcessState> state, const TExecutionContext& context) const {
    const TRTElementFinesCollectorState* rState = dynamic_cast<const TRTElementFinesCollectorState*>(state.Get());
    TInstant lastUpdate = rState ? rState->GetLastInstant() : MinRulingDate;
    const auto& server = context.GetServerAs<NDrive::IServer>();
    if (!server.GetDriveAPI() || !server.GetDriveAPI()->HasFinesManager() || !server.GetDriveAPI()->HasElementFineClient() || !GetNotifyHandlers()) {
        ERROR_LOG << "Failed to get toolkit" << Endl;
        return nullptr;
    }
    auto& notifier = *GetNotifyHandlers();
    NDrive::NFine::TFineConstructor fineConstructor(GetFineConstructorConfig(), server);

    TPrefechedFinesContainer prefechedFines;
    {
        const auto& finesManager = Yensured(server.GetDriveAPI())->GetFinesManager();
        auto tx = finesManager.BuildTx<NSQL::ReadOnly>();
        R_ENSURE(prefechedFines.Init(finesManager, GetFilters(), tx), {}, "Fail to fetch fines for ElementFinesCollector", tx);
    }

    ui64 offset = 0;
    ui64 limit = (IsRequestSizeUnlimited(MaxRequestFinesCount)) ? DefaultMaxRequestFinesCount : MaxRequestFinesCount;
    bool hasMore = false;
    TMessagesCollector errors;
    do {
        auto fines = CollectFines(server, offset, limit, MinRulingDate, lastUpdate, errors);
        if (!fines) {
            notifier.Handle(ECollectingStatus::CollectingFail, server);
            return MakeUnexpected<TString>("Failed to collect fines: " + errors.GetStringReport());
        }
        notifier.Handle(ECollectingStatus::CollectingSuccess, server);
        notifier.Handle(ECollectingStatus::CollectedTotal, server, fines->size());

        hasMore = !fines->empty();
        if (hasMore) {
            auto filteredFines = FilterCollectedFines(std::move(*fines));
            auto contextPtr = MakeAtomicShared<TContext>(server, fineConstructor, prefechedFines, std::move(filteredFines));
            if (!ProcessFines(contextPtr, errors)) {
                return MakeUnexpected<TString>("Failed process fines: " + errors.GetStringReport());
            }
        }

        offset += limit;
    } while (!IsRequestSizeUnlimited(MaxRequestFinesCount) && hasMore);


    auto result = MakeAtomicShared<TRTElementFinesCollectorState>();
    result->SetLastInstant(StartInstant);
    return result;
}

TVector<NDrive::TElementFine> TRTElementFinesCollector::FilterCollectedFines(TVector<TElementFine>&& fines) const {
    TVector<TElementFine> filteredFines;
    for (auto&& fine : fines) {
        if (GetPenaltyCheckPolicy() != EPenaltyCheckPolicy::All
            && ((GetPenaltyCheckPolicy() == EPenaltyCheckPolicy::Paid) != fine.IsPaid()))
        {
            continue;
        }
        if (StsToCheck.contains(ToString(fine.GetCarSTS())) ^ IsExcludeSts()) {
            filteredFines.push_back(std::move(fine));
        }
    }
    return filteredFines;
}

bool TRTElementFinesCollector::ConstructFines(IFineProcessingContextBase::TPtr contextPtr, TVector<TAutocodeFineEntry>& nativeFines, TMessagesCollector& errors) const {
    auto handlerContextPtr = std::dynamic_pointer_cast<TContext>(contextPtr);
    if (!handlerContextPtr) {
        return false;
    }

    for (auto&& fine : handlerContextPtr->GetExternalFines()) {
        TAutocodeFineEntry nativeFine;
        if (!ProcessFine(handlerContextPtr, fine, nativeFine, errors)) {
            if (errors.HasMessages()) {
                return false;
            }
            continue;
        }
        nativeFines.push_back(std::move(nativeFine));
    }
    return true;
}

bool TRTElementFinesCollector::ProcessFine(const TContextPtr contextPtr, const TElementFine& fine, TAutocodeFineEntry& nativeFine, TMessagesCollector& errors) const {
    TString rulingNumber = fine.GetProtocolNumber();
    if (!rulingNumber) {
        return false;
    }

    auto articleMatcherPtr = contextPtr->GetFineArticleMatcherPtr();
    if (!articleMatcherPtr) {
        return false;
    }
    auto unknownArticleFilter = NDrive::NFine::TFineUnknownArticleFilter(articleMatcherPtr);
    auto notChargedFineFilter = NDrive::NFine::TFineNotChargedFilter();

    auto elementContextPtr = dynamic_cast<TContext*>(contextPtr.Get());
    if (!elementContextPtr) {
        errors.AddMessage(__LOCATION__, "fail to cast context");
        return false;
    }
    auto equalRange = contextPtr->GetFineMapByRulingNumber().equal_range(rulingNumber);
    for (auto it = equalRange.first; it != equalRange.second; ++it) {  // (*it) is a std::pair<...>(ruling_number, id)
        const NDrive::NFine::TAutocodeFineEntry& fine = it->second;
        if (fine.GetSourceType() != ToString(NDrive::NFine::ESourceType::Element)) {
            return false;
        }
        if (GetUpdateUnknownArticleOnly() && !unknownArticleFilter.Match(fine)) {
            return false;
        }
        if (GetUpdateAllowedFlag() && notChargedFineFilter.Match(fine)) {
            nativeFine = std::move(fine);  // update, set fine id in particular
        } else {
            return false;  // fine already exists and update is not permitted
        }
    }

    if (!contextPtr->GetFineConstructor().ConstructAutocodeFineEntry(fine, nativeFine)) {
        ERROR_LOG <<  "Error converting fine to db entry: " << fine.GetProtocolNumber() << Endl;
        GetNotifyHandlers()->Handle(ECollectingStatus::CollectedConversionError, contextPtr->GetServer());
        return false;
    }

    return true;
}
