#include "events.h"

#include <drive/backend/abstract/frontend.h>
#include <drive/backend/database/drive_api.h>
#include <drive/backend/tags/tag_description.h>
#include <drive/backend/tags/tags.h>
#include <drive/backend/tags/tags_manager.h>

#include <library/cpp/regex/pcre/regexp.h>

#include <rtline/util/algorithm/ptr.h>

bool TagNameEventsFilter::DoFilter(const NDrive::IServer& /* server */, TEventPtr eventPtr) const {
    return TagNames.contains((*eventPtr)->GetName());
}

TagNameRegexpEventsFilter::TagNameRegexpEventsFilter(const EObjectHistoryAction historyAction, const TString& pattern)
    : TBase(historyAction)
    , Pattern(pattern)
{
    try {
        Re = MakeHolder<TRegExMatch>(Pattern);
    } catch (yexception& e) {
        ERROR_LOG << "Error initializing tag name pattern: " << e.what() << Endl;
    }
}

bool TagNameRegexpEventsFilter::IsValid() const {
    return !!Re && Re->IsCompiled();
}

bool TagNameRegexpEventsFilter::DoFilter(const NDrive::IServer& /* server */, TEventPtr eventPtr) const {
    return IsValid() && Re->Match((*eventPtr)->GetName().data());
}

// fine specific filter

TFineActionTagFilter::TFineActionTagFilter(const EObjectHistoryAction historyAction, EFineActionType fineActionType)
    : TBase(historyAction)
    , FineActionType(fineActionType)
{
}

bool TFineActionTagFilter::DoFilter(const NDrive::IServer& server, TEventPtr eventPtr) const {
    auto fineActionTag = eventPtr->GetTagAs<TFineActionTag>();
    if (!fineActionTag) {
        return false;
    }

    auto tagDescriptionPtr = server.GetDriveAPI()->GetTagsManager().GetTagsMeta().GetDescriptionByName(fineActionTag->GetName());
    if (!tagDescriptionPtr) {
        return false;
    }

    auto fineActionTagDescriptionPtr = std::dynamic_pointer_cast<const TFineActionTag::TTagDescription>(tagDescriptionPtr);
    return fineActionTagDescriptionPtr && fineActionTagDescriptionPtr->GetActionType() == FineActionType;
}

TEventsHandler::TEventsHandler(const NEntityTagsManager::EEntityType entityType, std::initializer_list<IEventsFilter::TPtr> il)
    : EntityType(entityType)
    , Filters(il)
{
    Y_ENSURE(entityType == NEntityTagsManager::EEntityType::Car || entityType == NEntityTagsManager::EEntityType::User);
}

const IEntityTagsManager& TEventsHandler::GetEntityTagsManager(const NDrive::IServer& server) const {
    if (EntityType == NEntityTagsManager::EEntityType::Car) {
        return server.GetDriveAPI()->GetTagsManager().GetDeviceTags();
    } else if (EntityType == NEntityTagsManager::EEntityType::User) {
        return server.GetDriveAPI()->GetTagsManager().GetUserTags();
    } else {
        Y_UNREACHABLE();
    }
}

void TEventsHandler::AddFilter(TFilterPtr filter) {
    if (!!filter) {
        Filters.push_back(filter);
    }
}

void TEventsHandler::Clear() {
    Filters.clear();
}

bool TEventsHandler::AreFiltersValid() const {
    return AllOf(Filters, [](auto filterPtr) { return !!filterPtr && filterPtr->IsValid(); });
}

bool TEventsHandler::Fetch(const NDrive::IServer& server, ui64 lastProcessedEventId, const TInstant actuality) {
    Y_UNUSED(actuality);
    const auto& tagManager = GetEntityTagsManager(server);

    Events.clear();

    auto session = tagManager.BuildSession();
    auto optionalLockedMaxEventId = tagManager.GetLockedMaxEventId(session);
    if (!optionalLockedMaxEventId) {
        ERROR_LOG << "cannot GetLockedMaxEventId: " << session.GetStringReport() << Endl;
        return false;
    }
    LastFetchedEventId = *optionalLockedMaxEventId;
    LastProducedEventId = LastProcessedEventId = Min(lastProcessedEventId, LastFetchedEventId);

    auto limit = 1000000;
    auto events = tagManager.GetEventsSince(LastProcessedEventId + 1, session, limit);
    if (!events) {
        ERROR_LOG << "cannot GetEventsSince: " << session.GetStringReport() << Endl;
        return false;
    }
    for (auto&& ev : *events) {
        Events.push_back(MakeAtomicShared<TTagHistoryEvent>(std::move(ev)));
    }

    CurrentEventIt = Events.cbegin();

    return true;
}

TEventsHandler::TEventPtr TEventsHandler::Next(const NDrive::IServer& server) {
    while (CurrentEventIt != Events.cend()) {
        TEventPtr currentEventPtr = *(CurrentEventIt++);

        LastProcessedEventId = LastProducedEventId;
        LastProducedEventId = currentEventPtr->GetHistoryEventId();

        if (currentEventPtr->GetHistoryEventId() <= LastFetchedEventId && FilterEvent(server, currentEventPtr)) {
            return currentEventPtr;
        }
    }

    LastProcessedEventId = LastProducedEventId;
    return nullptr;
}

bool TEventsHandler::DoesTagExist(TEventPtr eventPtr, const NDrive::IServer& server, NDrive::TEntitySession& session, TVector<TDBTag>* tagsPtr) const {
    TVector<TDBTag> tags;

    const auto& tagId = eventPtr->GetTagId();
    const auto& tagsManager = GetEntityTagsManager(server);
    if (!tagsManager.RestoreTags(tagId, tags, session)) {
        return false;
    }

    const bool tagExists = (tags.size() == 1);

    if (tagExists && !!tagsPtr) {
        (*tagsPtr) = std::move(tags);
    }

    return tagExists;
}

bool TEventsHandler::RemoveTagIfExists(TEventPtr eventPtr, const TString& userId, const NDrive::IServer& server, NDrive::TEntitySession& session) const {
    TVector<TDBTag> tags;

    // skip removed tags and try to remove existing only
    if (DoesTagExist(eventPtr, server, session, &tags)) {
        return GetEntityTagsManager(server).RemoveTag(tags.front(), userId, &server, session);
    }

    return true;
}

bool TEventsHandler::FilterEvent(const NDrive::IServer& server, TEventPtr eventPtr) const {
    return AnyOf(Filters, [&server, eventPtr](auto filterPtr) { return !!filterPtr && filterPtr->Filter(server, eventPtr); });
}
