#pragma once

#include <drive/backend/data/fine.h>  // fine specific

#include <rtline/library/unistat/cache.h>
#include <rtline/util/types/accessor.h>

#include <util/generic/algorithm.h>

class TRegExMatch;

class TConstDBTag;

template <typename T>
class TObjectEvent;

class IEntityTagsManager;

namespace NDrive {
    class IServer;
    class TEntitySession;
}

class IEventsFilter {
public:
    using TEvent = TObjectEvent<TConstDBTag>;
    using TEventPtr = TAtomicSharedPtr<TEvent>;
    using TPtr = TAtomicSharedPtr<IEventsFilter>;

    explicit IEventsFilter(const EObjectHistoryAction historyAction)
        : HistoryAction(historyAction)
    {
    }

    virtual ~IEventsFilter() = default;

    virtual bool IsValid() const {
        return true;
    }

    virtual bool Filter(const NDrive::IServer& server, TEventPtr eventPtr) const final {
        return !!eventPtr && HistoryAction == eventPtr->GetHistoryAction() && DoFilter(server, eventPtr);
    }

private:
    virtual bool DoFilter(const NDrive::IServer& server, TEventPtr eventPtr) const = 0;

    R_READONLY(EObjectHistoryAction, HistoryAction);
};

class TagNameEventsFilter : public IEventsFilter {
    using TBase = IEventsFilter;

public:
    template <typename C>
    TagNameEventsFilter(const EObjectHistoryAction historyAction, const C& container)
        : TBase(historyAction)
        , TagNames(std::begin(container), std::end(container))
    {
    }

private:
    virtual bool DoFilter(const NDrive::IServer& server, TEventPtr eventPtr) const override;

    const TSet<TString> TagNames;
};

class TagNameRegexpEventsFilter : public IEventsFilter {
    using TBase = IEventsFilter;

public:
    TagNameRegexpEventsFilter(const EObjectHistoryAction historyAction, const TString& pattern);

    virtual bool IsValid() const override;

private:
    virtual bool DoFilter(const NDrive::IServer& server, TEventPtr eventPtr) const override;

    const TString Pattern;
    THolder<TRegExMatch> Re;
};

// fine specific filter

class TFineActionTagFilter : public IEventsFilter {
    using TBase = IEventsFilter;

public:
    TFineActionTagFilter(const EObjectHistoryAction historyAction, EFineActionType fineActionType);

private:
    virtual bool DoFilter(const NDrive::IServer& server, TEventPtr eventPtr) const override;

    const EFineActionType FineActionType;
};

class TEventsHandler {
public:
    using TPtr = TAtomicSharedPtr<TEventsHandler>;
    using TFilterPtr = IEventsFilter::TPtr;
    using TEventPtr = IEventsFilter::TEventPtr;
    using TEvents = TVector<TEventPtr>;

    explicit TEventsHandler(const NEntityTagsManager::EEntityType entityType, std::initializer_list<IEventsFilter::TPtr> il = {});

    const IEntityTagsManager& GetEntityTagsManager(const NDrive::IServer& server) const;

    void AddFilter(TFilterPtr filter);
    void Clear();

    constexpr bool Empty() const noexcept {
        return Filters.empty();
    }

    bool AreFiltersValid() const;

    bool Fetch(const NDrive::IServer& server, ui64 lastProcessedEventId, const TInstant actuality = TInstant::Zero());

    TEventPtr Next(const NDrive::IServer& server);

    bool DoesTagExist(TEventPtr eventPtr, const NDrive::IServer& server, NDrive::TEntitySession& session, TVector<TDBTag>* tagsPtr = nullptr) const;
    bool RemoveTagIfExists(TEventPtr eventPtr, const TString& userId, const NDrive::IServer& server, NDrive::TEntitySession& session) const;

protected:
    bool FilterEvent(const NDrive::IServer& server, TEventPtr eventPtr) const;

private:
    NEntityTagsManager::EEntityType EntityType;

    R_READONLY(ui64, LastFetchedEventId, 0);
    R_READONLY(ui64, LastProcessedEventId, 0);  // last successfully processed by handler
    R_READONLY(ui64, LastProducedEventId, 0);  // last produced to handler

    TVector<TFilterPtr> Filters;

    TEvents Events;
    TEvents::const_iterator CurrentEventIt;
};
