#include "fine_documents_handler.h"

#include <drive/backend/database/drive_api.h>
#include <drive/backend/fines/manager.h>

#include <rtline/util/json_processing.h>

const TString TRTFineDocumentHandler::TypeName = "fine_documents_handler";

IRTRegularBackgroundProcess::TFactory::TRegistrator<TRTFineDocumentHandler> TRTFineDocumentHandler::Registrator(TRTFineDocumentHandler::TypeName);

TString TRTFineDocumentHandlerState::GetType() const {
    return TRTFineDocumentHandler::TypeName;
}

TRTFineDocumentHandlerState::TFactory::TRegistrator<TRTFineDocumentHandlerState> TRTFineDocumentHandlerState::Registrator(TRTFineDocumentHandler::TypeName);

TRTFineDocumentHandler::TRTFineDocumentHandler()
    : TBase(MakeAtomicShared<TEventsHandler>(NEntityTagsManager::EEntityType::Car))
{
    MutableTagEventsHandlerPtr()->AddFilter(MakeAtomicShared<TFineActionTagFilter>(EObjectHistoryAction::Add, EFineActionType::RequestViolationDetailedDocument));

    InitNotifyHandlers();
}

TAtomicSharedPtr<TRTFinesHandlerBaseState> TRTFineDocumentHandler::BuildState(const ui64 lastProcessedSerialId, const ui64 lastProcessedEventId) const {
    auto state = MakeAtomicShared<TRTFineDocumentHandlerState>();
    state->SetLastSerialId(lastProcessedSerialId);
    state->SetLastTagHistoryEventId(lastProcessedEventId);
    return state;
}

TRTFineDocumentHandler::IUpdateContext::TPtr TRTFineDocumentHandler::ConstructUpdateContext() const {
    return MakeAtomicShared<TUpdateContext>();
}

NDrive::NFine::EProcessingStatus TRTFineDocumentHandler::PrepareUpdate(const NDrive::IServer& server, IUpdateContext::TPtr updateContextPtr, NDrive::NFine::TAutocodeFineEntry& fine, TFineIdToEventPtrMapping& fineIdToEventPtrMapping, TMessagesCollector& errors) const {
    auto handlerUpdateContextPtr = VerifyDynamicCast<TUpdateContext*>(updateContextPtr.Get());

    NDrive::NFine::TFineFetchContext context(GetFetchContextConfig(), server, fine);
    TMessagesCollector dataFetchingErrors;

    TString documentUrl;

    NDrive::NFine::TFineViolationDetailedDocumentUrlContextFetcher fetcher;
    if (!fetcher.Fetch(context, documentUrl, dataFetchingErrors) || !documentUrl) {
        errors.AddMessage(__LOCATION__, TStringBuilder()
                                        << "Cannot obtain document file url for fine " << context.GetEntry().GetId() << ": "
                                        << dataFetchingErrors.GetStringReport());
        return EProcessingStatus::DataFetchingError;
    }

    handlerUpdateContextPtr->SetViolationDetailedDocumentFileUrl(documentUrl);

    handlerUpdateContextPtr->SetRelatedEventPtr(fineIdToEventPtrMapping.Value(fine.GetId(), nullptr));

    return EProcessingStatus::DataFetchingSuccess;
}

NDrive::NFine::EProcessingStatus TRTFineDocumentHandler::ApplyUpdate(const NDrive::IServer& server, const IUpdateContext::TPtr updateContextPtr, NDrive::NFine::TAutocodeFineEntry& fine, NDrive::TEntitySession& session, TMessagesCollector& errors) const {
    auto handlerUpdateContextPtr = VerifyDynamicCast<const TUpdateContext*>(updateContextPtr.Get());

    NJson::TJsonValue metaInfoAttachments = fine.GetMetaInfoProperty(NDrive::NFine::TAutocodeFineEntry::EMetaInfoProperty::Attachments, NJson::JSON_ARRAY);

    NDrive::NFine::TAutocodeFineEntry::TMetaInfoAttachment metaInfoAttachment(handlerUpdateContextPtr->GetViolationDetailedDocumentFileUrl(), ModelingNow());
    metaInfoAttachments.AppendValue(metaInfoAttachment.SerializeToJson());

    fine.SetMetaInfoProperty(NDrive::NFine::TAutocodeFineEntry::EMetaInfoProperty::Attachments, std::move(metaInfoAttachments));

    const auto& finesManager = server.GetDriveAPI()->GetFinesManager();

    if (!finesManager.UpsertFine(fine, session)) {
        errors.AddMessage(__LOCATION__, TStringBuilder() << "Error upserting fine " << fine.GetId() << ": " << session.GetStringReport());
        return EProcessingStatus::FineUpsertError;
    }

    return EProcessingStatus::FineUpsertSuccess;
}
