#include "fines_handler.h"

#include <drive/backend/data/billing_tags.h>
#include <drive/backend/data/notifications_tags.h>
#include <drive/backend/data/user_tags.h>
#include <drive/backend/database/drive_api.h>
#include <drive/backend/database/transaction/assert.h>
#include <drive/backend/fines/manager.h>
#include <drive/backend/tags/tags_manager.h>

#include <rtline/util/json_processing.h>

const TString TRTFinesHandler::TypeName = "fines_handler";

IRTRegularBackgroundProcess::TFactory::TRegistrator<TRTFinesHandler> TRTFinesHandler::Registrator(TRTFinesHandler::TypeName);

TString TRTFinesHandlerState::GetType() const {
    return TRTFinesHandler::TypeName;
}

TRTFinesHandlerState::TFactory::TRegistrator<TRTFinesHandlerState> TRTFinesHandlerState::Registrator(TRTFinesHandler::TypeName);

TRTFinesHandler::TRTFinesHandler()
    : TBase(MakeAtomicShared<TEventsHandler>(NEntityTagsManager::EEntityType::Car))
{
    MutableTagEventsHandlerPtr()->AddFilter(MakeAtomicShared<TFineActionTagFilter>(EObjectHistoryAction::Add, EFineActionType::Charge));

    InitNotifyHandlers();
}

NDrive::NFine::EProcessingStatus TRTFinesHandler::CheckFineRestrictions(const NDrive::IServer& server, const NDrive::NFine::TAutocodeFineEntry& fine, NDrive::TEntitySession& tx) const {
    auto status = TBase::CheckFineRestrictions(server, fine, tx);
    if (status != EProcessingStatus::CheckFineSuccess) {
        return status;
    }

    TMessagesCollector foundUserRestrictions, userCheckErrors;
    if (auto check = UserRestrictions.Check(server, fine, tx); check != NFineUserRestrictions::EUserRestrictionsStatus::Allowed) {
        R_ENSURE(check != NFineUserRestrictions::EUserRestrictionsStatus::Error, {}, "transaction error on user restrictions check", tx);
        NDrive::TEventLog::Log("UserRestrictionsSkip", NJson::TMapBuilder
            ("reason", ToString(check))
            ("fine_id", fine.GetId())
        );
        return EProcessingStatus::UserRestrictionsSkip;
    }
    return EProcessingStatus::CheckFineSuccess;
}

TAtomicSharedPtr<TRTFinesHandlerBaseState> TRTFinesHandler::BuildState(const ui64 lastProcessedSerialId, const ui64 lastProcessedEventId) const {
    auto state = MakeAtomicShared<TRTFinesHandlerState>();
    state->SetLastSerialId(lastProcessedSerialId);
    state->SetLastTagHistoryEventId(lastProcessedEventId);
    return state;
}

TRTFinesHandler::IUpdateContext::TPtr TRTFinesHandler::ConstructUpdateContext() const {
    return MakeAtomicShared<TUpdateContext>();
}

NDrive::NFine::EProcessingStatus TRTFinesHandler::PrepareUpdate(const NDrive::IServer& server, IUpdateContext::TPtr updateContextPtr, NDrive::NFine::TAutocodeFineEntry& fine, TFineIdToEventPtrMapping& fineIdToEventPtrMapping, TMessagesCollector& errors) const {
    auto handlerUpdateContextPtr = VerifyDynamicCast<TUpdateContext*>(updateContextPtr.Get());

    NDrive::NFine::TFineFetchContext context(GetFetchContextConfig(), server, fine);
    {
        auto tx = server.GetDriveAPI()->GetFinesManager().BuildTx<NSQL::ReadOnly>();
        if (!context.GetPrefetchedPhotos(errors, &tx)) {
            return EProcessingStatus::TagConstructingError;
        }
    }
    TMessagesCollector tagConstructingErrors;

    // note: if article code cannot be determined (due to malformed or _empty_ fine article), tag won't be constructed
    if (!CreateTags(context, handlerUpdateContextPtr->MutableTagPtrs(), tagConstructingErrors)) {
        errors.AddMessage(__LOCATION__, TStringBuilder()
                                        << "Cannot create tags for fine " << context.GetEntry().GetId() << ": "
                                        << tagConstructingErrors.GetStringReport());
        return EProcessingStatus::TagConstructingError;
    }

    handlerUpdateContextPtr->SetRelatedEventPtr(fineIdToEventPtrMapping.Value(fine.GetId(), nullptr));

    return EProcessingStatus::DataFetchingSuccess;
}

bool TRTFinesHandler::CreateTags(const NDrive::NFine::TFineFetchContext& context, TVector<ITag::TPtr>& tagPtrs, TMessagesCollector& errors) const {
    TVector<const NDrive::NFine::IFineTagConfig*> allTagConfigs = { &ChargeTagConfig, &MailTagConfig, &PushTagConfig };
    for (auto tagConfigPtr : TagConfigs) {
        allTagConfigs.push_back(tagConfigPtr.Get());
    }

    for (const auto* tagConfig : allTagConfigs) {
        ITag::TPtr tagPtr = tagConfig->CreateTag(context, errors);
        if (!tagConfig->IsTagNullable() && tagPtr == nullptr) {
            errors.AddMessage(__LOCATION__, TStringBuilder() << "Cannot create tag for fine " << context.GetEntry().GetId() << ": " << errors.GetStringReport());
            return false;
        }

        if (tagPtr != nullptr) {
            tagPtrs.push_back(tagPtr);
        }
    }
    return true;
}

NDrive::NFine::EProcessingStatus TRTFinesHandler::ApplyUpdate(const NDrive::IServer& server, const IUpdateContext::TPtr updateContextPtr, NDrive::NFine::TAutocodeFineEntry& fine, NDrive::TEntitySession& session, TMessagesCollector& errors) const {
    auto handlerUpdateContextPtr = VerifyDynamicCast<const TUpdateContext*>(updateContextPtr.Get());

    TVector<TDBTag> affectedTags;
    if (!AddTags(server, fine, handlerUpdateContextPtr->GetTagPtrs(), affectedTags, session, errors)) {
        return EProcessingStatus::TagUpsertError;
    }

    if (!UpsertFine(server, affectedTags, fine, session, errors)) {
        return EProcessingStatus::FineUpsertError;
    }

    return EProcessingStatus::FineUpsertSuccess;
}

bool TRTFinesHandler::AddTags(const NDrive::IServer& server, const NDrive::NFine::TAutocodeFineEntry& fine, const TVector<ITag::TPtr>& tagPtrs, TVector<TDBTag>& affectedTags, NDrive::TEntitySession& session, TMessagesCollector& errors) const {
    const auto& tagsManager = server.GetDriveAPI()->GetTagsManager();

    auto optionalAddedTags = tagsManager.GetUserTags().AddTags(tagPtrs, GetRobotUserId(), fine.GetUserId(), &server, session, EUniquePolicy::Undefined);
    if (!optionalAddedTags) {
        errors.AddMessage(__LOCATION__, TStringBuilder() << "Error adding tags for fine " << fine.GetId() << ": " << session.GetStringReport());
        return false;
    }

    affectedTags = std::move(*optionalAddedTags);
    return true;
}

bool TRTFinesHandler::UpsertFine(const NDrive::IServer& server, const TVector<TDBTag>& affectedTags, NDrive::NFine::TAutocodeFineEntry& fine, NDrive::TEntitySession& session, TMessagesCollector& errors) const {
    const TInstant now = ModelingNow();

    TString chargeTagId;

    NJson::TJsonValue relatedTagIds = fine.GetMetaInfoProperty(NDrive::NFine::TAutocodeFineEntry::EMetaInfoProperty::RelatedTagIds, NJson::JSON_ARRAY);
    NJson::TJsonValue metaInfoAttachments = fine.GetMetaInfoProperty(NDrive::NFine::TAutocodeFineEntry::EMetaInfoProperty::Attachments, NJson::JSON_ARRAY);

    TSet<TString> existingAttachmentUrls;
    for (const auto& metaInfoAttachment : metaInfoAttachments.GetArray()) {
        existingAttachmentUrls.emplace(metaInfoAttachment["url"].GetString());
    }

    for (const auto& affectedTag : affectedTags) {
        relatedTagIds.AppendValue(affectedTag.GetTagId());

        auto mailTagPtr = affectedTag.GetTagAs<TUserMailTag>();
        if (mailTagPtr != nullptr) {
            fine.SetChargeEmailSentAt(now);
            for (const auto& attachment : mailTagPtr->GetAttachments()) {
                TString url = attachment.Url;
                if (!existingAttachmentUrls.contains(url)) {
                    NDrive::NFine::TAutocodeFineEntry::TMetaInfoAttachment metaInfoAttachment(url, now);
                    metaInfoAttachments.AppendValue(metaInfoAttachment.SerializeToJson());
                    existingAttachmentUrls.insert(url);
                }
            }
            continue;
        }

        auto pushTagPtr = affectedTag.GetTagAs<TUserPushTag>();
        if (pushTagPtr != nullptr) {
            fine.SetChargePushSentAt(now);
            continue;
        }

        auto chargeTagPtr = affectedTag.GetTagAs<TBillingTag>();
        if (chargeTagPtr != nullptr) {
            chargeTagId = affectedTag.GetTagId();
            fine.SetChargedAt(now);
            fine.SetMetaInfoProperty(NDrive::NFine::TAutocodeFineEntry::EMetaInfoProperty::ChargeTagId, chargeTagId);
            continue;
        }
    }

    fine.SetMetaInfoProperty(NDrive::NFine::TAutocodeFineEntry::EMetaInfoProperty::RelatedTagIds, std::move(relatedTagIds));
    fine.SetMetaInfoProperty(NDrive::NFine::TAutocodeFineEntry::EMetaInfoProperty::Attachments, std::move(metaInfoAttachments));

    const auto& finesManager = server.GetDriveAPI()->GetFinesManager();

    if (!!chargeTagId) {
        if (!finesManager.UpsertFineNotCharged(fine, session)) {
            errors.AddMessage(__LOCATION__, TStringBuilder() << "Error upserting fine " << fine.GetId() << " to charge: " << session.GetStringReport());
            return false;
        }
    } else {
        if (!finesManager.UpsertFine(fine, session)) {
            errors.AddMessage(__LOCATION__, TStringBuilder() << "Error upserting fine " << fine.GetId() << ": " << session.GetStringReport());
            return false;
        }
    }

    return true;
}

NDrive::TScheme TRTFinesHandler::DoGetScheme(const IServerBase& server) const {
    auto scheme = TBase::DoGetScheme(server);

    scheme.Add<TFSStructure>("charge_tag_config", "Параметры тега списания").SetStructure(NDrive::NFine::TFineChargeTagConfig::GetScheme(server));
    scheme.Add<TFSStructure>("mail_tag_config", "Параметры тега письма").SetStructure(NDrive::NFine::TFineMailTagConfig::GetScheme(server));
    scheme.Add<TFSStructure>("push_tag_config", "Параметры тега пуша").SetStructure(NDrive::NFine::TFinePushTagConfig::GetScheme(server));
    scheme.Add<TFSArray>("tag_configs", "Параметры прочих тегов штрафов").SetElement(NDrive::NFine::TFineDefaultTagConfig::GetScheme(server));

    scheme.Add<TFSStructure>("user_restrictions", "Ограничения на списания по пользователю").SetStructure(NFineUserRestrictions::TFineUserRestrictions::GetScheme());

    return scheme;
}

bool TRTFinesHandler::DoDeserializeFromJson(const NJson::TJsonValue& data) {
    if (!TBase::DoDeserializeFromJson(data)) {
        return false;
    }

    if (!ChargeTagConfig.DeserializeFromJson(data["charge_tag_config"])) {
        return false;
    }
    if (!MailTagConfig.DeserializeFromJson(data["mail_tag_config"])) {
        return false;
    }
    if (!PushTagConfig.DeserializeFromJson(data["push_tag_config"])) {
        return false;
    }

    {
        for (const auto& rawTagConfig : data["tag_configs"].GetArray()) {
            auto tagConfigPtr = MakeAtomicShared<NDrive::NFine::TFineDefaultTagConfig>();
            if (!tagConfigPtr->DeserializeFromJson(rawTagConfig)) {
                return false;
            }
            TagConfigs.push_back(tagConfigPtr);
        }
    }

    if (!UserRestrictions.DeserializeFromJson(data["user_restrictions"])) {
        return false;
    }

    // extra restriction on chargeable tag processing:
    //     enabled charge tag config requires not charged and chargeable tags only
    if (ChargeTagConfig.IsEnabled() && !(IsNotChargedOnly() && IsChargeableOnly())) {
        return false;
    }

    return true;
}

NJson::TJsonValue TRTFinesHandler::DoSerializeToJson() const {
    NJson::TJsonValue result = TBase::DoSerializeToJson();

    TJsonProcessor::WriteSerializable(result, "charge_tag_config", ChargeTagConfig);
    TJsonProcessor::WriteSerializable(result, "mail_tag_config", MailTagConfig);
    TJsonProcessor::WriteSerializable(result, "push_tag_config", PushTagConfig);

    {
        auto& serializedTagConfigs = result.InsertValue("tag_configs", NJson::JSON_ARRAY);
        for (auto tagConfigPtr : TagConfigs) {
            serializedTagConfigs.AppendValue(tagConfigPtr->SerializeToJson());
        }
    }

    TJsonProcessor::WriteSerializable(result, "user_restrictions", UserRestrictions);

    return result;
}
