#include "handler_base.h"

#include <drive/backend/database/drive_api.h>
#include <drive/backend/database/transaction/assert.h>
#include <drive/backend/fines/manager.h>
#include <drive/backend/tags/tags_manager.h>

#include <library/cpp/threading/future/async.h>
#include <library/cpp/threading/future/future.h>

#include <rtline/util/json_processing.h>
#include <rtline/util/algorithm/ptr.h>

#include <util/generic/algorithm.h>

void TRTFinesHandlerBaseState::SerializeToProto(NDrive::NProto::TRTFinesHandlerState& proto) const {
    proto.SetLastSerialId(LastSerialId);
    proto.SetLastTagHistoryEventId(LastTagHistoryEventId);
}

bool TRTFinesHandlerBaseState::DeserializeFromProto(const NDrive::NProto::TRTFinesHandlerState& proto) {
    LastSerialId = proto.GetLastSerialId();
    LastTagHistoryEventId = proto.GetLastTagHistoryEventId();
    return true;
}

NJson::TJsonValue TRTFinesHandlerBaseState::GetReport() const {
    NJson::TJsonValue result = TBase::GetReport();
    TJsonProcessor::Write(result, "last_serial_id", LastSerialId);
    TJsonProcessor::Write(result, "last_tag_history_event_id", LastTagHistoryEventId);
    return result;
}

NDrive::TScheme TRTFinesHandlerBaseState::DoGetScheme() const {
    NDrive::TScheme result = TBase::DoGetScheme();
    result.Add<TFSNumeric>("last_serial_id", "Последний обработанный штраф");
    result.Add<TFSNumeric>("last_tag_history_event_id", "Последнее обработанное событие");
    return result;
}

const TDuration TRTFinesHandlerBase::DefaultPhotoWaitTimeout = TDuration::Hours(1);

TRTFinesHandlerBase::TRTFinesHandlerBase(TEventsHandler::TPtr tagEventsHandlerPtr)
    : TBase()
    , TagEventsHandlerPtr(tagEventsHandlerPtr)
    , DerivedTotalAmountCentsLimit(0)
    , DerivedChargesLimit(0)
{
}

bool TRTFinesHandlerBase::DoStart(const TRTBackgroundProcessContainer& container) {
    if (!TBase::DoStart(container)) {
        return false;
    }
    if (!NotifyHandlers) {
        return false;
    }
    NotifyHandlers->SetSignalName(GetRTProcessName());
    return true;
}

void TRTFinesHandlerBase::InitNotifyHandlers() {
    if (!NotifyHandlers) {
        NotifyHandlers = MakeAtomicShared<TNotifyHandlers>(GetType());
    }
}

TExpectedState TRTFinesHandlerBase::DoExecute(TAtomicSharedPtr<IRTBackgroundProcessState> state, const TExecutionContext& context) const {
    const auto& server = context.GetServerAs<NDrive::IServer>();
    CHECK_WITH_LOG(!!server.GetDriveAPI());

    if (!server.GetDriveAPI()->HasFinesManager() || !NotifyHandlers) {
        return MakeUnexpected<TString>({});
    }

    auto nativeState = std::dynamic_pointer_cast<TRTFinesHandlerBaseState>(state);
    const ui64 lastProcessedSerialId = (!!nativeState) ? nativeState->GetLastSerialId() : 0;
    const ui64 lastProcessedEventId = (!!nativeState) ? nativeState->GetLastTagHistoryEventId() : std::numeric_limits<ui64>::max();

    auto nextNativeState = BuildState(lastProcessedSerialId, lastProcessedEventId);

    {
        TMessagesCollector foundCalendarRestrictions;
        if (!CalendarRestrictions.Check(ModelingNow(), foundCalendarRestrictions)) {
            WARNING_LOG << "Skip all fines charging due to calendar restrictions: " << foundCalendarRestrictions.GetStringReport() << Endl;
            NotifyHandlers->Handle(EProcessingStatus::CalendarRestrictionsSkip, server);
            return nextNativeState;
        }
    }


    TVector<NDrive::NFine::TAutocodeFineEntry> fines;
    TFineIdToEventPtrMapping fineIdToEventPtrMapping;
    {
        auto tx = server.GetDriveAPI()->GetFinesManager().BuildTx<NSQL::ReadOnly | NSQL::Deferred>();
        EProcessingStatus limitsCalculationFetchResult = CalculateDerivedLimits(server, tx);
        NotifyHandlers->Handle(limitsCalculationFetchResult, server);
        if (limitsCalculationFetchResult != EProcessingStatus::FetchFinesSuccess) {
            if (limitsCalculationFetchResult == EProcessingStatus::TimeIntervalLimitSkip) {
                return nextNativeState;
            }
            return MakeUnexpected<TString>({});
        }

        auto optFines = FetchFines(server, fineIdToEventPtrMapping, lastProcessedSerialId, lastProcessedEventId, tx);
        R_ENSURE(optFines, {}, "Fail to fetch fines", tx);
        fines = std::move(*optFines);
    }

    if (!fines.empty()) {
        ThreadPool.Start(ProcessingThreadsCount);  // restart old pool

        TVector<NThreading::TFuture<void>> tasks;

        std::atomic<EProcessingResult> status = EProcessingResult::OK;
        TAtomicCounter processedCount = 0;
        i64 processedTotalAmountCents = 0;  // increased in case of amount limit set only as mutex is used

        for (const auto& fine : fines) {
            auto task = NThreading::Async([this, &fine, &server, &fineIdToEventPtrMapping, &status, &processedCount, &processedTotalAmountCents] {
                if (status.load() >= EProcessingResult::STOP_REQUIRED) {
                    return;
                }

                EProcessingResult processingResult = ProcessFine(server, fine, fineIdToEventPtrMapping, processedCount, processedTotalAmountCents);
                TUnistatSignalsCache::SignalAdd(GetRTProcessName() + "-" + "process_fine", ::ToString(processingResult), 1);

                if (processingResult >= EProcessingResult::STOP_REQUIRED) {
                    auto expected = EProcessingResult::OK;
                    status.compare_exchange_strong(expected, processingResult);
                }
            }, ThreadPool);

            tasks.push_back(task);
        }

        auto waiter = NThreading::WaitExceptionOrAll(tasks);
        waiter.Wait();  // no timeout

        ThreadPool.Stop();  // destroy pool; it will terminate as much in dtor in case of exc

        if (status.load() == EProcessingResult::CRITICAL_ERROR) {
            return MakeUnexpected<TString>({});
        }

        auto extreme = MaxElementBy(fines, [](const auto& entry) { return entry.GetSerialId(); });  // fines are not empty
        nextNativeState->SetLastSerialId(extreme->GetSerialId());
    }

    if (!!TagEventsHandlerPtr) {
        nextNativeState->SetLastTagHistoryEventId(TagEventsHandlerPtr->GetLastProcessedEventId());
    }

    return nextNativeState;
}

NDrive::NFine::TOptionalFines TRTFinesHandlerBase::FetchFines(const NDrive::IServer& server, TFineIdToEventPtrMapping& fineIdToEventPtrMapping, const ui64 lastProcessedSerialId, const ui64 lastProcessedEventId, NDrive::TEntitySession& tx) const {
    const auto& finesManager = server.GetDriveAPI()->GetFinesManager();

    if (!PrepareEventsToProcess(server, lastProcessedEventId, fineIdToEventPtrMapping)) {
        tx.SetErrorInfo("FetchFinesError", "Fail to process events", EDriveSessionResult::InternalError);
        return {};
    }

    NDrive::NFine::TFineFilterGroup filters;
    if (!GetFilters(server, lastProcessedSerialId, fineIdToEventPtrMapping, filters)) {
        tx.SetErrorInfo("FetchFinesError", "Fail to get filters", EDriveSessionResult::InternalError);
        return {};
    }

    if (!!UserFilter) {
        return finesManager.GetFinesByUserId(UserFilter, filters, tx);
    } else {
        return finesManager.GetFines(filters, tx);
    }

    return {};
}

bool TRTFinesHandlerBase::PrepareEventsToProcess(const NDrive::IServer& server, const ui64 lastProcessedEventId, TFineIdToEventPtrMapping& fineIdToEventPtrMapping) const {
    if (!!TagEventsHandlerPtr && IsHandleActionTags()) {
        const bool fetchResult = TagEventsHandlerPtr->Fetch(server, lastProcessedEventId, DataActuality);
        if (!fetchResult) {
            return false;
        }

        auto tx = TagEventsHandlerPtr->GetEntityTagsManager(server).BuildTx<NSQL::ReadOnly>();
        while (auto eventPtr = TagEventsHandlerPtr->Next(server)) {
            auto fineActionTagPtr = eventPtr->GetTagAs<TFineActionTag>();
            if (!!fineActionTagPtr && TagEventsHandlerPtr->DoesTagExist(eventPtr, server, tx)) {
                fineIdToEventPtrMapping.emplace(fineActionTagPtr->GetFineId(), eventPtr);
            }
        }
    }
    return true;
}

bool TRTFinesHandlerBase::GetFilters(const NDrive::IServer& server, const ui64 lastProcessedSerialId, TFineIdToEventPtrMapping& fineIdToEventPtrMapping, NDrive::NFine::TFineFilterGroup& filters) const {
    const auto& finesManager = server.GetDriveAPI()->GetFinesManager();

    auto articleMatcherPtr = finesManager.GetFineArticleMatcherPtr(/* ensureActual = */ true);
    if (!articleMatcherPtr) {
        return false;
    }

    if (IsHandleActionTags()) {
        if (!!fineIdToEventPtrMapping) {
            filters.Append(MakeAtomicShared<NDrive::NFine::TMultipleFineIdsFilter>(NContainer::Keys(fineIdToEventPtrMapping)));
        } else {
            filters.Append(MakeAtomicShared<NDrive::NFine::TRejectFilter>());  // skip all fines
        }
    }

    if (IsNotChargedOnly()) {
        filters.Append(MakeAtomicShared<NDrive::NFine::TFineNotChargedFilter>());
    }

    if (IsChargeableOnly()) {
        filters.Append(MakeAtomicShared<NDrive::NFine::TFineChargeableFilter>());
    }

    if (IsProcessIteratively()) {
        filters.Append(MakeAtomicShared<NDrive::NFine::TFineSerialIdFilter>(static_cast<ui32>(lastProcessedSerialId + 1)));
    }

    if (IsAbsentDetailedViolationDocumentOnly()) {
        filters.Append(MakeAtomicShared<NDrive::NFine::TFineAbsentDetailedViolationDocumentFilter>());
    }

    if (HasDetailedViolationDocumentOnly) {
        filters.Append(MakeAtomicShared<NDrive::NFine::TFineHasDetailedViolationDocumentFilter>());
    }

    if (!!ArticlesFilter) {
        filters.Append(MakeAtomicShared<NDrive::NFine::TFineArticleFilter>(articleMatcherPtr, ArticlesFilter, IsExcludeArticles()));
    }

    if (IsUnknownArticleOnly()) {
        filters.Append(MakeAtomicShared<NDrive::NFine::TFineUnknownArticleFilter>(articleMatcherPtr));
    }

    if (!!CarsFilter) {
        if (CarsFilter.size() == 1) {
            filters.Append(MakeAtomicShared<NDrive::NFine::TFineCarFilter>(*CarsFilter.begin()));
        } else {
            filters.Append(MakeAtomicShared<NDrive::NFine::TFineMultipleCarsFilter>(CarsFilter));
        }
    }

    if (!UsersIgnore.empty()) {
        filters.Append(MakeAtomicShared<NDrive::NFine::TFineMultipleUsersFilter>(UsersIgnore, /* contains = */ false));
    }

    if (!!UserFilter) {
        filters.Append(MakeAtomicShared<NDrive::NFine::TFineUserFilter>(UserFilter));
    }

    if (!!SourceTypeFilter) {
        if (SourceTypeFilter.size() == 1) {
            filters.Append(MakeAtomicShared<NDrive::NFine::TFineSourceTypeFilter>(*SourceTypeFilter.begin()));
        } else {
            filters.Append(MakeAtomicShared<NDrive::NFine::TFineMultipleSourcesTypeFilter>(SourceTypeFilter));
        }
    }

    if (MinSumToPayCents > 0 || MaxSumToPayCents > 0) {
        filters.Append(MakeAtomicShared<NDrive::NFine::TFineSumToPayCentsFilter>(MinSumToPayCents, (MaxSumToPayCents > 0) ? MaxSumToPayCents : std::numeric_limits<decltype(MaxSumToPayCents)>::max()));
    }

    TVector<const NDrive::NFine::IDateTimeFilterConfig*> dateFilterConfigs = { &GetRulingDateFilterConfig(), &GetViolationTimeFilterConfig(), &GetFineReceiveTimeFilterConfig() };
    for (auto dateFilterConfig : dateFilterConfigs) {
        if (dateFilterConfig->HasRestrictions()) {
            filters.Append(dateFilterConfig->GetFilter());
        }
    }

    return true;
}

TRTFinesHandlerBase::EProcessingStatus TRTFinesHandlerBase::CalculateDerivedLimits(const NDrive::IServer& server, NDrive::TEntitySession& tx) const {
    // NB. Limits are not **intended** to be updated during fines processing however other charging processes may affect to the current one
    // set derived limits to current run limits
    DerivedTotalAmountCentsLimit = TotalAmountCentsLimit;
    DerivedChargesLimit = ChargesLimit;

    if (!TimeChargesLimitInterval || (!TimeTotalAmountCentsLimit && !TimeChargesLimit)) {
        return EProcessingStatus::FetchFinesSuccess;
    }

    const auto now = ModelingNow();

    NDrive::NFine::TFineFilterGroup filters = { MakeAtomicShared<NDrive::NFine::TFineChargeTimeFilter>(now - TimeChargesLimitInterval, now) };

    const auto& amount = server.GetDriveAPI()->GetFinesManager().GetFineAmounts(filters, tx);
    R_ENSURE(amount, {}, "Fail to fetch amount fo derived limits", tx);

    if (!!TimeTotalAmountCentsLimit) {
        i64 totalAmountCents = std::accumulate(amount->cbegin(), amount->cend(), 0ll);

        if (totalAmountCents >= TimeTotalAmountCentsLimit) {
            return EProcessingStatus::TimeIntervalLimitSkip;
        }

        if (!!DerivedTotalAmountCentsLimit) {
            DerivedTotalAmountCentsLimit = Min<decltype(DerivedTotalAmountCentsLimit)>(TimeTotalAmountCentsLimit - totalAmountCents, DerivedTotalAmountCentsLimit);
        } else {
            DerivedTotalAmountCentsLimit = TimeTotalAmountCentsLimit - totalAmountCents;
        }
    }

    if (!!TimeChargesLimit) {
        size_t count = amount->size();

        if (count >= TimeChargesLimit) {
            return EProcessingStatus::TimeIntervalLimitSkip;
        }

        if (!!DerivedChargesLimit) {
            DerivedChargesLimit = Min<decltype(DerivedChargesLimit)>(TimeChargesLimit - count, DerivedChargesLimit);
        } else {
            DerivedChargesLimit = TimeChargesLimit - count;
        }
    }

    return EProcessingStatus::FetchFinesSuccess;
}

TRTFinesHandlerBase::EProcessingResult TRTFinesHandlerBase::ProcessFine(const NDrive::IServer& server, NDrive::NFine::TAutocodeFineEntry fine, TFineIdToEventPtrMapping& fineIdToEventPtrMapping, TAtomicCounter& processedCount, i64& processedTotalAmountCents) const {
    if (!!DerivedChargesLimit) {  // unlimited if zero
        if (processedCount.Val() >= static_cast<TAtomicBase>(DerivedChargesLimit)) {
            WARNING_LOG << "Processed items limit exceeded: " << processedCount.Val() << " of " << DerivedChargesLimit << Endl;
            return EProcessingResult::LIMIT_REACHED;
        }
    }

    if (!!DerivedTotalAmountCentsLimit) {  // unlimited if zero
        TReadGuard rg(TotalAmountCentsMutex);
        if (processedTotalAmountCents >= DerivedTotalAmountCentsLimit) {
            WARNING_LOG << "Processed items total amount cents limit exceeded: " << processedTotalAmountCents << " of " << DerivedTotalAmountCentsLimit << Endl;
            return EProcessingResult::LIMIT_REACHED;
        }
    }

    NDrive::NFine::TFineFetchContext fineFetchContext(GetFetchContextConfig(), server, fine);

    auto session = server.GetDriveAPI()->GetTagsManager().GetUserTags().BuildTx<NSQL::Writable | NSQL::Deferred>();
    EProcessingStatus status = CheckFineRestrictions(server, fine, session);
    NotifyHandlers->Handle(status, fineFetchContext);
    if (status != EProcessingStatus::CheckFineSuccess) {
        NDrive::TEventLog::Log("SkipFineProcess", NJson::TMapBuilder
            ("reason", ToString(status))
            ("fine_id", fine.GetId())
        );
        ERROR_LOG << "Error checking fines " << session.GetStringReport() << Endl;
        return EProcessingResult::SKIP;
    }

    TMessagesCollector fineProcessingErrors;
    IUpdateContext::TPtr updateContext = ConstructUpdateContext();
    status = PrepareUpdate(server, updateContext, fine, fineIdToEventPtrMapping, fineProcessingErrors);
    NotifyHandlers->Handle(status, fineFetchContext);
    if (status != EProcessingStatus::DataFetchingSuccess) {
        ERROR_LOG << fineProcessingErrors.GetStringReport() << Endl;
        return EProcessingResult::SKIP;
    }

    if (!HandleFineRelatedTags(server, updateContext, session)) {
        NotifyHandlers->Handle(EProcessingStatus::TagUpsertError, fineFetchContext);
        ERROR_LOG << "Error handle tags related to fine " << fine.GetId() << ": " << session.GetStringReport() << Endl;
        return EProcessingResult::CRITICAL_ERROR;
    }

    status = ApplyUpdate(server, updateContext, fine, session, fineProcessingErrors);
    NotifyHandlers->Handle(status, fineFetchContext);
    if (status != EProcessingStatus::FineUpsertSuccess) {
        NDrive::TEventLog::Log("FineUpsertError", fineProcessingErrors.GetReport());
        ERROR_LOG << fineProcessingErrors.GetStringReport() << Endl;
        return EProcessingResult::CRITICAL_ERROR;
    }

    if (!session.Commit()) {
        NDrive::TEventLog::Log("FineUpsertCommitError", fineProcessingErrors.GetReport());
        ERROR_LOG << "Error processing fine " << fine.GetId() << ": " << session.GetStringReport() << Endl;
        return EProcessingResult::CRITICAL_ERROR;
    }

    processedCount.Inc();

    if (!!DerivedTotalAmountCentsLimit) {  // unlimited if zero
        TWriteGuard wg(TotalAmountCentsMutex);
        processedTotalAmountCents += fine.GetSumToPayCents();
    }

    NotifyHandlers->Handle(EProcessingStatus::Success, fineFetchContext);

    return EProcessingResult::OK;
}

NDrive::NFine::EProcessingStatus TRTFinesHandlerBase::CheckFineRestrictions(const NDrive::IServer& server, const NDrive::NFine::TAutocodeFineEntry& fine, NDrive::TEntitySession& tx) const {
    const NDrive::NFine::TFinesManager& finesManager = server.GetDriveAPI()->GetFinesManager();

    auto finePhotos = finesManager.GetFineAttachments(fine.GetId(), NDrive::NFine::TFinesManager::EFineAttachmentType::Photo, tx);
    if (!finePhotos) {
        return EProcessingStatus::DataFetchingError;
    }
    if (finePhotos->empty()) {
        if (fine.GetFineInformationReceivedAt() + PhotoWaitTimeout > ModelingNow()) {
            return EProcessingStatus::WaitingForPhotoSkip;  // EProcessingStatus::PhotoRestrictionsSkip can be used instead
        }
    } else {
        if (IsWithoutPhotoOnly()) {
            return EProcessingStatus::PhotoRestrictionsSkip;  // logging will be too verbose here
        }
    }

    if (IsAbsentDecreeOnly() || HasDecreeOnly) {
        auto fineDecrees = finesManager.GetFineAttachments(fine.GetId(), NDrive::NFine::TFinesManager::EFineAttachmentType::Decree, tx);
        if (!fineDecrees) {
            return EProcessingStatus::DataFetchingError;
        }
        if ((fineDecrees->empty() && HasDecreeOnly) || (!fineDecrees->empty() && IsAbsentDecreeOnly())) {
            return EProcessingStatus::DecreeRestrictionsSkip;
        }
    }

    return EProcessingStatus::CheckFineSuccess;
}

bool TRTFinesHandlerBase::HandleFineRelatedTags(const NDrive::IServer& server, IUpdateContext::TPtr context, NDrive::TEntitySession& tx) const {
    if (IsHandleActionTags() && !!TagEventsHandlerPtr && !!context->GetRelatedEventPtr()) {
        return TagEventsHandlerPtr->RemoveTagIfExists(context->GetRelatedEventPtr(), GetRobotUserId(), server, tx);
    }
    return true;
}

NDrive::TScheme TRTFinesHandlerBase::DoGetScheme(const IServerBase& server) const {
    auto scheme = TBase::DoGetScheme(server);

    scheme.Add<TFSNumeric>("processing_threads_count", "Количество потоков обработки").SetMin(1).SetMax(64).SetDefault(1);
    scheme.Add<TFSBoolean>("process_iteratively", "Обрабатывать данные итеративно (учитывать уже обработанные)").SetDefault(false);
    scheme.Add<TFSBoolean>("handle_action_tags", "Обрабатывать теги действий").SetDefault(false);

    scheme.Add<TFSArray>("articles_filter", "Статьи штрафов для списания (напр. 8_25, 12_09_3); все если не задано").SetElement<TFSString>();
    scheme.Add<TFSBoolean>("do_exclude_articles", "Для всех статей, кроме указанных").SetDefault(false);
    scheme.Add<TFSBoolean>("unknown_article_only", "Только с неизвестным кодом").SetDefault(false);

    scheme.Add<TFSArray>("cars_filter", "Определенные машины (id)").SetElement<TFSString>();
    scheme.Add<TFSArray>("users_ignore", "Игнорировать пользователей (id)").SetElement<TFSString>();
    scheme.Add<TFSString>("user_filter", "Определенный пользователь (id)");

    scheme.Add<TFSBoolean>("not_charged_only", "Только не списанные штрафы (согласуется с активностью тега списаний)").SetDefault(true);
    scheme.Add<TFSBoolean>("chargeable_only", "Только штрафы к списанию").SetDefault(true);

    scheme.Add<TFSBoolean>("absent_detailed_violation_document_only", "Только без сгенерированных документов").SetDefault(false);
    scheme.Add<TFSBoolean>("has_detailed_violation_document_only", "Только со сгенерированным документом").SetDefault(false);

    scheme.Add<TFSBoolean>("absent_decree_only", "Только без копии постановления").SetDefault(false);
    scheme.Add<TFSBoolean>("has_decree_only", "Только с копией постановления").SetDefault(false);

    scheme.Add<TFSVariants>("source_type_filter", "Источник штрафов").SetVariants(GetEnumAllValues<NDrive::NFine::ESourceType>()).SetMultiSelect(true);

    scheme.Add<TFSNumeric>("sum_to_pay_cents_min", "Ограничение на мин. сумму штрафа (коп.)").SetMin(0).SetDefault(0);
    scheme.Add<TFSNumeric>("sum_to_pay_cents_limit", "Ограничение на макс. сумму штрафа (коп.); без ограничений если ноль").SetMin(0).SetDefault(500000);
    scheme.Add<TFSNumeric>("total_amount_cents_limit", "Ограничение на общую сумму штрафов (коп.); без ограничений если ноль").SetMin(0).SetDefault(0);
    scheme.Add<TFSNumeric>("charges_limit", "Ограничение на количество обработанных штрафов; без ограничений если ноль").SetMin(0).SetDefault(0);

    scheme.Add<TFSNumeric>("time_total_amount_cents_limit", "Ограничение на общую сумму штрафов (коп.) за промежуток времени; без ограничений если ноль").SetMin(0).SetDefault(0);
    scheme.Add<TFSNumeric>("time_charges_limit", "Ограничение на количество штрафов за промежуток времени; без ограничений если ноль").SetMin(0).SetDefault(0);
    scheme.Add<TFSDuration>("time_charges_limit_interval", "Промежуток времени для проверки ограничений").SetDefault(TDuration::Zero());

    scheme.Add<TFSDuration>("photo_wait_timeout", "Время ожидания загрузки фото").SetDefault(DefaultPhotoWaitTimeout);
    scheme.Add<TFSBoolean>("without_photo_only", "Только без фото").SetDefault(false);

    scheme.Add<TFSStructure>("ruling_date_filter", "Диапазон дат постановлений").SetStructure(NDrive::NFine::TRulingDateFilterConfig::GetScheme());
    scheme.Add<TFSStructure>("violation_time_filter", "Диапазон времени нарушения").SetStructure(NDrive::NFine::TViolationTimeFilterConfig::GetScheme());
    scheme.Add<TFSStructure>("fine_receive_time_filter", "Диапазон времени получения информации").SetStructure(NDrive::NFine::TFineReceiveTimeFilterConfig::GetScheme());

    scheme.Add<TFSStructure>("calendar_restrictions", "Ограничения на списания по дате и времени").SetStructure(NCalendarRestrictions::TCalendarRestrictions::GetScheme());

    scheme.Add<TFSStructure>("fetch_context_config", "Параметры подстановщика шаблонов").SetStructure(NDrive::NFine::TFineFetchContextConfig::GetScheme());

    scheme.Add<TFSStructure>("notify_handlers", "Настройки нотификаторов").SetStructure(TNotifyHandlers::GetScheme(server));

    return scheme;
}

bool TRTFinesHandlerBase::DoDeserializeFromJson(const NJson::TJsonValue& data) {
    if (!TBase::DoDeserializeFromJson(data)) {
        return false;
    }

    if (!TJsonProcessor::Read(data, "processing_threads_count", ProcessingThreadsCount)) {
        return false;
    }
    if (!TJsonProcessor::Read(data, "process_iteratively", ProcessIteratively)) {
        return false;
    }
    if (!TJsonProcessor::Read(data, "handle_action_tags", HandleActionTags)) {
        return false;
    }

    if (!TJsonProcessor::ReadContainer(data, "articles_filter", ArticlesFilter)) {
        return false;
    }
    if (!TJsonProcessor::Read(data, "do_exclude_articles", ExcludeArticles)) {
        return false;
    }
    if (!TJsonProcessor::Read(data, "unknown_article_only", UnknownArticleOnly)) {
        return false;
    }

    if (!TJsonProcessor::ReadContainer(data, "cars_filter", CarsFilter)) {
        return false;
    }
    if (!TJsonProcessor::ReadContainer(data, "users_ignore", UsersIgnore)) {
        return false;
    }
    if (!TJsonProcessor::Read(data, "user_filter", UserFilter)) {
        return false;
    }

    if (!TJsonProcessor::Read(data, "not_charged_only", NotChargedOnly)) {
        return false;
    }
    if (!TJsonProcessor::Read(data, "chargeable_only", ChargeableOnly)) {
        return false;
    }

    if (!TJsonProcessor::Read(data, "absent_detailed_violation_document_only", AbsentDetailedViolationDocumentOnly)) {
        return false;
    }
    if (!TJsonProcessor::Read(data, "has_detailed_violation_document_only", HasDetailedViolationDocumentOnly)) {
        return false;
    }

    if (!TJsonProcessor::Read(data, "absent_decree_only", AbsentDecreeOnly)) {
        return false;
    }
    if (!TJsonProcessor::Read(data, "has_decree_only", HasDecreeOnly)) {
        return false;
    }

    if (!TJsonProcessor::ReadContainer(data, "source_type_filter", SourceTypeFilter)) {
        return false;
    }

    JREAD_INT_OPT(data, "sum_to_pay_cents_min", MinSumToPayCents);
    JREAD_INT_OPT(data, "sum_to_pay_cents_limit", MaxSumToPayCents);
    JREAD_INT_OPT(data, "total_amount_cents_limit", TotalAmountCentsLimit);
    if (!TJsonProcessor::Read(data, "charges_limit", ChargesLimit)) {
        return false;
    }

    JREAD_INT_OPT(data, "time_total_amount_cents_limit", TimeTotalAmountCentsLimit);
    if (!TJsonProcessor::Read(data, "time_charges_limit", TimeChargesLimit)) {
        return false;
    }
    if (!TJsonProcessor::Read(data, "time_charges_limit_interval", TimeChargesLimitInterval)) {
        return false;
    }

    if (!TJsonProcessor::Read(data, "photo_wait_timeout", PhotoWaitTimeout)) {
        return false;
    }
    if (!TJsonProcessor::Read(data, "without_photo_only", WithoutPhotoOnly)) {
        return false;
    }

    if (!RulingDateFilterConfig.DeserializeFromJson(data["ruling_date_filter"])) {
        return false;
    }
    if (!ViolationTimeFilterConfig.DeserializeFromJson(data["violation_time_filter"])) {
        return false;
    }
    if (!FineReceiveTimeFilterConfig.DeserializeFromJson(data["fine_receive_time_filter"])) {
        return false;
    }

    if (!CalendarRestrictions.DeserializeFromJson(data["calendar_restrictions"])) {
        return false;
    }

    if (!FetchContextConfig.DeserializeFromJson(data["fetch_context_config"])) {
        return false;
    }

    if (!!NotifyHandlers && !NotifyHandlers->DeserializeFromJson(data["notify_handlers"])) {
        return false;
    }

    if (!!ArticlesFilter && IsUnknownArticleOnly()) {  // to fix empty article parsing and merge this two cases
        ERROR_LOG << "Article filter and only unknown article processing are not compatible" << Endl;
        return false;
    }

    return true;
}

NJson::TJsonValue TRTFinesHandlerBase::DoSerializeToJson() const {
    NJson::TJsonValue result = TBase::DoSerializeToJson();

    TJsonProcessor::Write(result, "processing_threads_count", ProcessingThreadsCount);
    TJsonProcessor::Write(result, "process_iteratively", ProcessIteratively);
    TJsonProcessor::Write(result, "handle_action_tags", HandleActionTags);

    TJsonProcessor::WriteContainerArray(result, "articles_filter", ArticlesFilter);
    TJsonProcessor::Write(result, "do_exclude_articles", ExcludeArticles);
    TJsonProcessor::Write(result, "unknown_article_only", UnknownArticleOnly);

    TJsonProcessor::WriteContainerArray(result, "cars_filter", CarsFilter);
    TJsonProcessor::WriteContainerArray(result, "users_ignore", UsersIgnore);
    TJsonProcessor::Write(result, "user_filter", UserFilter);

    TJsonProcessor::Write(result, "not_charged_only", NotChargedOnly);
    TJsonProcessor::Write(result, "chargeable_only", ChargeableOnly);

    TJsonProcessor::Write(result, "absent_detailed_violation_document_only", AbsentDetailedViolationDocumentOnly);
    TJsonProcessor::Write(result, "has_detailed_violation_document_only", HasDetailedViolationDocumentOnly);

    TJsonProcessor::Write(result, "absent_decree_only", AbsentDecreeOnly);
    TJsonProcessor::Write(result, "has_decree_only", HasDecreeOnly);

    TJsonProcessor::WriteContainerArrayStrings(result, "source_type_filter", SourceTypeFilter);

    TJsonProcessor::Write(result, "sum_to_pay_cents_min", MinSumToPayCents);
    TJsonProcessor::Write(result, "sum_to_pay_cents_limit", MaxSumToPayCents);
    TJsonProcessor::Write(result, "total_amount_cents_limit", TotalAmountCentsLimit);
    TJsonProcessor::Write(result, "charges_limit", ChargesLimit);

    TJsonProcessor::Write(result, "time_total_amount_cents_limit", TimeTotalAmountCentsLimit);
    TJsonProcessor::Write(result, "time_charges_limit", TimeChargesLimit);
    TJsonProcessor::WriteDurationString(result, "time_charges_limit_interval", TimeChargesLimitInterval);

    TJsonProcessor::WriteDurationString(result, "photo_wait_timeout", PhotoWaitTimeout);
    TJsonProcessor::Write(result, "without_photo_only", WithoutPhotoOnly);

    TJsonProcessor::WriteSerializable(result, "ruling_date_filter", RulingDateFilterConfig);
    TJsonProcessor::WriteSerializable(result, "violation_time_filter", ViolationTimeFilterConfig);
    TJsonProcessor::WriteSerializable(result, "fine_receive_time_filter", FineReceiveTimeFilterConfig);

    TJsonProcessor::WriteSerializable(result, "calendar_restrictions", CalendarRestrictions);

    TJsonProcessor::WriteSerializable(result, "fetch_context_config", FetchContextConfig);

    if (!!NotifyHandlers) {
        TJsonProcessor::WriteSerializable(result, "notify_handlers", *NotifyHandlers);
    }

    return result;
}
