#pragma once

#include "events.h"
#include "statuses.h"

#include <drive/backend/rt_background/manager/settings.h>
#include <drive/backend/rt_background/manager/state.h>

#include <drive/backend/fines/constants.h>
#include <drive/backend/fines/filters.h>
#include <drive/backend/fines/context_fetchers/fine_context.h>
#include <drive/backend/fines/restrictions/calendar.h>
#include <drive/backend/notifications/collection.h>
#include <drive/backend/proto/background.pb.h>

#include <rtline/util/types/accessor.h>

#include <util/generic/refcount.h>
#include <util/generic/serialized_enum.h>
#include <util/thread/pool.h>

class TRTFinesHandlerBaseState : public IProtoStateSerializable<NDrive::NProto::TRTFinesHandlerState> {
    using TBase = IProtoStateSerializable<NDrive::NProto::TRTFinesHandlerState>;

public:
    using TBase::TBase;

    TString GetType() const override = 0;

    virtual NJson::TJsonValue GetReport() const override;
    virtual NDrive::TScheme DoGetScheme() const override;

protected:
    virtual void SerializeToProto(NDrive::NProto::TRTFinesHandlerState& proto) const override;
    virtual bool DeserializeFromProto(const NDrive::NProto::TRTFinesHandlerState& proto) override;

private:
    R_FIELD(ui64, LastSerialId, Max<ui64>());
    R_FIELD(ui64, LastTagHistoryEventId, Max<ui64>());
};

class IFinesHandlerUpdateContext {
public:
    using TPtr = TAtomicSharedPtr<IFinesHandlerUpdateContext>;

    virtual ~IFinesHandlerUpdateContext() = default;

private:
    R_FIELD(TEventsHandler::TEventPtr, RelatedEventPtr, nullptr);
};

class TRTFinesHandlerBase: public IRTRegularBackgroundProcess {
private:
    using TBase = IRTRegularBackgroundProcess;

    static const TDuration DefaultPhotoWaitTimeout;

protected:
    using EProcessingStatus = NDrive::NFine::EProcessingStatus;
    using EProcessingResult = NDrive::NFine::EFineProcessingResult;

    using TNotifyHandlers = TContextNotifyHandlerCollection<EProcessingStatus, NDrive::NFine::IFineContextFetcher>;

    using IUpdateContext = IFinesHandlerUpdateContext;

    using TFineIdToEventPtrMapping = TMap<TString, TEventsHandler::TEventPtr>;

public:
    explicit TRTFinesHandlerBase(TEventsHandler::TPtr tagEventsHandlerPtr);

    virtual TExpectedState DoExecute(TAtomicSharedPtr<IRTBackgroundProcessState> state, const TExecutionContext& context) const override;

    virtual EProcessingResult ProcessFine(const NDrive::IServer& server, NDrive::NFine::TAutocodeFineEntry fine, TFineIdToEventPtrMapping& fineIdToEventPtrMapping, TAtomicCounter& chargedCount, i64& totalAmountCentsCharged) const;

    virtual NDrive::TScheme DoGetScheme(const IServerBase& server) const override;
    virtual bool DoDeserializeFromJson(const NJson::TJsonValue& data) override;
    virtual NJson::TJsonValue DoSerializeToJson() const override;

protected:
    virtual bool DoStart(const TRTBackgroundProcessContainer& container) override;
    virtual void InitNotifyHandlers() final;

    virtual TAtomicSharedPtr<TRTFinesHandlerBaseState> BuildState(const ui64 lastProcessedSerialId, const ui64 lastProcessedEventId) const = 0;

    virtual NDrive::NFine::TOptionalFines FetchFines(const NDrive::IServer& server, TFineIdToEventPtrMapping& fineIdToEventPtrMapping, const ui64 lastProcessedSerialId, const ui64 lastProcessedEventId, NDrive::TEntitySession& tx) const;
    virtual bool PrepareEventsToProcess(const NDrive::IServer& server, const ui64 lastProcessedEventId, TFineIdToEventPtrMapping& fineIdToEventPtrMapping) const;
    virtual bool GetFilters(const NDrive::IServer& server, const ui64 lastProcessedSerialId, TFineIdToEventPtrMapping& fineIdToEventPtrMapping, NDrive::NFine::TFineFilterGroup& filters) const;

    virtual EProcessingStatus CalculateDerivedLimits(const NDrive::IServer& server, NDrive::TEntitySession& tx) const;

    virtual EProcessingStatus CheckFineRestrictions(const NDrive::IServer& server, const NDrive::NFine::TAutocodeFineEntry& fine, NDrive::TEntitySession& tx) const;

    virtual IUpdateContext::TPtr ConstructUpdateContext() const = 0;
    virtual EProcessingStatus PrepareUpdate(const NDrive::IServer& server, IUpdateContext::TPtr updateContextPtr, NDrive::NFine::TAutocodeFineEntry& fine, TFineIdToEventPtrMapping& fineIdToEventPtrMapping, TMessagesCollector& errors) const = 0;
    virtual bool HandleFineRelatedTags(const NDrive::IServer& server, IUpdateContext::TPtr updateContextPtr, NDrive::TEntitySession& tx) const;
    virtual EProcessingStatus ApplyUpdate(const NDrive::IServer& server, IUpdateContext::TPtr updateContextPtr, NDrive::NFine::TAutocodeFineEntry& fine, NDrive::TEntitySession& tx, TMessagesCollector& errors) const = 0;

private:
    R_FIELD(ui32, ProcessingThreadsCount, 1);
    R_FIELD(bool, ProcessIteratively, false);
    R_FIELD(bool, HandleActionTags, false);

    // note: no need to filter empty fine articles as charge tags won't be constructed
    R_FIELD(TSet<TString>, ArticlesFilter);
    R_FIELD(bool, ExcludeArticles, false);
    R_FIELD(bool, UnknownArticleOnly, false);

    R_FIELD(TSet<TString>, CarsFilter);
    R_FIELD(TSet<TString>, UsersIgnore);
    R_FIELD(TString, UserFilter);

    R_FIELD(bool, NotChargedOnly, true);
    R_FIELD(bool, ChargeableOnly, true);

    R_FIELD(bool, AbsentDetailedViolationDocumentOnly, false);
    R_FIELD(bool, HasDetailedViolationDocumentOnly, false);

    R_FIELD(bool, AbsentDecreeOnly, false);
    R_FIELD(bool, HasDecreeOnly, false);

    R_FIELD(TSet<NDrive::NFine::ESourceType>, SourceTypeFilter);

    R_FIELD(i64, MinSumToPayCents, 0);
    R_FIELD(i64, MaxSumToPayCents, 0);
    R_FIELD(i64, TotalAmountCentsLimit, 0);
    R_FIELD(ui32, ChargesLimit, 0);

    R_FIELD(i64, TimeTotalAmountCentsLimit, 0);
    R_FIELD(ui32, TimeChargesLimit, 0);
    R_FIELD(TDuration, TimeChargesLimitInterval, TDuration::Zero());

    R_FIELD(TDuration, PhotoWaitTimeout, DefaultPhotoWaitTimeout);
    R_FIELD(bool, WithoutPhotoOnly, false);

    R_FIELD(NDrive::NFine::TRulingDateFilterConfig, RulingDateFilterConfig);
    R_FIELD(NDrive::NFine::TViolationTimeFilterConfig, ViolationTimeFilterConfig);
    R_FIELD(NDrive::NFine::TFineReceiveTimeFilterConfig, FineReceiveTimeFilterConfig);

    R_FIELD(NCalendarRestrictions::TCalendarRestrictions, CalendarRestrictions);

    R_FIELD(NDrive::NFine::TFineFetchContextConfig, FetchContextConfig);

    R_FIELD(TNotifyHandlers::TPtr, NotifyHandlers, nullptr);

    R_FIELD(TEventsHandler::TPtr, TagEventsHandlerPtr, {}, mutable);

    mutable TSimpleThreadPool ThreadPool;

    // run limits corrected with time interval limitations (refer to CalculateDerivedLimits)
    mutable TRWMutex TotalAmountCentsMutex;
    mutable i64 DerivedTotalAmountCentsLimit;
    mutable ui32 DerivedChargesLimit;
};
