#include "major_decree_processor.h"

#include <drive/backend/database/drive_api.h>
#include <drive/backend/fines/manager.h>
#include <drive/backend/fines/constructors/fine_attachment_constructor.h>
#include <drive/backend/major/client.h>

const TString TRTMajorFineDecreeCollector::TypeName = "fines_major_decree_processor";

TString TRTMajorFineDecreeCollectorState::GetTypeName() {
    return TRTMajorFineDecreeCollector::TypeName;
}

IRTRegularBackgroundProcess::TFactory::TRegistrator<TRTMajorFineDecreeCollector> TRTMajorFineDecreeCollector::Registrator(TRTMajorFineDecreeCollector::TypeName);
TRTHistoryWatcherState::TFactory::TRegistrator<TRTMajorFineDecreeCollectorState> TRTMajorFineDecreeCollectorState::Registrator(TRTMajorFineDecreeCollectorState::GetTypeName());

TMajorFineDecreeCollectorContext::TMajorFineDecreeCollectorContext(
    const NDrive::IServer& server,
    const TAttachmentConstructor& fineAttachmentConstructor,
    const TString& fineId,
    const TString& rulingNumber,
    TEventsHandler::TEventPtr relatedEventPtr
)
    : TBase(server, fineAttachmentConstructor, fineId, rulingNumber, relatedEventPtr)
{
}

size_t TMajorFineDecreeCollectorContext::GetCollectedTotal() const {
    return 1;
}

TRTMajorFineDecreeCollector::TRTMajorFineDecreeCollector()
    : TBase(MakeAtomicShared<TEventsHandler>(NEntityTagsManager::EEntityType::Car))
{
    MutableProtectedTagEventsHandlerPtr()->AddFilter(MakeAtomicShared<TFineActionTagFilter>(EObjectHistoryAction::Add, EFineActionType::RequestDecree));

    InitNotifyHandlers();
}

TExpectedState TRTMajorFineDecreeCollector::DoExecute(TAtomicSharedPtr<IRTBackgroundProcessState> state, const TExecutionContext& context) const {
    const auto& server = context.GetServerAs<NDrive::IServer>();
    CHECK_WITH_LOG(!!server.GetDriveAPI());

    if (!server.GetDriveAPI()->HasFinesManager() || !server.GetDriveAPI()->HasMajorClient()) {
        ERROR_LOG << "Fines manager or Major client is undefined" << Endl;
        return nullptr;
    }

    return TBase::DoExecute(state, context);
}

NDrive::NFine::TFineFilterGroup TRTMajorFineDecreeCollector::GetFilters(const NDrive::IServer& server, const TFineIdToEventPtrMapping& fineIdToEventPtrMapping) const {
    NDrive::NFine::TFineFilterGroup filters = TBase::GetFilters(server, fineIdToEventPtrMapping);

    if (!!RulingNumbers) {
        filters.Append(MakeAtomicShared<NDrive::NFine::TFineMultipleRulingNumbersFilter>(RulingNumbers));
    }

    if (GetExplicitDecreeOnlyFlag()) {
        filters.Append(MakeAtomicShared<NDrive::NFine::TFineExplicitDecreeFilter>());
    }

    return filters;
}

TAtomicSharedPtr<TRTMajorFineDecreeCollector::IState> TRTMajorFineDecreeCollector::BuildState(const ui64 lastEventId) const {
    auto state = MakeAtomicShared<TState>();
    state->SetLastEventId(lastEventId);
    return state;
}

TRTMajorFineDecreeCollector::IBaseContext::TPtr TRTMajorFineDecreeCollector::ConstructContext(
    const NDrive::IServer& server,
    const NDrive::NFine::TFineAttachmentConstructor& fineAttachmentConstructor,
    const TString& fineId,
    const TString& rulingNumber,
    TEventsHandler::TEventPtr relatedEventPtr
) const {
    return MakeAtomicShared<TContext>(server, fineAttachmentConstructor, fineId, rulingNumber, relatedEventPtr);
}

bool TRTMajorFineDecreeCollector::FilterFine(const IBaseContext::TPtr context, TMessagesCollector& errors) const {
    if (!TBase::FilterFine(context, errors)) {
        return false;
    }

    auto existingFineDecreesPtr = context->GetFineAttachmentConstructor().GetKnownDegrees().FindPtr(context->GetFineId());
    if (!existingFineDecreesPtr || !existingFineDecreesPtr->empty()) {
        return false;
    }

    if (GetWithoutPhotoOnly()) {
        auto existingFinePhotosPtr = context->GetFineAttachmentConstructor().GetKnownPhotos().FindPtr(context->GetFineId());
        if (!existingFinePhotosPtr || !existingFinePhotosPtr->empty()) {
            return false;
        }
    }

    return true;
}

bool TRTMajorFineDecreeCollector::CollectExternalAttachments(IBaseContext::TPtr context, TMessagesCollector& errors) const {
    auto derivedContextPtr = VerifyDynamicCast<TContext*>(context.Get());

    const auto& majorClient = context->GetServer().GetDriveAPI()->GetMajorClient();
    if (!majorClient.GetCarPenaltyDecree(context->GetRulingNumber(), derivedContextPtr->MutableDecree(), errors)) {
        return false;
    }

    return true;
}

bool TRTMajorFineDecreeCollector::ConstructNativeFineAttachments(IBaseContext::TPtr context, TMessagesCollector& errors) const {
    auto derivedContextPtr = VerifyDynamicCast<TContext*>(context.Get());

    NDrive::NFine::TAutocodeFineAttachmentEntry attachmentEntry;

    if (!context->GetFineAttachmentConstructor().ConstructAutocodeFineAttachmentEntry(context->GetFineId(), derivedContextPtr->GetDecree(), attachmentEntry, errors)) {
        return false;
    }

    context->MutableAttachmentEntries().push_back(std::move(attachmentEntry));
    return true;
}

NDrive::TScheme TRTMajorFineDecreeCollector::DoGetScheme(const IServerBase& server) const {
    auto scheme = TBase::DoGetScheme(server);

    scheme.Add<TFSString>("ruling_numbers", "Определенные номера постановлений (через пробел или запятую)");

    scheme.Add<TFSBoolean>("without_photo_only", "Только без фото").SetDefault(true);
    scheme.Add<TFSBoolean>("explicit_decree_only", "Только с явным признаком наличия документа").SetDefault(true);

    return scheme;
}

bool TRTMajorFineDecreeCollector::DoDeserializeFromJson(const NJson::TJsonValue& data) {
    if (!TBase::DoDeserializeFromJson(data)) {
        return false;
    }

    if (!TJsonProcessor::ReadContainer(data, "ruling_numbers", RulingNumbers)) {
        return false;
    }

    if (!TJsonProcessor::Read(data, "without_photo_only", WithoutPhotoOnly)) {
        return false;
    }
    if (!TJsonProcessor::Read(data, "explicit_decree_only", ExplicitDecreeOnlyFlag)) {
        return false;
    }

    return true;
}

NJson::TJsonValue TRTMajorFineDecreeCollector::DoSerializeToJson() const {
    NJson::TJsonValue result = TBase::DoSerializeToJson();

    TJsonProcessor::WriteContainerString(result, "ruling_numbers", RulingNumbers);

    TJsonProcessor::Write(result, "without_photo_only", WithoutPhotoOnly);
    TJsonProcessor::Write(result, "explicit_decree_only", ExplicitDecreeOnlyFlag);

    return result;
}
