#include "major_processor.h"

#include <drive/backend/cars/car.h>
#include <drive/backend/database/transaction/assert.h>
#include <drive/backend/fines/filters.h>
#include <drive/backend/major/client.h>
#include <drive/backend/tags/tags_manager.h>

#include <rtline/util/algorithm/ptr.h>

namespace {
    static const size_t CarVinSize = 17;

    static TSet<TString> GetAllCarVins(const NDrive::IServer& server, const TSet<TString>& excludedVins = {}) {
        TSet<TString> vins;

        auto gCars = server.GetDriveAPI()->GetCarsData()->GetCachedObjectsVector();
        for (auto&& i : gCars) {
            const TString& vin = i.GetVin();
            if (!!vin && vin.size() == CarVinSize && !excludedVins.contains(vin)) {  // filter fake cars
                vins.insert(vin);
            }
        }

        return vins;
    }
}

void TRTMajorFinesCollectorState::SerializeToProto(NDrive::NProto::TRTFinesProcessorState& proto) const {
    TBase::SerializeToProto(proto);
    proto.SetLastTagHistoryEventId(LastTagHistoryEventId);
}

bool TRTMajorFinesCollectorState::DeserializeFromProto(const NDrive::NProto::TRTFinesProcessorState& proto) {
    if (proto.HasLastTagHistoryEventId()) {
        LastTagHistoryEventId = proto.GetLastTagHistoryEventId();
    }
    return TBase::DeserializeFromProto(proto);
}

NJson::TJsonValue TRTMajorFinesCollectorState::GetReport() const {
    NJson::TJsonValue result = TBase::GetReport();
    TJsonProcessor::Write(result, "last_tag_history_event_id", LastTagHistoryEventId);
    return result;
}

NDrive::TScheme TRTMajorFinesCollectorState::DoGetScheme() const {
    NDrive::TScheme result = TBase::DoGetScheme();
    result.Add<TFSNumeric>("last_tag_history_event_id", "Последнее обработанное событие");
    return result;
}

const TString TRTMajorFinesCollector::TypeName = "fines_major_processor";

IRTRegularBackgroundProcess::TFactory::TRegistrator<TRTMajorFinesCollector> TRTMajorFinesCollector::Registrator(TRTMajorFinesCollector::TypeName);

TString TRTMajorFinesCollectorState::GetType() const {
    return TRTMajorFinesCollector::TypeName;
}

TRTMajorFinesCollectorState::TFactory::TRegistrator<TRTMajorFinesCollectorState> TRTMajorFinesCollectorState::Registrator(TRTMajorFinesCollector::TypeName);

TMajorFinesProcessingContext::TMajorFinesProcessingContext(const NDrive::IServer& server
    , const NDrive::NFine::TFineConstructor& fineConstructor
    , const TFineIdToEventPtrMapping& fineIdToEventPtrMapping
    , const TPrefechedFinesContainer& prefechedFines
    , TVector<TExternalFine>&& externalFines
)
    : TBase(server, fineConstructor, prefechedFines)
    , ExternalFines(std::move(externalFines))
    , FineIdToEventPtrMapping(fineIdToEventPtrMapping)
{
}


const TMajorFinesProcessingContext::TFineIdToEventPtrMapping& TMajorFinesProcessingContext::GetFineIdToEventPtrMapping() const {
    return FineIdToEventPtrMapping;
}

TRTMajorFinesCollector::TRTMajorFinesCollector()
    : TBase()
    , TagEventsHandlerPtr(MakeAtomicShared<TEventsHandler>(NEntityTagsManager::EEntityType::Car))
{
    MutableTagEventsHandlerPtr()->AddFilter(MakeAtomicShared<TFineActionTagFilter>(EObjectHistoryAction::Add, EFineActionType::Update));

    InitNotifyHandlers();
}

TExpectedState TRTMajorFinesCollector::DoExecute(TAtomicSharedPtr<IRTBackgroundProcessState> state, const TExecutionContext& context) const {
    const auto& server = context.GetServerAs<NDrive::IServer>();
    CHECK_WITH_LOG(!!server.GetDriveAPI());

    if (!server.GetDriveAPI()->HasFinesManager() || !server.GetDriveAPI()->HasMajorClient() || !GetNotifyHandlers()) {
        return MakeUnexpected<TString>({});
    }

    auto nextState = BuildState(state);

    TSet<TString> totalRunVins;
    TFineIdToEventPtrMapping fineIdToEventPtrMapping;
    std::tie(totalRunVins, fineIdToEventPtrMapping) = GetTotalRunVinsToCheck(server, nextState->GetLastTagHistoryEventId());

    TPrefechedFinesContainer prefechedFines;
    {
        const auto& finesManager = Yensured(server.GetDriveAPI())->GetFinesManager();
        auto tx = finesManager.BuildTx<NSQL::ReadOnly>();
        R_ENSURE(prefechedFines.Init(finesManager, GetFilters(), tx), {}, "Fail to fetch fines for MajorFinesCollector", tx);
    }

    NDrive::NFine::TFineConstructor fineConstructor(GetFineConstructorConfig(), server);

    ui32 totalProcessed = 0;
    ui32 requestLimit;

    while (GetRequestLimit(totalProcessed, requestLimit)) {
        auto vinsChunk = GetRequestVinsToCheck(nextState, totalRunVins, requestLimit);
        if (!vinsChunk) {
            break;
        }

        TVector<TCarPenaltyInfo> finesChunk;
        if (!CollectFines(server, vinsChunk, finesChunk)) {
            return MakeUnexpected<TString>({});
        }

        if (!finesChunk.empty()) {
            IFineProcessingContextBase::TPtr contextPtr = ConstructProcessingContext(server
                , fineConstructor
                , fineIdToEventPtrMapping
                , prefechedFines
                , std::move(finesChunk)
            );
            TMessagesCollector errors;

            if (!ProcessFines(contextPtr, errors)) {
                ERROR_LOG << errors.GetStringReport() << Endl;
                return MakeUnexpected<TString>({});
            }
        }

        for (const auto& vin : vinsChunk) {
            nextState->MutableCarChecks()[vin] = TState::TCarCheckInfo(TState::ECarIdType::VIN, vin);
        }
        totalProcessed += vinsChunk.size();
    }

    if (!!TagEventsHandlerPtr) {
        nextState->SetLastTagHistoryEventId(TagEventsHandlerPtr->GetLastProcessedEventId());
    }

    ReportCurrentState(server, nextState, totalRunVins);
    return nextState;
}

TAtomicSharedPtr<TRTMajorFinesCollector::TState> TRTMajorFinesCollector::BuildState(TAtomicSharedPtr<IRTBackgroundProcessState> state) const {
    auto nativeState = std::dynamic_pointer_cast<TState>(state);
    return (!!nativeState) ? MakeAtomicShared<TState>(*nativeState) : MakeAtomicShared<TState>();
}

TRTMajorFinesCollector::TRequestVinsInfo TRTMajorFinesCollector::GetTotalRunVinsToCheck(const NDrive::IServer& server, const ui64 lastProcessedEventId) const {
    TSet<TString> allVins;
    TFineIdToEventPtrMapping fineIdToEventPtrMapping;

    if (GetHandleActionTagsFlag()) {
        std::tie(allVins, fineIdToEventPtrMapping) = GetVinsFromTagEvents(server, lastProcessedEventId);
    } else {
        if (!GetVinsToCheck()) {
            allVins = GetAllCarVins(server);
        } else {
            if (GetExcludeVinsFlag()) {
                allVins = GetAllCarVins(server, GetVinsToCheck());
            } else {
                allVins = GetVinsToCheck();
            }
        }
    }

    return {allVins, fineIdToEventPtrMapping};
}

TRTMajorFinesCollector::TRequestVinsInfo TRTMajorFinesCollector::GetVinsFromTagEvents(const NDrive::IServer& server, const ui64 lastProcessedEventId) const {
    TSet<TString> requestedVins;
    TFineIdToEventPtrMapping fineIdToEventPtrMapping;

    if (!!TagEventsHandlerPtr && GetHandleActionTagsFlag()) {
        const bool fetchResult = TagEventsHandlerPtr->Fetch(server, lastProcessedEventId, DataActuality);
        if (!fetchResult) {
            return {requestedVins, fineIdToEventPtrMapping};
        }

        auto tx = TagEventsHandlerPtr->GetEntityTagsManager(server).BuildTx<NSQL::ReadOnly>();
        while (auto eventPtr = TagEventsHandlerPtr->Next(server)) {
            auto fineActionTagPtr = eventPtr->GetTagAs<TFineActionTag>();
            if (!!fineActionTagPtr && TagEventsHandlerPtr->DoesTagExist(eventPtr, server, tx)) {
                TString carId = eventPtr->GetObjectId();
                auto gCars = server.GetDriveAPI()->GetCarsData()->FetchInfo(carId, TInstant::Zero());
                auto carInfoPtr = gCars.GetResultPtr(carId);
                if (!carInfoPtr || !carInfoPtr->GetVin()) {
                    ERROR_LOG << "Cannot fetch car vin for tag " << eventPtr->GetTagId() << Endl;
                    continue;
                }

                requestedVins.insert(carInfoPtr->GetVin());
                fineIdToEventPtrMapping.emplace(fineActionTagPtr->GetFineId(), eventPtr);
            }
        }
    }

    return {requestedVins, fineIdToEventPtrMapping};
}

bool TRTMajorFinesCollector::GetRequestLimit(const ui32 totalProcessed, ui32& requestLimit) const {
    if (MaxTotalRunVinsCount > 0) {
        if (totalProcessed >= MaxTotalRunVinsCount) {
            return false;
        }
        auto remaining = MaxTotalRunVinsCount - totalProcessed;
        requestLimit = (MaxRequestVinsCount > 0) ? Min(remaining, MaxRequestVinsCount) : remaining;
    } else {
        requestLimit = MaxRequestVinsCount;
    }
    return true;
}

TVector<TString> TRTMajorFinesCollector::GetRequestVinsToCheck(TAtomicSharedPtr<TState> state, const TSet<TString>& allVins, ui32 requestLimit) const {
    TVector<TString> vinsToCheck;

    for (const auto& vin : allVins) {
        const auto& checkInfo = state->MutableCarChecks()[vin];
        if (checkInfo.Timestamp + PenaltyCheckPeriod > ModelingNow() || checkInfo.Timestamp > StartInstant) {
            continue;  // too early or already checked in current run
        }

        vinsToCheck.push_back(vin);

        if (requestLimit > 0 && vinsToCheck.size() >= requestLimit) {
            break;
        }
    }

    return vinsToCheck;
}

bool TRTMajorFinesCollector::CollectFines(const NDrive::IServer& server, const TVector<TString>& vins, TVector<TCarPenaltyInfo>& fines) const {
    const auto& majorClient = server.GetDriveAPI()->GetMajorClient();

    GetNotifyHandlers()->Handle(ECollectingStatus::CollectingTotal, server, vins.size());

    TMessagesCollector errors;
    if (!majorClient.GetCarPenalties(vins, GetPenaltyCheckPolicy(), fines, errors)) {
        ERROR_LOG << "Error collecting major fines: " << errors.GetStringReport() << Endl;
        GetNotifyHandlers()->Handle(ECollectingStatus::CollectingFail, server);
        return false;
    }

    GetNotifyHandlers()->Handle(ECollectingStatus::CollectingSuccess, server);
    GetNotifyHandlers()->Handle(ECollectingStatus::CollectedTotal, server, fines.size());

    return true;
}

void TRTMajorFinesCollector::ReportCurrentState(const NDrive::IServer& server, TAtomicSharedPtr<TState> state, const TSet<TString>& allVins) const {
    const auto now = ModelingNow();
    TDuration maxCheckDelay = TDuration::Zero();

    for (auto&& [vin, checkInfo] : state->GetCarChecks()) {
        if (allVins.contains(vin)) {
            maxCheckDelay = Max(maxCheckDelay, now - checkInfo.Timestamp);
        }
    }

    if (!!maxCheckDelay) {
        GetNotifyHandlers()->Handle(ECollectingStatus::MaxCheckDelay, server, maxCheckDelay.Seconds());
    }
}

IFineProcessingContextBase::TPtr TRTMajorFinesCollector::ConstructProcessingContext(const NDrive::IServer& server
    , const NDrive::NFine::TFineConstructor& fineConstructor
    , const TFineIdToEventPtrMapping& fineIdToEventPtrMapping
    , const TPrefechedFinesContainer& prefechedFines
    , TVector<TCarPenaltyInfo>&& fines
) const {
    return MakeAtomicShared<TContext>(server, fineConstructor, fineIdToEventPtrMapping, prefechedFines, std::move(fines));
}

bool TRTMajorFinesCollector::ConstructFines(IFineProcessingContextBase::TPtr contextPtr, TVector<TAutocodeFineEntry>& nativeFines, TMessagesCollector& /* errors */) const {
    auto handlerContextPtr = std::dynamic_pointer_cast<TContext>(contextPtr);
    if (!handlerContextPtr) {
        return false;
    }

    for (auto&& fine : handlerContextPtr->GetExternalFines()) {
        TMessagesCollector fineProcessingErrors;

        TAutocodeFineEntry nativeFine;
        if (!ProcessFine(handlerContextPtr, fine, nativeFine, fineProcessingErrors)) {
            if (fineProcessingErrors.HasMessages()) {
                ERROR_LOG << fineProcessingErrors.GetStringReport() << Endl;
            }
            continue;
        }

        nativeFines.push_back(std::move(nativeFine));
    }

    return true;
}

bool TRTMajorFinesCollector::ProcessFine(const TContextPtr contextPtr, const TCarPenaltyInfo& fine, TAutocodeFineEntry& nativeFine, TMessagesCollector& errors) const {
    if (!!MinRulingDate && fine.GetRulingDate() < MinRulingDate) {
        return false;
    }

    TString rulingNumber = fine.GetRulingNumber();
    if (!rulingNumber) {
        return false;
    }

    auto articleMatcherPtr = contextPtr->GetFineArticleMatcherPtr();
    if (!articleMatcherPtr) {
        return false;
    }
    auto unknownArticleFilter = NDrive::NFine::TFineUnknownArticleFilter(articleMatcherPtr);

    auto unknownIncludeToBillInfoFilter = NDrive::NFine::TFineUnknownIncludeToBillInfoFilter();

    auto notChargedFineFilter = NDrive::NFine::TFineNotChargedFilter();

    bool areAllFineFieldsUpdating = false;
    bool areMetaInfoOnlyFineFieldsUpdating = false;

    bool otherSourceFinesExist = false;
    auto equalRange = contextPtr->GetFineMapByRulingNumber().equal_range(rulingNumber);
    if (equalRange.first == equalRange.second && GetUpdateAllowedFlag() && fine.GetId()) {
        if (auto dbFine = contextPtr->GetFineMapBySystemId().FindPtr(fine.GetId()); dbFine && dbFine->get().GetSourceType() == ToString(NDrive::NFine::ESourceType::Major)) {
            nativeFine = *dbFine;
        }
    }

    for (auto it = equalRange.first; it != equalRange.second; ++it) {
        const TAutocodeFineEntry& fine = it->second;

        if (fine.GetSourceType() == ToString(NDrive::NFine::ESourceType::Major)) {
            if (GetUpdateUnknownArticleOnly() && !unknownArticleFilter.Match(fine)) {
                return false;
            }

            if (GetUpdateUnknownIncludeToBillInfoFlag() && !unknownIncludeToBillInfoFilter.Match(fine)) {
                return false;
            }

            areAllFineFieldsUpdating = (  // update all fine fields, only not charged fines
                GetUpdateAllowedFlag() ||
                (GetHandleActionTagsFlag() && contextPtr->GetFineIdToEventPtrMapping().contains(fine.GetId()))
            ) && notChargedFineFilter.Match(fine);

            areMetaInfoOnlyFineFieldsUpdating = GetMetaInfoUpdateAllowedFlag();  // update meta info only, any fine

            if (areAllFineFieldsUpdating || areMetaInfoOnlyFineFieldsUpdating) {
                nativeFine = std::move(fine);  // update, set fine id in particular
            } else {
                return false;  // fine already exists and update is not permitted
            }
        } else {
            otherSourceFinesExist = true;
        }
    }

    if (IsCheckUniqueAllSourceTypes() && otherSourceFinesExist) {
        return false;
    }

    bool processingResult = false;
    if (areAllFineFieldsUpdating) {
        processingResult = contextPtr->GetFineConstructor().ConstructAutocodeFineEntry(fine, nativeFine);  // overwrite all fields
    } else if (areMetaInfoOnlyFineFieldsUpdating) {
        processingResult = contextPtr->GetFineConstructor().UpdateMetaInfoFields(fine, nativeFine);  // overwrite meta fields only
    } else {
        processingResult = contextPtr->GetFineConstructor().ConstructAutocodeFineEntry(fine, nativeFine);  // a new fine is constructing
    }

    if (!processingResult) {
        errors.AddMessage(__LOCATION__, "Error converting fine to db entry: " + fine.GetRulingNumber());
        GetNotifyHandlers()->Handle(ECollectingStatus::CollectedConversionError, contextPtr->GetServer());
        return false;
    }

    return true;
}

bool TRTMajorFinesCollector::FilterNativeFine(const IFineProcessingContextBase::TPtr contextPtr, const NDrive::NFine::TFineFetchContext& fineContext, TMessagesCollector& errors) const {
    if (!TBase::FilterNativeFine(contextPtr, fineContext, errors)) {
        return false;
    }
    if (GetSkipUnknownArticleFinesFlag() && !fineContext.GetEntry().GetArticleCode()) {
        return false;
    }
    return true;
}

bool TRTMajorFinesCollector::OnBeforeCommit(const IFineProcessingContextBase::TPtr contextPtr, const TVector<TAutocodeFineEntry>& fines, NDrive::TEntitySession& session, TMessagesCollector& /* errors */) const {
    auto handlerContextPtr = std::dynamic_pointer_cast<TContext>(contextPtr);
    return handlerContextPtr && HandleFineRelatedTags(handlerContextPtr, fines, session);
}

bool TRTMajorFinesCollector::HandleFineRelatedTags(const TContextPtr contextPtr, const TVector<TAutocodeFineEntry>& fines, NDrive::TEntitySession& session) const {
    if (!!TagEventsHandlerPtr && GetHandleActionTagsFlag()) {
        for (const auto& fine : fines) {
            auto eventPtr = contextPtr->GetFineIdToEventPtrMapping().Value(fine.GetId(), nullptr);
            if (!!eventPtr && !TagEventsHandlerPtr->RemoveTagIfExists(eventPtr, GetRobotUserId(), contextPtr->GetServer(), session)) {
                return false;
            }
        }
    }
    return true;
}

NDrive::TScheme TRTMajorFinesCollector::DoGetScheme(const IServerBase& server) const {
    auto scheme = TBase::DoGetScheme(server);

    scheme.Add<TFSString>("min_ruling_date", "Дата, с которой забираются штрафы; все если не задано");

    scheme.Add<TFSArray>("vins_to_check", "VIN автомобилей для проверки; все если не задано").SetElement<TFSString>();
    scheme.Add<TFSBoolean>("exclude_vins", "Для всех VIN, кроме указанных").SetDefault(false);

    scheme.Add<TFSNumeric>("max_request_vins_count", "Ограничение на кол-во единовременно запрашиваемых автомобилей; без ограничений если ноль")
          .SetMin(0)
          .SetDefault(100);
    scheme.Add<TFSNumeric>("max_total_run_vins_count", "Ограничение на кол-во запрашиваемых автомобилей за запуск; без ограничений если ноль")
          .SetMin(0)
          .SetDefault(0);

    scheme.Add<TFSDuration>("penalty_check_period", "Интервал проверок штрафов для каждого авто").SetDefault(TDuration::Days(1));
    scheme.Add<TFSVariants>("penalty_check_policy", "Способ сбора штрафов, по умолчанию все")
          .SetVariants(GetEnumAllValues<EPenaltyCheckPolicy>())
          .SetRequired(true);

    scheme.Add<TFSBoolean>("skip_unknown_article_fines", "Не сохранять штрафы с неизвестной статьей нарушения").SetDefault(false);

    scheme.Add<TFSBoolean>("is_update_allowed", "Разрешено ли дообновление информации о штрафах").SetDefault(false);
    scheme.Add<TFSBoolean>("is_meta_info_update_allowed", "Разрешено ли дообновление мета-информации (вспомогательной) о штрафах").SetDefault(false);
    scheme.Add<TFSBoolean>("unknown_article_only", "Дообновлять только с неизвестной статьей нарушения").SetDefault(false);
    scheme.Add<TFSBoolean>("unknown_include_to_bill_info_only", "Дообновлять только с неизвестным статусом включения в счет").SetDefault(false);

    scheme.Add<TFSBoolean>("handle_action_tags", "Обрабатывать теги действий").SetDefault(false);

    return scheme;
}

bool TRTMajorFinesCollector::DoDeserializeFromJson(const NJson::TJsonValue& data) {
    if (!TBase::DoDeserializeFromJson(data)) {
        return false;
    }

    if (data["min_ruling_date"].IsDefined() && !TJsonProcessor::Read(data, "min_ruling_date", MinRulingDate)) {
        return false;
    }

    if (!TJsonProcessor::ReadContainer(data, "vins_to_check", VinsToCheck)) {
        return false;
    }
    if (!TJsonProcessor::Read(data, "exclude_vins", ExcludeVinsFlag)) {
        return false;
    }

    if (!TJsonProcessor::Read(data, "max_request_vins_count", MaxRequestVinsCount)) {
        return false;
    }
    if (!TJsonProcessor::Read(data, "max_total_run_vins_count", MaxTotalRunVinsCount)) {
        return false;
    }

    if (!TJsonProcessor::Read(data, "penalty_check_period", PenaltyCheckPeriod)) {
        return false;
    }
    if (!TJsonProcessor::ReadFromString("penalty_check_policy", data, PenaltyCheckPolicy)) {
        return false;
    }

    if (!TJsonProcessor::Read(data, "skip_unknown_article_fines", SkipUnknownArticleFinesFlag)) {
        return false;
    }

    if (!TJsonProcessor::Read(data, "is_update_allowed", UpdateAllowedFlag)) {
        return false;
    }
    if (!TJsonProcessor::Read(data, "is_meta_info_update_allowed", MetaInfoUpdateAllowedFlag)) {
        return false;
    }
    if (!TJsonProcessor::Read(data, "unknown_article_only", UpdateUnknownArticleOnly)) {
        return false;
    }
    if (!TJsonProcessor::Read(data, "unknown_include_to_bill_info_only", UpdateUnknownIncludeToBillInfoFlag)) {
        return false;
    }

    if (!TJsonProcessor::Read(data, "handle_action_tags", HandleActionTagsFlag)) {
        return false;
    }

    return true;
}

NJson::TJsonValue TRTMajorFinesCollector::DoSerializeToJson() const {
    NJson::TJsonValue result = TBase::DoSerializeToJson();

    TJsonProcessor::Write(result, "min_ruling_date", (!!MinRulingDate) ? NJson::TJsonValue(MinRulingDate.ToString()) : NJson::JSON_NULL);

    TJsonProcessor::WriteContainerArray(result, "vins_to_check", VinsToCheck);
    TJsonProcessor::Write(result, "exclude_vins", ExcludeVinsFlag);

    TJsonProcessor::Write(result, "max_request_vins_count", MaxRequestVinsCount);
    TJsonProcessor::Write(result, "max_total_run_vins_count", MaxTotalRunVinsCount);

    TJsonProcessor::WriteAsString(result, "penalty_check_period", PenaltyCheckPeriod);
    TJsonProcessor::WriteAsString(result, "penalty_check_policy", PenaltyCheckPolicy);

    TJsonProcessor::Write(result, "skip_unknown_article_fines", SkipUnknownArticleFinesFlag);

    TJsonProcessor::Write(result, "is_update_allowed", UpdateAllowedFlag);
    TJsonProcessor::Write(result, "is_meta_info_update_allowed", MetaInfoUpdateAllowedFlag);
    TJsonProcessor::Write(result, "unknown_article_only", UpdateUnknownArticleOnly);
    TJsonProcessor::Write(result, "unknown_include_to_bill_info_only", UpdateUnknownIncludeToBillInfoFlag);

    TJsonProcessor::Write(result, "handle_action_tags", HandleActionTagsFlag);

    return result;
}
