#pragma once

#include "events.h"
#include "processor_base.h"

class TRTMajorFinesCollectorState : public TRTFinesCollectorBaseState {
    using TBase = TRTFinesCollectorBaseState;

public:
    virtual TString GetType() const override;

    virtual NJson::TJsonValue GetReport() const override;
    virtual NDrive::TScheme DoGetScheme() const override;

protected:
    virtual void SerializeToProto(NDrive::NProto::TRTFinesProcessorState& proto) const override;
    virtual bool DeserializeFromProto(const NDrive::NProto::TRTFinesProcessorState& proto) override;

private:
    static TFactory::TRegistrator<TRTMajorFinesCollectorState> Registrator;

    R_FIELD(ui64, LastTagHistoryEventId, Max<ui64>());
};

class TMajorFinesProcessingContext : public IFineProcessingContextBase {
    using TBase = IFineProcessingContextBase;

public:
    using TExternalFine = NMajorClient::TCarPenaltiesRequest::TCarPenaltyInfo;
    using TFineIdToEventPtrMapping = TMap<TString, TEventsHandler::TEventPtr>;
    using TRulingNumberMap = TMultiMap<TString, std::reference_wrapper<const NDrive::NFine::TAutocodeFineEntry>>;
    using TFineMap = TMap<i64, std::reference_wrapper<const NDrive::NFine::TAutocodeFineEntry>>;

public:
    TMajorFinesProcessingContext(const NDrive::IServer& server
        , const NDrive::NFine::TFineConstructor& fineConstructor
        , const TFineIdToEventPtrMapping& fineIdToEventPtrMapping
        , const TPrefechedFinesContainer& prefechedFines
        , TVector<TExternalFine>&& externalFines
    );

    const TFineIdToEventPtrMapping& GetFineIdToEventPtrMapping() const;

private:
    R_FIELD(TVector<TExternalFine>, ExternalFines);

    const TFineIdToEventPtrMapping& FineIdToEventPtrMapping;
};

class TRTMajorFinesCollector: public TRTFinesCollectorBase {
    using TBase = TRTFinesCollectorBase;

    using TState = TRTMajorFinesCollectorState;
    using TContext = TMajorFinesProcessingContext;
    using TContextPtr = TAtomicSharedPtr<TContext>;

    using TCarPenaltyInfo = NMajorClient::TCarPenaltiesRequest::TCarPenaltyInfo;
    using ERequisiteFieldName = TCarPenaltyInfo::TRequisites::ERequisiteFieldName;
    using EPenaltyCheckPolicy = NMajorClient::TCarPenaltiesRequest::EPenaltyCheckPolicy;

    using TFineIdToEventPtrMapping = typename TContext::TFineIdToEventPtrMapping;
    using TRequestVinsInfo = std::tuple<TSet<TString>, TFineIdToEventPtrMapping>;

public:
    TRTMajorFinesCollector();

    virtual TExpectedState DoExecute(TAtomicSharedPtr<IRTBackgroundProcessState> state, const TExecutionContext& context) const override;

    virtual TString GetType() const override {
        return TypeName;
    }

    virtual NDrive::NFine::ESourceType GetSourceType() const override {
        return NDrive::NFine::ESourceType::Major;
    }

    virtual NDrive::TScheme DoGetScheme(const IServerBase& server) const override;
    virtual bool DoDeserializeFromJson(const NJson::TJsonValue& data) override;
    virtual NJson::TJsonValue DoSerializeToJson() const override;

    static const TString TypeName;

private:
    TAtomicSharedPtr<TState> BuildState(TAtomicSharedPtr<IRTBackgroundProcessState> state) const;

    TRequestVinsInfo GetTotalRunVinsToCheck(const NDrive::IServer& server, const ui64 lastProcessedEventId) const;
    TRequestVinsInfo GetVinsFromTagEvents(const NDrive::IServer& server, const ui64 lastProcessedEventId) const;

    bool GetRequestLimit(const ui32 totalProcessed, ui32& requestLimit) const;
    TVector<TString> GetRequestVinsToCheck(TAtomicSharedPtr<TState> state, const TSet<TString>& allVins, ui32 maxRequestSize) const;

    bool CollectFines(const NDrive::IServer& server, const TVector<TString>& vins, TVector<TCarPenaltyInfo>& fines) const;
    void ReportCurrentState(const NDrive::IServer& server, TAtomicSharedPtr<TState> state, const TSet<TString>& allVins) const;

    IFineProcessingContextBase::TPtr ConstructProcessingContext(const NDrive::IServer& server
        , const NDrive::NFine::TFineConstructor& fineConstructor
        , const TFineIdToEventPtrMapping& fineIdToEventPtrMapping
        , const TPrefechedFinesContainer& prefechedFines
        , TVector<TCarPenaltyInfo>&& fines
    ) const;

    virtual bool ConstructFines(IFineProcessingContextBase::TPtr contextPtr, TVector<TAutocodeFineEntry>& nativeFines, TMessagesCollector& errors) const override;
    bool ProcessFine(const TContextPtr contextPtr, const TCarPenaltyInfo& fine, TAutocodeFineEntry& nativeFine, TMessagesCollector& errors) const;

    virtual bool FilterNativeFine(const IFineProcessingContextBase::TPtr contextPtr, const NDrive::NFine::TFineFetchContext& fineContext, TMessagesCollector& errors) const override;

    bool OnBeforeCommit(const IFineProcessingContextBase::TPtr contextPtr, const TVector<TAutocodeFineEntry>& fines, NDrive::TEntitySession& session, TMessagesCollector& errors) const override;
    bool HandleFineRelatedTags(const TContextPtr contextPtr, const TVector<TAutocodeFineEntry>& fines, NDrive::TEntitySession& session) const;

private:
    static TFactory::TRegistrator<TRTMajorFinesCollector> Registrator;

    R_FIELD(TInstant, MinRulingDate, TInstant::Zero());

    R_FIELD(TSet<TString>, VinsToCheck);
    R_FIELD(bool, ExcludeVinsFlag, false);

    R_FIELD(ui32, MaxRequestVinsCount, 0);
    R_FIELD(ui32, MaxTotalRunVinsCount, 0);

    R_FIELD(TDuration, PenaltyCheckPeriod, TDuration::Days(1));
    R_FIELD(EPenaltyCheckPolicy, PenaltyCheckPolicy, EPenaltyCheckPolicy::All);

    R_FIELD(bool, SkipUnknownArticleFinesFlag, false);

    R_FIELD(bool, UpdateAllowedFlag, false);
    R_FIELD(bool, MetaInfoUpdateAllowedFlag, false);

    R_FIELD(bool, UpdateUnknownArticleOnly, false);
    R_FIELD(bool, UpdateUnknownIncludeToBillInfoFlag, false);

    R_FIELD(bool, HandleActionTagsFlag, false);

    R_FIELD(TEventsHandler::TPtr, TagEventsHandlerPtr, {}, mutable);
};
