#include "processor_base.h"

#include <drive/backend/database/transaction/assert.h>

#include <rtline/library/json/cast.h>

#include <util/string/builder.h>

namespace {
    template <typename T>
    static bool IsMultipleOf(const T value, const ui32 divisor) {
        return divisor * static_cast<T>(value / divisor) == value;
    }

    static TString GetFineLogInfo(const NDrive::NFine::TAutocodeFineEntry& fine) {
        return (TStringBuilder()
                << "ruling number - " << fine.GetRulingNumber() << "; "
                << "article - " << fine.GetArticleKoap() << "; "
                << "car document (" << fine.GetViolationDocumentType() << "): " << fine.GetViolationDocumentNumber());
    }

    static TString GetFineLogFinancialInfo(const NDrive::NFine::TAutocodeFineEntry& fine) {
        return (TStringBuilder()
                << "total amount cents - " << fine.GetSumToPayWithoutDiscountCents() << "; "
                << "discounted amount cents - " << fine.GetSumToPayCents() << "; "
                << GetFineLogInfo(fine));
    }
}

void TRTFinesCollectorBaseState::SerializeToProto(NDrive::NProto::TRTFinesProcessorState& proto) const {
    for (auto&& [_, carCheck] : CarChecks) {
        auto carCheckInfo = proto.AddCarChecks();
        carCheckInfo->SetIdType(static_cast<NDrive::NProto::TRTFinesProcessorState::TCarCheckInfo::ECarIdType>(carCheck.IdType));
        carCheckInfo->SetIdValue(carCheck.IdValue);
        carCheckInfo->SetTimestamp(carCheck.Timestamp.Seconds());
    }
}

bool TRTFinesCollectorBaseState::DeserializeFromProto(const NDrive::NProto::TRTFinesProcessorState& proto) {
    for (auto&& i : proto.GetCarChecks()) {
        CarChecks.emplace(i.GetIdValue(), TCarCheckInfo(static_cast<ECarIdType>(i.GetIdType()), i.GetIdValue(), i.GetTimestamp()));
    }
    return true;
}

bool TPrefechedFinesContainer::Init(const NDrive::NFine::TFinesManager& finesManager, const NDrive::NFine::TFineFilterGroup& filterGroup, NDrive::TEntitySession& tx) {
    auto fines = finesManager.GetFines(filterGroup, tx);
    if (!fines) {
        return false;
    }
    for (auto& fine : *fines) {
        TString number = fine.GetRulingNumber();
        auto it = Fines.emplace(std::move(number), std::move(fine));
        MapBySystemId.emplace(it->second.GetAutocodeId(), it->second);
    }
    return true;
}

const TPrefechedFinesContainer::TMapByRulingNumber& TPrefechedFinesContainer::GetMapByRulingNumber() const {
    return Fines;
}

const TPrefechedFinesContainer::TMapBySystemId& TPrefechedFinesContainer::GetMapBySystemId() const {
    return MapBySystemId;
}

IFineProcessingContextBase::IFineProcessingContextBase(const NDrive::IServer& server, const NDrive::NFine::TFineConstructor& fineConstructor, const TPrefechedFinesContainer& fines)
    : Server(server)
    , FineConstructor(fineConstructor)
    , Fines(fines)
{
    FineArticleMatcherPtr = GetFinesManager().GetFineArticleMatcherPtr(/* ensureActual = */ true);
}

const NDrive::IServer& IFineProcessingContextBase::GetServer() const {
    return Server;
}

const NDrive::NFine::TFineConstructor& IFineProcessingContextBase::GetFineConstructor() const {
    return FineConstructor;
}

const NDrive::NFine::TFinesManager& IFineProcessingContextBase::GetFinesManager() const {
    return Server.GetDriveAPI()->GetFinesManager();
}

TAtomicSharedPtr<NDrive::NFine::TFineArticleMatcher> IFineProcessingContextBase::GetFineArticleMatcherPtr() const {
    return FineArticleMatcherPtr;
}

const TPrefechedFinesContainer::TMapByRulingNumber& IFineProcessingContextBase::GetFineMapByRulingNumber() const {
    return Fines.GetMapByRulingNumber();
}

const TPrefechedFinesContainer::TMapBySystemId& IFineProcessingContextBase::GetFineMapBySystemId() const {
    return Fines.GetMapBySystemId();
}

bool TRTFinesCollectorBase::DoStart(const TRTBackgroundProcessContainer& container) {
    if (!TBase::DoStart(container)) {
        return false;
    }
    if (!NotifyHandlers) {
        return false;
    }
    NotifyHandlers->SetSignalName(GetRTProcessName());
    return true;
}

void TRTFinesCollectorBase::InitNotifyHandlers() {
    if (!NotifyHandlers) {
        NotifyHandlers = MakeAtomicShared<TNotifyHandlers>(GetType());
    }
}

bool TRTFinesCollectorBase::ProcessFines(IFineProcessingContextBase::TPtr contextPtr, TMessagesCollector& errors) const {
    TVector<TAutocodeFineEntry> nativeFines;
    if (!ConstructFines(contextPtr, nativeFines, errors)) {
        return false;
    }

    TVector<TAutocodeFineEntry> filteredNativeFines = FilterFines(contextPtr, std::move(nativeFines));

    if (!UpsertFines(contextPtr, std::move(filteredNativeFines), errors)) {
        return false;
    }

    return true;
}

NDrive::NFine::TFineFilterGroup TRTFinesCollectorBase::GetFilters() const {
    NDrive::NFine::TFineFilterGroup filters;

    if (!IsCheckUniqueAllSourceTypes()) {
        filters.Append(MakeAtomicShared<NDrive::NFine::TFineSourceTypeFilter>(GetSourceType()));
    }
    filters.Append(MakeAtomicShared<NDrive::NFine::TFineReceiveTimeFilter>(ModelingNow() - FinesHistoryDeep));

    return filters;
}

TVector<NDrive::NFine::TAutocodeFineEntry> TRTFinesCollectorBase::FilterFines(IFineProcessingContextBase::TPtr contextPtr, TVector<TAutocodeFineEntry>&& nativeFines) const {
    TVector<TAutocodeFineEntry> filteredNativeFines;

    for (auto&& nativeFine : nativeFines) {
        TMessagesCollector errors;

        NDrive::NFine::TFineFetchContext fineFetchContext(GetFetchContextConfig(), contextPtr->GetServer(), nativeFine);

        if (!FilterNativeFine(contextPtr, fineFetchContext, errors) || !CheckNativeFine(contextPtr, fineFetchContext, errors)) {
            if (errors.HasMessages()) {
                ERROR_LOG << errors.GetStringReport() << Endl;
            }
            continue;
        }

        filteredNativeFines.push_back(std::move(nativeFine));

        NotifyHandlers->Handle(ECollectingStatus::CollectedFiltered, contextPtr->GetServer());
    }

    return filteredNativeFines;
}

bool TRTFinesCollectorBase::FilterNativeFine(const IFineProcessingContextBase::TPtr /* contextPtr */, const NDrive::NFine::TFineFetchContext& fineContext, TMessagesCollector& /* errors */) const {
    TString articleCode = fineContext.GetEntry().GetArticleCode();
    if (!!ArticlesFilter && (!articleCode || !ArticlesFilter.contains(articleCode))) {
        return false;
    }
    return true;
}

bool TRTFinesCollectorBase::CheckNativeFine(const IFineProcessingContextBase::TPtr /* contextPtr */, const NDrive::NFine::TFineFetchContext& fineContext, TMessagesCollector& errors) const {
    using EChargeStatusReason = NDrive::NFine::NFineChargeStatusTraits::EChargeStatusReason;

    const auto& fine = fineContext.GetEntry();

    if (fine.GetSkipped() > 0) {
        errors.AddMessage(__LOCATION__,
                          TStringBuilder()
                          << "Some sessions have been skipped during fine matching: "
                          << "total skipped - " << fine.GetSkipped() << "; "
                          << GetFineLogInfo(fine));
        NotifyHandlers->Handle(ECollectingStatus::CollectedMatchingSkipCount, fineContext, fine.GetSkipped());
    }

    if (fine.HasBinding()) {
        NotifyHandlers->Handle(ECollectingStatus::CollectedMatched, fineContext);
    } else {
        NotifyHandlers->Handle(ECollectingStatus::CollectedNotMatched, fineContext);
    }

    const TMap<EChargeStatusReason, ECollectingStatus> chargeReasonReportMapping = {
        {EChargeStatusReason::InvalidAmount, ECollectingStatus::CollectedInvalidAmount},
        {EChargeStatusReason::InvalidDiscount, ECollectingStatus::CollectedInvalidDiscount},
        {EChargeStatusReason::NotRecognizedArticle, ECollectingStatus::CollectedUnrecognizedArticle},
        {EChargeStatusReason::UnknownArticle, ECollectingStatus::CollectedBadArticle},
        {EChargeStatusReason::ManuallyPaidDuplicate, ECollectingStatus::CollectedManuallyPaidDuplicate},
        {EChargeStatusReason::InappropriateCityParking, ECollectingStatus::CollectedInappropriateCityParking},
        {EChargeStatusReason::InappropriateLongTermParking, ECollectingStatus::CollectedInappropriateLongTermParking},
    };

    TSet<TString> chargeStatusTraits;
    if (!NJson::TryFromJson(fine.GetMetaInfoProperty(TAutocodeFineEntry::EMetaInfoProperty::ChargeStatusTraits, NJson::JSON_ARRAY), chargeStatusTraits)) {
        errors.AddMessage(__LOCATION__, TStringBuilder() << "Inconsistent fine charge status traits meta info: " << GetFineLogInfo(fine));
        NotifyHandlers->Handle(ECollectingStatus::CollectedInconsistentMetaInfo, fineContext);
        return false;
    }

    for (auto&& [chargeReason, collectingStatus]: chargeReasonReportMapping) {
        if (chargeStatusTraits.contains(::ToString(chargeReason))) {
            TString fineInfo = (chargeReason == EChargeStatusReason::InvalidAmount || chargeReason == EChargeStatusReason::InvalidDiscount)
                                ? GetFineLogFinancialInfo(fine)
                                : GetFineLogInfo(fine);
            errors.AddMessage(__LOCATION__, TStringBuilder() << "Fine has a specific charge reason - " << ::ToString(chargeReason) << ": " << fineInfo);
            NotifyHandlers->Handle(collectingStatus, fineContext);
        }
    }

    return true;
}

bool TRTFinesCollectorBase::UpsertFines(const IFineProcessingContextBase::TPtr contextPtr, TVector<TAutocodeFineEntry>&& fines, TMessagesCollector& errors) const {
    const auto& finesManager = contextPtr->GetFinesManager();

    auto session = finesManager.BuildTx<NSQL::Writable>();

    if (!OnBeforeCommit(contextPtr, fines, session, errors)) {
        return false;
    }

    for (auto&& fine : fines) {
        if (!finesManager.UpsertFine(fine, session)) {
            NDrive::NFine::TFineFetchContext fineFetchContext(GetFetchContextConfig(), contextPtr->GetServer(), fine);
            errors.AddMessage(__LOCATION__, TStringBuilder() << "Error upserting fine: " << GetFineLogInfo(fine) << ": " << session.GetStringReport());
            NotifyHandlers->Handle(ECollectingStatus::UpsertError, fineFetchContext);
            return false;
        }
    }

    if (!session.Commit()) {
        errors.AddMessage(__LOCATION__, TStringBuilder() << "Error committing fines: " << session.GetStringReport());
        NotifyHandlers->Handle(ECollectingStatus::CommitError, contextPtr->GetServer());
        return false;
    }

    if (!OnAfterCommit(contextPtr, errors))  {
        return false;
    }

    return true;
}

bool TRTFinesCollectorBase::OnBeforeCommit(const IFineProcessingContextBase::TPtr /* contextPtr */, const TVector<TAutocodeFineEntry>& /* fines */, NDrive::TEntitySession& /* session */, TMessagesCollector& /* errors */) const {
    return true;
}

bool TRTFinesCollectorBase::OnAfterCommit(const IFineProcessingContextBase::TPtr /* contextPtr */, TMessagesCollector& /* errors */) const {
    return true;
}

NDrive::TScheme TRTFinesCollectorBase::DoGetScheme(const IServerBase& server) const {
    auto scheme = TBase::DoGetScheme(server);

    scheme.Add<TFSArray>("articles_filter", "Статьи штрафов для сбора (напр. 8_25, 12_09_3); все если не задано").SetElement<TFSString>();

    scheme.Add<TFSBoolean>("check_unique_all_source_types", "Проверять ли на уникальность среди штрафов изо всех источников").SetDefault(false);

    // scheme.Add<TFSStructure>("fine_constructor_config", "Настройки конструктора").SetStructure(NDrive::NFine::TFineConstructorConfig::GetScheme(server));  // use defaults only
    scheme.Add<TFSStructure>("fetch_context_config", "Параметры подстановщика шаблонов").SetStructure(NDrive::NFine::TFineFetchContextConfig::GetScheme());

    scheme.Add<TFSStructure>("notify_handlers", "Настройки нотификаторов").SetStructure(TNotifyHandlers::GetScheme(server));

    scheme.Add<TFSDuration>("fine_fetch_deep", "Период выборки штрафов из базы").SetDefault(FinesHistoryDeep);

    return scheme;
}

bool TRTFinesCollectorBase::DoDeserializeFromJson(const NJson::TJsonValue& data) {
    if (!TBase::DoDeserializeFromJson(data)) {
        return false;
    }

    if (!TJsonProcessor::ReadContainer(data, "articles_filter", ArticlesFilter)) {
        return false;
    }

    if (!TJsonProcessor::Read(data, "check_unique_all_source_types", CheckUniqueAllSourceTypes)) {
        return false;
    }

    if (!TJsonProcessor::Read(data, "fine_fetch_deep", FinesHistoryDeep)) {
        return false;
    }

    if (!FineConstructorConfig.DeserializeFromJson({})) {  // use defaults only
        return false;
    }
    if (!FetchContextConfig.DeserializeFromJson(data["fetch_context_config"])) {
        return false;
    }

    if (!!NotifyHandlers && !NotifyHandlers->DeserializeFromJson(data["notify_handlers"])) {
        return false;
    }

    return true;
}

NJson::TJsonValue TRTFinesCollectorBase::DoSerializeToJson() const {
    NJson::TJsonValue result = TBase::DoSerializeToJson();

    TJsonProcessor::WriteContainerArray(result, "articles_filter", ArticlesFilter);

    TJsonProcessor::Write(result, "check_unique_all_source_types", CheckUniqueAllSourceTypes);

    // TJsonProcessor::WriteObject(result, "fine_constructor_config", FineConstructorConfig);  // use defaults only
    TJsonProcessor::WriteSerializable(result, "fetch_context_config", FetchContextConfig);

    if (!!NotifyHandlers) {
        TJsonProcessor::WriteSerializable(result, "notify_handlers", *NotifyHandlers);
    }

    TJsonProcessor::WriteAsString(result, "fine_fetch_deep", FinesHistoryDeep);

    return result;
}
