#pragma once

#include "statuses.h"

#include <drive/backend/rt_background/manager/settings.h>
#include <drive/backend/rt_background/manager/state.h>

#include <drive/backend/database/drive_api.h>
#include <drive/backend/fines/config.h>
#include <drive/backend/fines/constants.h>
#include <drive/backend/fines/manager.h>
#include <drive/backend/fines/constructors/fine_constructor.h>
#include <drive/backend/fines/context_fetchers/fine_context.h>
#include <drive/backend/major/entities.h>
#include <drive/backend/notifications/collection.h>
#include <drive/backend/proto/background.pb.h>

#include <drive/library/cpp/autocode/entities.h>

#include <rtline/util/types/accessor.h>

class TRTFinesCollectorBaseState : public IProtoStateSerializable<NDrive::NProto::TRTFinesProcessorState> {
    using TBase = IProtoStateSerializable<NDrive::NProto::TRTFinesProcessorState>;

public:
    enum class ECarIdType : ui32 {
        INTERNAL_ID = 0,
        VIN = 1,
        STS = 2
    };

    struct TCarCheckInfo {
        ECarIdType IdType;
        TString IdValue;
        TInstant Timestamp;

        TCarCheckInfo() = default;

        TCarCheckInfo(const ECarIdType idType, const TString& idValue, const TInstant& timestamp = TInstant::Now())
            : IdType(idType)
            , IdValue(idValue)
            , Timestamp(timestamp)
        {
        }

        TCarCheckInfo(const ECarIdType idType, const TString& idValue, const ui64 timestamp)
            : TCarCheckInfo(idType, idValue, TInstant::Seconds(timestamp))
        {
        }
    };

    using TCarChecksMapping = TMap<TString, TCarCheckInfo>;

    using TBase::TBase;

    TString GetType() const override = 0;

protected:
    virtual void SerializeToProto(NDrive::NProto::TRTFinesProcessorState& proto) const override;
    virtual bool DeserializeFromProto(const NDrive::NProto::TRTFinesProcessorState& proto) override;

private:
    R_FIELD(TCarChecksMapping, CarChecks);
};

class TPrefechedFinesContainer {
public:
    using TMapByRulingNumber = TMultiMap<TString, NDrive::NFine::TAutocodeFineEntry>;
    using TMapBySystemId = TMap<i64, std::reference_wrapper<const NDrive::NFine::TAutocodeFineEntry>>;

private:
    TMapByRulingNumber Fines;
    TMapBySystemId MapBySystemId;

public:
    bool Init(const NDrive::NFine::TFinesManager& finesManager, const NDrive::NFine::TFineFilterGroup& filterGroup, NDrive::TEntitySession& tx);
    const TMapByRulingNumber& GetMapByRulingNumber() const;
    const TMapBySystemId& GetMapBySystemId() const;
};

class IFineProcessingContextBase {
public:
    using TPtr = TAtomicSharedPtr<IFineProcessingContextBase>;

    IFineProcessingContextBase(const NDrive::IServer& server, const NDrive::NFine::TFineConstructor& fineConstructor, const TPrefechedFinesContainer& fines);
    virtual ~IFineProcessingContextBase() = default;

    const NDrive::IServer& GetServer() const;
    const NDrive::NFine::TFineConstructor& GetFineConstructor() const;
    const NDrive::NFine::TFinesManager& GetFinesManager() const;
    const TPrefechedFinesContainer::TMapByRulingNumber& GetFineMapByRulingNumber() const;
    const TPrefechedFinesContainer::TMapBySystemId& GetFineMapBySystemId() const;
    NDrive::NFine::TFineArticleMatcher::TPtr GetFineArticleMatcherPtr() const;

private:
    const NDrive::IServer& Server;
    const NDrive::NFine::TFineConstructor& FineConstructor;
    const TPrefechedFinesContainer& Fines;

    TAtomicSharedPtr<NDrive::NFine::TFineArticleMatcher> FineArticleMatcherPtr;
};

class TRTFinesCollectorBase: public IRTRegularBackgroundProcess {
    using TBase = IRTRegularBackgroundProcess;

protected:
    using ESourceType = NDrive::NFine::ESourceType;
    using ECollectingStatus = NDrive::NFine::ECollectingStatus;
    using TAutocodeFineEntry = NDrive::NFine::TAutocodeFineEntry;

    using TNotifyHandlers = TContextNotifyHandlerCollection<ECollectingStatus, NDrive::NFine::IFineContextFetcher>;

public:
    using TBase::TBase;

    virtual ESourceType GetSourceType() const = 0;

    virtual NDrive::TScheme DoGetScheme(const IServerBase& server) const override;
    virtual bool DoDeserializeFromJson(const NJson::TJsonValue& data) override;
    virtual NJson::TJsonValue DoSerializeToJson() const override;

protected:
    virtual bool DoStart(const TRTBackgroundProcessContainer& container) override;
    virtual void InitNotifyHandlers() final;

    bool ProcessFines(IFineProcessingContextBase::TPtr contextPtr, TMessagesCollector& errors) const;

    NDrive::NFine::TFineFilterGroup GetFilters() const;

    virtual bool ConstructFines(IFineProcessingContextBase::TPtr contextPtr, TVector<TAutocodeFineEntry>& nativeFines, TMessagesCollector& errors) const = 0;

    virtual TVector<TAutocodeFineEntry> FilterFines(IFineProcessingContextBase::TPtr contextPtr, TVector<TAutocodeFineEntry>&& nativeFines) const;
    virtual bool FilterNativeFine(const IFineProcessingContextBase::TPtr contextPtr, const NDrive::NFine::TFineFetchContext& fineContext, TMessagesCollector& errors) const;
    virtual bool CheckNativeFine(const IFineProcessingContextBase::TPtr contextPtr, const NDrive::NFine::TFineFetchContext& fineContext, TMessagesCollector& errors) const;

    virtual bool UpsertFines(const IFineProcessingContextBase::TPtr contextPtr, TVector<TAutocodeFineEntry>&& fines, TMessagesCollector& errors) const;
    virtual bool OnBeforeCommit(const IFineProcessingContextBase::TPtr contextPtr, const TVector<TAutocodeFineEntry>& fines, NDrive::TEntitySession& session, TMessagesCollector& errors) const;
    virtual bool OnAfterCommit(const IFineProcessingContextBase::TPtr contextPtr, TMessagesCollector& errors) const;

private:
    R_FIELD(TSet<TString>, ArticlesFilter);
    R_FIELD(bool, CheckUniqueAllSourceTypes, false);
    R_FIELD(TDuration, FinesHistoryDeep, TDuration::Days(30));

    R_FIELD(NDrive::NFine::TFineConstructorConfig, FineConstructorConfig);
    R_FIELD(NDrive::NFine::TFineFetchContextConfig, FetchContextConfig);

    R_FIELD(TNotifyHandlers::TPtr, NotifyHandlers, nullptr);
};
