#include "config.h"

#include <drive/backend/data/chargable.h>
#include <drive/backend/database/drive_api.h>
#include <drive/backend/database/drive/url.h>
#include <drive/backend/device_snapshot/manager.h>
#include <drive/backend/offers/actions/fix_point.h>

#include <drive/library/cpp/scheme/scheme.h>

#include <rtline/library/unistat/cache.h>

TFixPointWatcher::TFactory::TRegistrator<TFixPointWatcher> TFixPointWatcher::Registrator("fixpoint_watcher");

TExpectedState TFixPointWatcher::DoExecuteFiltered(TAtomicSharedPtr<IRTBackgroundProcessState> /*state*/, const NDrive::IServer& server, TTagsModificationContext& /*context*/) const {
    TVector<::IEventsSession<TCarTagHistoryEvent>::TPtr> sessions = server.GetDriveAPI()->GetTagsManager().GetDeviceTags().GetHistoryManager().GetSessionsBuilder("billing", StartInstant)->GetSessionsActual();
    TVector<TString> criticalObjectsInfo;
    ui32 fixPointCount = 0;
    TMap<TString, double> signalsLocal;
    NUnistat::TIntervals intervals;
    for (auto&& i : PercentProblems) {
        intervals.emplace_back(i);
    }
    for (auto&& i : sessions) {
        const TBillingSession* bSession = dynamic_cast<const TBillingSession*>(i.Get());
        if (!bSession) {
            continue;
        }
        const TFixPointOffer* fOffer = dynamic_cast<const TFixPointOffer*>(bSession->GetCurrentOffer().Get());
        if (!fOffer) {
            continue;
        }
        ++fixPointCount;
        TBillingSession::TBillingCompilation bCompilation;
        TBillingSession::TBillingEventsCompilation eCompilation;
        if (!bSession->FillCompilation(bCompilation) || !bSession->FillCompilation(eCompilation)) {
            signalsLocal["compilation-problem"]++;
            continue;
        }

        double realDistance = eCompilation.GetMileageMax() - eCompilation.GetMileageOnStart();
        if (bCompilation.GetRideDuration() < ThresholdDuration) {
            continue;
        }
        if (realDistance < ThresholdDistance / 1000.0) {
            continue;
        }

        auto locale = fOffer->GetLocale();

        WARNING_LOG << realDistance << " / " << fOffer->GetRouteLength() / 1000 << " / " << bCompilation.GetRideDuration() << " / " << bCompilation.GetSumDuration() << " / " << fOffer->GetRouteDuration() << Endl;
        for (auto&& i : PercentProblems) {
            const TDuration dDuration = fOffer->GetRouteDuration() * (1.0 + i / 100.0);
            if (bCompilation.GetRideDuration() > dDuration) {
                signalsLocal["offers_rduration_g" + ::ToString(i)]++;
                signalsLocal["rduration_g" + ::ToString(i)] += (bCompilation.GetRideDuration() - dDuration).Seconds();
            }

            double dDistance = fOffer->GetRouteLength() * (1.0 + i / 100.0) / 1000.0;
            if (realDistance > dDistance) {
                signalsLocal["offers_distance_g" + ::ToString(i)]++;
                signalsLocal["distance_g" + ::ToString(i)] += (realDistance - dDistance);
            }

            if (bCompilation.GetSumDuration() > dDuration) {
                signalsLocal["offers_duration_g" + ::ToString(i)]++;
                signalsLocal["duration_g" + ::ToString(i)] += (bCompilation.GetSumDuration() - dDuration).Seconds();
            }

        }
        TString problemsDuration;
        TString problemsDistance;
        TString problemsWaiting;
        if (bCompilation.GetRideDuration().Seconds() > fOffer->GetRouteDuration().Seconds() * (1.0 + CriticalPercentDuration / 100.0)) {
            const TString rideDuration = server.GetLocalization()->FormatDuration(locale, bCompilation.GetRideDuration());
            const TString routeDuration = server.GetLocalization()->FormatDuration(locale, fOffer->GetRouteDuration());
            problemsDuration = Sprintf("%s vs %s (duration); ", rideDuration.data(), routeDuration.data());
        }
        if (realDistance > fOffer->GetRouteLength() / 1000.0 * (1.0 + CriticalPercentDistance / 100.0)) {
            problemsDistance = Sprintf("%0.2f vs %0.2f (distance); ", realDistance, fOffer->GetRouteLength() / 1000);
        }
        const TFixPointOfferState* state = bCompilation.GetCurrentOfferStateAs<TFixPointOfferState>();
        TString prefix;
        if (state) {
            if (state->HasDropAvailable() && state->GetDropAvailableUnsafe()) {
                prefix = "!! ";
            }
            if (state->GetWaitingDuration() > CriticalWaitingDuration) {
                const TString waitingDuration = server.GetLocalization()->FormatDuration(locale, state->GetWaitingDuration());
                problemsWaiting = Sprintf("%s (waiting); ", waitingDuration.data());
            }
        }
        if (problemsDuration || problemsDistance || problemsWaiting) {
            criticalObjectsInfo.emplace_back(prefix + Sprintf("<a href=\"%s\">поездка</a> %s%s%s", TCarsharingUrl().SessionPage(bSession->GetSessionId()).data(), problemsDuration.data(), problemsDistance.data(), problemsWaiting.data()));
        }
        if (fOffer->GetRouteDuration().Seconds()) {
            TUnistatSignalsCache::SignalHistogram(GetType(), "hdurations", 100.0 * bCompilation.GetRideDuration().Seconds() / fOffer->GetRouteDuration().Seconds(), intervals);
        }
        if (fOffer->GetRouteLength()) {
            TUnistatSignalsCache::SignalHistogram(GetType(), "hdistances", 100.0 * 1000 * realDistance / fOffer->GetRouteLength(), intervals);
        }
    }
    for (auto&& i : signalsLocal) {
        TUnistatSignalsCache::SignalAddX(GetType(), i.first, i.second);
    }
    NDrive::INotifier::MultiLinesNotify(server.GetNotifier(NotifierName), "Проблемы с fixpoint (всего поездок " + ::ToString(fixPointCount) + ")", criticalObjectsInfo);
    return new IRTBackgroundProcessState();
}

NDrive::TScheme TFixPointWatcher::DoGetScheme(const IServerBase& server) const {
    NDrive::TScheme scheme = TBase::DoGetScheme(server);
    scheme.Add<TFSString>("percent_problems", "Гистограмма");
    scheme.Add<TFSNumeric>("critical_percent_duration", "Критический процент для длительности").SetDefault(30);
    scheme.Add<TFSNumeric>("critical_percent_distance", "Критический процент для расстояния").SetDefault(30);
    scheme.Add<TFSNumeric>("threshold_distance", "Порог по расстоянию для начала слежения (м)").SetDefault(500);
    scheme.Add<TFSDuration>("threshold_duration", "Порог по времени для начала слежения").SetDefault(TDuration::Minutes(5));
    scheme.Add<TFSDuration>("critical_waiting_duration", "Критичное время дополнительной оплаты (ожидание)").SetDefault(TDuration::Minutes(5));

    scheme.Add<TFSVariants>("notifier", "Нотификации").SetVariants(server.GetNotifierNames());
    return scheme;
}
