#pragma once

#include <drive/backend/rt_background/common/config.h>
#include <drive/backend/rt_background/manager/manager.h>

#include <drive/backend/tags/tags_filter.h>

#include <drive/library/cpp/scheme/scheme.h>

#include <rtline/util/json_processing.h>

class TFixPointWatcher: public IRTCarsProcess {
private:
    using TBase = IRTCarsProcess;

public:
    R_READONLY(TSet<i32>, PercentProblems);
    R_READONLY(ui32, CriticalPercentDuration, 0);
    R_READONLY(ui32, CriticalPercentDistance, 0);
    R_READONLY(TDuration, CriticalWaitingDuration, TDuration::Minutes(5));

    R_READONLY(double, ThresholdDistance, 0);
    R_READONLY(TDuration, ThresholdDuration, TDuration::Minutes(5));
    R_READONLY(TString, NotifierName);

private:
    static TFactory::TRegistrator<TFixPointWatcher> Registrator;

public:
    virtual bool IsSimultaneousProcess() const override {
        return false;
    }

    virtual TString GetType() const override {
        return "fixpoint_watcher";
    }

    using TBase::TBase;

    virtual TExpectedState DoExecuteFiltered(TAtomicSharedPtr<IRTBackgroundProcessState> state, const NDrive::IServer& server, TTagsModificationContext& context) const override;

    virtual NDrive::TScheme DoGetScheme(const IServerBase& server) const override;

    virtual NJson::TJsonValue DoSerializeToJson() const override {
        NJson::TJsonValue result = TBase::DoSerializeToJson();
        JWRITE(result, "percent_problems", JoinSeq(",", PercentProblems));
        JWRITE(result, "critical_percent_duration", CriticalPercentDuration);
        JWRITE(result, "critical_percent_distance", CriticalPercentDistance);
        JWRITE(result, "threshold_distance", ThresholdDistance);
        JWRITE_DURATION(result, "threshold_duration", ThresholdDuration);
        JWRITE_DURATION(result, "critical_waiting_duration", CriticalWaitingDuration);

        JWRITE(result, "notifier", NotifierName);

        return result;
    }

    virtual bool DoDeserializeFromJson(const NJson::TJsonValue& jsonInfo) override {
        if (!TBase::DoDeserializeFromJson(jsonInfo)) {
            return false;
        }
        TString percentProblems;
        JREAD_STRING_OPT(jsonInfo, "percent_problems", percentProblems);
        try {
            StringSplitter(percentProblems).SplitBySet(", ").SkipEmpty().ParseInto(&PercentProblems);
        } catch (const std::exception& e) {
            ERROR_LOG << "cannot deserialize FixPointWatcher: " << FormatExc(e) << Endl;
            return false;
        }
        JREAD_INT_OPT(jsonInfo, "critical_percent_duration", CriticalPercentDuration);
        JREAD_INT_OPT(jsonInfo, "critical_percent_distance", CriticalPercentDistance);
        JREAD_INT_OPT(jsonInfo, "threshold_distance", ThresholdDistance);
        JREAD_DURATION_OPT(jsonInfo, "threshold_duration", ThresholdDuration);
        JREAD_DURATION_OPT(jsonInfo, "critical_waiting_duration", CriticalWaitingDuration);

        JREAD_STRING_OPT(jsonInfo, "notifier", NotifierName);
        return true;
    }
};
