#include "process.h"

#include <drive/backend/data/fueling.h>
#include <drive/backend/database/drive_api.h>
#include <drive/backend/tags/tags_manager.h>


TExpectedState TUserFuelingTagCleanProcess::DoExecute(TAtomicSharedPtr<IRTBackgroundProcessState> /*state*/, const TExecutionContext& context) const {
    const auto& server = context.GetServerAs<NDrive::IServer>();
    TVector<TDBTag> tagsFullList;
    if (!server.GetFuelingManager()) {
        ERROR_LOG << GetRobotId() << ": Cannot get user tags " << Endl;
        return nullptr;
    }
    {
        auto session = server.GetDriveAPI()->template BuildTx<NSQL::ReadOnly>();
        if (!server.GetDriveAPI()->GetTagsManager().GetUserTags().RestoreTags({}, {TUserFuelingTag::GetTypeName()}, tagsFullList, session)) {
            return MakeUnexpected<TString>(GetRobotId() + ": Cannot get user tags " + session.GetStringReport());
        }
    }
    TVector<TDBTag> tagsToRemove;
    const TInstant startProcess = Now();
    const auto& driveApi = server.GetDriveAPI();
    for (auto&& i : tagsFullList) {
        TUserFuelingTag* ufTag = i.MutableTagAs<TUserFuelingTag>();
        if (ufTag && StatusesToRemove.contains(ufTag->GetCurrentState())) {
            TVector<ISession::TConstPtr> currentSessions;
            if (!driveApi->GetCurrentUserSessions(i.GetObjectId(), currentSessions, startProcess)) {
                return MakeUnexpected<TString>(GetRobotId() + ": Cannot get user sessions");
            }
            bool activeTag = false;
            for (const auto& session : currentSessions) {
                if (session->GetSessionId() == ufTag->GetSessionId() && !session->IsCompiled()) {
                    activeTag = true;
                    break;
                }
            }
            if (!activeTag) {
                tagsToRemove.emplace_back(i);
            }
        }
    }
    auto session = server.GetDriveAPI()->template BuildTx<NSQL::Writable>();
    if (!server.GetDriveAPI()->GetTagsManager().GetUserTags().RemoveTags(tagsToRemove, GetRobotUserId(), &server, session) || !session.Commit()) {
        return MakeUnexpected<TString>(GetRobotId() + ": Cannot get user tags " + session.GetStringReport());
    }
    return new IRTBackgroundProcessState();
}

NDrive::TScheme TUserFuelingTagCleanProcess::DoGetScheme(const IServerBase& server) const {
    NDrive::TScheme scheme = TBase::DoGetScheme(server);
    scheme.Add<TFSVariants>("statuses_to_remove", "Статусы подлежащих удалению тегов").SetMultiSelect(true).InitVariants<EFuelingStatus>().SetRequired(true);
    return scheme;
}

bool TUserFuelingTagCleanProcess::DoDeserializeFromJson(const NJson::TJsonValue& value) {
    JREAD_CONTAINER_OPT(value, "statuses_to_remove", StatusesToRemove);
    return TBase::DoDeserializeFromJson(value);
}

NJson::TJsonValue TUserFuelingTagCleanProcess::DoSerializeToJson() const {
    NJson::TJsonValue result = TBase::DoSerializeToJson();
    TJsonProcessor::WriteContainerArrayStrings(result, "statuses_to_remove", StatusesToRemove);
    return result;
}

TUserFuelingTagCleanProcess::TFactory::TRegistrator<TUserFuelingTagCleanProcess> TUserFuelingTagCleanProcess::Registrator(TUserFuelingTagCleanProcess::GetTypeName());
