#include "config.h"

#include <drive/backend/data/chargable.h>
#include <drive/backend/data/transformation.h>
#include <drive/backend/database/drive_api.h>
#include <drive/backend/device_snapshot/manager.h>

#include <drive/library/cpp/scheme/scheme.h>

#include <rtline/library/unistat/cache.h>

TRTFuturesBook::TFactory::TRegistrator<TRTFuturesBook> TRTFuturesBook::Registrator("futures_book");

TExpectedState TRTFuturesBook::DoExecuteFiltered(TAtomicSharedPtr<IRTBackgroundProcessState> /*state*/, const NDrive::IServer& server, TTagsModificationContext& context) const {
    TVector<TTaggedDevice> devices;
    if (!server.GetDriveAPI()->GetTagsManager().GetDeviceTags().GetCustomObjectsFromCache(devices, context.GetFilteredCarIds())) {
        return MakeUnexpected<TString>({});
    }
    TDevicesSnapshot snapshots = server.GetSnapshotsManager().GetSnapshots(context.GetFilteredCarIds());
    NDrive::INotifier::TPtr notifier = server.GetNotifier(GetNotifierName());
    for (auto&& i : devices) {
        bool isReady = true;
        TDBTag fTag;
        const TTagReservationFutures* trf = nullptr;
        for (auto&& tag : i.GetTags()) {
            if (!!tag->GetPerformer()) {
                if (tag.GetTagAs<TTagReservationFutures>()) {
                    fTag = tag;
                    trf = fTag.GetTagAs<TTagReservationFutures>();
                } else if (tag.GetTagAs<TChargableTag>() || tag.GetTagAs<TTransformationTag>()) {
                    isReady = false;
                }
            }
        }
        if (!trf) {
            continue;
        }
        TString message;
        bool doBook = false;
        if (trf->IsExpired()) {
            message = "offers.futures.expired";
        } else if (isReady) {
            auto* itSnapshot = snapshots.GetSnapshot(fTag.GetObjectId());
            if (!itSnapshot) {
                message = "offers.futures.car_data_absent";
            } else {
                NDrive::TLocation location;
                if (!itSnapshot->GetRawLocation(location, TDuration::Max())) {
                    message = "offers.futures.car_location_undefined";
                } else {
                    if (!server.GetSettings().GetValueDef("futures.only_in_finish_area", false) || trf->IsCorrectPosition(location.GetCoord())) {
                        doBook = true;
                    } else {
                        message = "offers.futures.incorrect_car_finish";
                    }
                }
            }
        } else {
            continue;
        }
        auto offer = trf->GetVehicleOffer();
        auto locale = offer ? offer->GetLocale() : DefaultLocale;
        auto session = server.GetDriveAPI()->template BuildTx<NSQL::Writable>();
        if (doBook) {
            auto bookPermissions = server.GetDriveAPI()->GetUserPermissions(fTag->GetPerformer(), TUserPermissionsFeatures());
            if (!bookPermissions) {
                message = "offers.futures.server_error_on_booking";
            } else {
                TChargableTag::TBookOptions bookOptions;
                bookOptions.MultiRent = bookPermissions->GetSetting<bool>("futures.multi_rent", false);
                auto br = TChargableTag::Book(offer, *bookPermissions, server, session, bookOptions);
                if (!br) {
                    message = "offers.futures.server_error_on_booking";
                    NDrive::TEventLog::TUserIdGuard userIdGuard(bookPermissions->GetUserId());
                    NDrive::TEventLog::Log("BookFuturesOfferError", NJson::TMapBuilder
                        ("offer_id", offer ? NJson::ToJson(offer->GetOfferId()) : NJson::JSON_NULL)
                        ("tag_id", fTag.GetTagId())
                        ("message", message)
                        ("error", session.GetReport())
                    );
                    session = server.GetDriveAPI()->template BuildTx<NSQL::Writable>();
                } else {
                    NOTICE_LOG << "futures booked" << Endl;
                }
            }
        }
        if (message) {
            ERROR_LOG << message << ": for " << trf->GetPerformer() << Endl;
            if (!server.GetDriveAPI()->GetTagsManager().GetUserTags().AddTag(trf->BuildFailed(message), GetRobotUserId(), fTag->GetPerformer(), &server, session)) {
                TUnistatSignalsCache::SignalAdd("futures", "errors", 1);
                ERROR_LOG << GetRobotId() << ": cannot add failed TagReservationFutures tag: " << session.GetStringReport() << Endl;
                continue;
            }
        } else {
            message = "offers.futures.success";
        }
        if (!server.GetDriveAPI()->GetTagsManager().GetDeviceTags().RemoveTag(fTag, GetRobotUserId(), &server, session, true)) {
            TUnistatSignalsCache::SignalAdd("futures", "errors", 1);
            ERROR_LOG << GetRobotId() << ": cannot remove tag: " << session.GetStringReport() << Endl;
            continue;
        }
        auto gUsers = server.GetDriveAPI()->GetUsersData()->FetchInfo(fTag->GetPerformer(), session);
        if (session.Commit()) {
            auto* userData = gUsers.GetResultPtr(fTag->GetPerformer());
            if (userData) {
                if (message != "offers.futures.success") {
                    TUnistatSignalsCache::SignalAdd("futures", "errors", 1);
                } else {
                    TUnistatSignalsCache::SignalAdd("futures", "success", 1);
                }
                TUnistatSignalsCache::SignalAdd("futures", message, 1);
                NDrive::INotifier::Notify(
                    notifier,
                    server.GetLocalization()->GetLocalString(locale, message),
                    NDrive::INotifier::TContext().SetRecipients({*userData})
                );
            } else {
                TUnistatSignalsCache::SignalAdd("futures", "user-id-error", 1);
                TUnistatSignalsCache::SignalAdd("futures", "errors", 1);
                WARNING_LOG << "Incorrect user id: " << fTag->GetPerformer() << Endl;
            }
        } else {
            TUnistatSignalsCache::SignalAdd("futures", "transaction-error", 1);
            TUnistatSignalsCache::SignalAdd("futures", "errors", 1);
            ERROR_LOG << GetRobotId() << ": " << session.GetStringReport() << Endl;
        }
    }
    return new IRTBackgroundProcessState();
}

NDrive::TScheme TRTFuturesBook::DoGetScheme(const IServerBase& server) const {
    NDrive::TScheme scheme = TBase::DoGetScheme(server);
    scheme.Add<TFSVariants>("NotifierName").SetVariants(server.GetNotifierNames());
    return scheme;
}
