#include "config.h"

#include <drive/library/cpp/solomon/client.h>

TRTGraphBuilderBackground::TFactory::TRegistrator<TRTGraphBuilderBackground> TRTGraphBuilderBackground::Registrator(TRTGraphBuilderBackground::GetTypeName());

TExpectedState TRTGraphBuilderBackground::DoExecute(TAtomicSharedPtr<IRTBackgroundProcessState> /*state*/, const TExecutionContext& context) const {
    const NDrive::IServer& frServer = context.GetServerAs<NDrive::IServer>();
    if (!frServer.GetDriveAPI()->HasSolomonClient()) {
        ERROR_LOG << "Solomon client not initialized" << Endl;
        return MakeUnexpected<TString>({});
    }

    TInstant startTime;
    switch (AggregationByTime) {
    case EAggregationByTime::EveryDay:
        startTime = TInstant::Days((StartInstant + TimeShift).Days()) - TimeShift;
        break;
    case EAggregationByTime::EveryHour:
        startTime = TInstant::Hours(StartInstant.Hours());
        break;
    default:
        ERROR_LOG << "Unknown aggregation type" << Endl;
        return MakeUnexpected<TString>({});
    }

    TVector<double> signalValues;
    for (const auto& signal : Signals) {

        TMessagesCollector errors;
        TSolomonClient::TTimeseries timeseries;
        if (!frServer.GetDriveAPI()->GetSolomonClient().GetData(Program, startTime - signal.GetDelta(), StartInstant - signal.GetDelta(), AggregationBySlots, timeseries, errors)) {
            NDrive::INotifier::Notify(frServer.GetNotifier(Notifier), GetRTProcessName() + " " + errors.GetStringReport());
            return MakeUnexpected<TString>({});
        }

        if (!timeseries.size()) {
            ERROR_LOG << "Empty signal" << Endl;
            return MakeUnexpected<TString>({});
        }

        TSolomonClient::TTimeseries::TPoint::TValue resultValue = timeseries[0].GetValue();
        for (size_t i = 1; i < timeseries.size(); ++i) {
            switch (AggregationByTimeline) {
            case EAggregationType::Average:
                resultValue.Aggregate(resultValue, timeseries[i].GetValue(), [](double a, double b) {
                    return (a + b) / 2.;
                });
                break;
            case EAggregationType::Max:
                resultValue.Aggregate(resultValue, timeseries[i].GetValue(), [](double a, double b) {
                    return Max<double>(a, b);
                });
                break;
            case EAggregationType::Min:
                resultValue.Aggregate(resultValue, timeseries[i].GetValue(), [](double a, double b) {
                    return Min<double>(a, b);
                });
                break;
            case EAggregationType::Sum:
            case EAggregationType::SumOne:
                resultValue.Aggregate(resultValue, timeseries[i].GetValue(), [](double a, double b) {
                    return a + b;
                });
                break;
            case EAggregationType::LastValue:
                resultValue = timeseries[i].GetValue();
                break;
            default:
                ERROR_LOG << "Incorrect aggregation type" << Endl;
                return MakeUnexpected<TString>({});
            }
        }

        double signalValue = 0;
        switch (AggregationBySignal) {
        case EAggregationType::Max:
            signalValue = resultValue.GetMax();
            break;
        case EAggregationType::Min:
            signalValue = resultValue.GetMin();
            break;
        case EAggregationType::Average:
        case EAggregationType::Sum:
        case EAggregationType::SumOne:
            signalValue = resultValue.GetSum();
            break;
        case EAggregationType::LastValue:
            signalValue = resultValue.GetLast();
            break;
        default:
            return MakeUnexpected<TString>("Incorrect aggregation type");
        }

        if (UseNotification) {
            NDrive::INotifier::Notify(frServer.GetNotifier(Notifier), signal.GetName() + ":" + ::ToString(signalValue));
        }
        TUnistatSignalsCache::SignalSpec("frontend", signal.GetName(), signalValue, AggregationByTass, Suffix);
        signalValues.push_back(signalValue);
    }

    for (const auto& exp : Expressions) {
        if (!exp.GetExpression()) {
            return MakeUnexpected<TString>("Incorrect expression");
        }
        auto expValue = exp.GetExpression()->Calc(signalValues);
        if (!expValue) {
            return MakeUnexpected(expValue.GetError());
        }
        NDrive::INotifier::Notify(frServer.GetNotifier(exp.GetNotifier()), exp.GetName() + " : " + ::ToString(std::ceil(*expValue * exp.GetPrecision()) * 1. / exp.GetPrecision()));
    }
    return new IRTBackgroundProcessState();
}

template <>
NJson::TJsonValue NJson::ToJson<TRTGraphBuilderBackground::TSignal>(const TRTGraphBuilderBackground::TSignal& object) {
    return object.ToJson();
}

template <>
bool NJson::TryFromJson<TRTGraphBuilderBackground::TSignal>(const TJsonValue& value, TRTGraphBuilderBackground::TSignal& result) {
    return result.FromJson(value);
}

template <>
NJson::TJsonValue NJson::ToJson<TRTGraphBuilderBackground::TExpression>(const TRTGraphBuilderBackground::TExpression& object) {
    return object.ToJson();
}

template <>
bool NJson::TryFromJson<TRTGraphBuilderBackground::TExpression>(const TJsonValue& value, TRTGraphBuilderBackground::TExpression& result) {
    return result.FromJson(value);
}

NDrive::TScheme TRTGraphBuilderBackground::DoGetScheme(const IServerBase& server) const {
    NDrive::TScheme scheme = TBase::DoGetScheme(server);
    scheme.Add<TFSVariants>("notifier", "Нотифаер").SetVariants(server.GetNotifierNames());
    scheme.Add<TFSText>("program", "Сигнал для агрегации").SetRequired(true);
    scheme.Add<TFSString>("signal_prefix", "Суффикс сигнала").SetRequired(true);
    scheme.Add<TFSVariants>("aggregation_type", "Агрегация по слотам").InitVariants<EAggregationType>().SetDefault(::ToString(EAggregationType::Sum)).SetRequired(true);
    scheme.Add<TFSVariants>("aggregation_signal", "Агрегация yasm(внутренняя)").InitVariants<EAggregationType>().SetDefault(::ToString(EAggregationType::Sum)).SetRequired(true);
    scheme.Add<TFSVariants>("aggregation_by_time", "Период агрегации по времени").InitVariants<EAggregationByTime>().SetDefault(::ToString(EAggregationByTime::EveryDay)).SetRequired(true);
    scheme.Add<TFSVariants>("aggregation_timeline", "Операция агрегации временного ряда").InitVariants<EAggregationType>().SetDefault(::ToString(EAggregationType::Sum)).SetRequired(true);
    scheme.Add<TFSBoolean>("use_notification", "Включить нотификацию").SetDefault(false);
    scheme.Add<TFSVariants>("aggregation_by_tass", "Агрегация tass").InitVariants<EAggregationType>().SetDefault(::ToString(EAggregationType::LastValue)).SetRequired(true);
    scheme.Add<TFSDuration>("time_shift", "Сдвиг времени(EveryDay aggregation)").SetDefault(TDuration::Hours(3));
    scheme.Add<TFSArray>("signals", "Сигналы").SetElement(TSignal::GetScheme()).SetRequired(true);
    scheme.Add<TFSArray>("expressions", "Выражения").SetElement(TExpression::GetScheme(server));
    return scheme;
}

bool TRTGraphBuilderBackground::DoDeserializeFromJson(const NJson::TJsonValue& jsonInfo) {
    JREAD_STRING_OPT(jsonInfo, "notifier", Notifier);
    JREAD_STRING(jsonInfo, "program", Program);
    JREAD_STRING(jsonInfo, "signal_prefix", Suffix);
    JREAD_FROM_STRING(jsonInfo, "aggregation_type", AggregationBySlots);
    JREAD_FROM_STRING_OPT(jsonInfo, "aggregation_signal", AggregationBySignal);
    JREAD_FROM_STRING_OPT(jsonInfo, "aggregation_timeline", AggregationByTimeline);
    JREAD_FROM_STRING(jsonInfo, "aggregation_by_time", AggregationByTime);
    JREAD_BOOL_OPT(jsonInfo, "use_notification", UseNotification);
    JREAD_FROM_STRING_OPT(jsonInfo, "aggregation_by_tass", AggregationByTass);
    JREAD_DURATION_OPT(jsonInfo, "time_shift", TimeShift);
    if (!NJson::ParseField(jsonInfo, "signals", Signals)) {
        return false;
    }
    if (jsonInfo.Has("signal_name")) {
        TDuration delta = TDuration::Zero();
        JREAD_DURATION_OPT(jsonInfo, "delta", delta);
        Signals.emplace_back(jsonInfo["signal_name"].GetStringRobust(), delta);
    }
    return NJson::ParseField(jsonInfo, "expressions", Expressions) && TBase::DoDeserializeFromJson(jsonInfo);
}

NJson::TJsonValue TRTGraphBuilderBackground::DoSerializeToJson() const {
    NJson::TJsonValue result = TBase::DoSerializeToJson();
    TJsonProcessor::Write(result, "notifier", Notifier);
    TJsonProcessor::Write(result, "program", Program);
    TJsonProcessor::Write(result, "signal_prefix", Suffix);
    TJsonProcessor::Write(result, "aggregation_type", ::ToString(AggregationBySlots));
    TJsonProcessor::Write(result, "aggregation_signal", ::ToString(AggregationBySignal));
    TJsonProcessor::Write(result, "aggregation_timeline", ::ToString(AggregationByTimeline));
    TJsonProcessor::Write(result, "aggregation_by_time", ::ToString(AggregationByTime));
    TJsonProcessor::Write(result, "use_notification", UseNotification);
    TJsonProcessor::Write(result, "aggregation_by_tass", ::ToString(AggregationByTass));
    TJsonProcessor::Write(result, "time_shift", TimeShift);
    TJsonProcessor::Write(result, "signals", NJson::ToJson(Signals));
    TJsonProcessor::Write(result, "expressions", NJson::ToJson(Expressions));
    return result;
}
