#pragma once
#include <drive/backend/rt_background/common/config.h>

#include <drive/backend/background/manager/regular.h>

#include <rtline/util/types/accessor.h>
#include <drive/library/cpp/exp_calculator/exp_calculator.h>

enum class EAggregationByTime {
    EveryHour /* "every_hour" */,
    EveryDay /* "every_day" */,
};

class TRTGraphBuilderBackground : public IRTRegularBackgroundProcess {
public:
    class TSignal {
        R_READONLY(TString, Name);
        R_READONLY(TDuration, Delta, TDuration::Zero());

    public:
        TSignal() = default;
        TSignal(const TString& name, const TDuration delta)
            : Name(name)
            , Delta(delta)
        {}

        static NDrive::TScheme GetScheme() {
            NDrive::TScheme scheme;
            scheme.Add<TFSString>("signal", "Имя сигнала").SetRequired(true);
            scheme.Add<TFSDuration>("delta", "Смещение").SetDefault(TDuration::Zero());
            return scheme;
        }

        NJson::TJsonValue ToJson() const {
            NJson::TJsonValue json;
            TJsonProcessor::Write(json, "signal", Name);
            TJsonProcessor::WriteDurationString(json, "delta", Delta);
            return json;
        }

        bool FromJson(const NJson::TJsonValue& json) {
            JREAD_STRING(json, "signal", Name);
            JREAD_DURATION_OPT(json, "delta", Delta);
            return true;
        }
    };

    class TExpression {
    public:
        R_READONLY(TString, Name);
        R_READONLY(THolder<TExpressionCalculator>, Expression);
        R_READONLY(TString, Notifier);
        R_READONLY(ui64, Precision, 1);

    public:
        static NDrive::TScheme GetScheme(const IServerBase& server) {
            NDrive::TScheme scheme;
            scheme.Add<TFSString>("name", "Имя").SetRequired(true);
            scheme.Add<TFSString>("expression", "Выражение").SetRequired(true);
            scheme.Add<TFSVariants>("notifier", "Нотификация").SetVariants(server.GetNotifierNames()).SetRequired(true);
            scheme.Add<TFSNumeric>("precision", "Точность").SetDefault(1);
            return scheme;
        }

        NJson::TJsonValue ToJson() const {
            NJson::TJsonValue json;
            json.InsertValue("name", Name);
            json.InsertValue("expression", Expression ? Expression->GetRawExpression() : "");
            json.InsertValue("notifier", Notifier);
            json.InsertValue("precision", Precision);
            return json;
        }

        bool FromJson(const NJson::TJsonValue& json) {
            JREAD_STRING(json, "name", Name);
            JREAD_STRING(json, "notifier", Notifier);
            JREAD_UINT_OPT(json, "precision", Precision);
            if (!json.Has("expression") || !json["expression"].IsString()) {
                return false;
            }
            Expression = MakeHolder<TExpressionCalculator>(json["expression"].GetString());
            return true;
        }
    };

private:
    using TBase = IRTRegularBackgroundProcess;
    R_READONLY(TString, Notifier);
    R_READONLY(TString, Program);
    R_READONLY(EAggregationType, AggregationBySlots, EAggregationType::Sum);
    R_READONLY(EAggregationType, AggregationByTimeline, EAggregationType::Sum);
    R_READONLY(EAggregationType, AggregationBySignal, EAggregationType::Sum);
    R_READONLY(TString, Suffix);
    R_READONLY(bool, CleanCTypeAggregation, false);
    R_READONLY(EAggregationByTime, AggregationByTime, EAggregationByTime::EveryDay);
    R_READONLY(bool, UseNotification, false);
    R_READONLY(EAggregationType, AggregationByTass, EAggregationType::LastValue);
    R_READONLY(TDuration, TimeShift, TDuration::Hours(3));
    R_READONLY(TVector<TSignal>, Signals);
    R_READONLY(TVector<TExpression>, Expressions);
private:
    static TFactory::TRegistrator<TRTGraphBuilderBackground> Registrator;
protected:
    virtual TExpectedState DoExecute(TAtomicSharedPtr<IRTBackgroundProcessState> state, const TExecutionContext& context) const override;
    virtual NDrive::TScheme DoGetScheme(const IServerBase& server) const override;
    virtual bool DoDeserializeFromJson(const NJson::TJsonValue& jsonInfo) override;
    virtual NJson::TJsonValue DoSerializeToJson() const override;
public:
    virtual TString GetType() const override {
        return GetTypeName();
    }

    virtual bool IsSimultaneousProcess() const override {
        return true;
    }

    using TBase::TBase;

    static TString GetTypeName() {
        return "graph_builder";
    }
};
