#include "config.h"

#include <drive/backend/rt_background/history_events_counter/message_providers/default.h>

#include <drive/backend/data/chargable.h>
#include <drive/backend/database/drive_api.h>
#include <drive/backend/logging/events.h>
#include <drive/backend/offers/actions/abstract.h>

#include <rtline/library/json/adapters.h>
#include <rtline/library/json/builder.h>
#include <rtline/library/json/cast.h>
#include <rtline/library/json/parse.h>
#include <rtline/util/algorithm/container.h>

#include <util/generic/adaptor.h>
#include <util/string/cast.h>
#include <util/string/join.h>

IRTRegularBackgroundProcess::TFactory::TRegistrator<TRTCarHistoryEventsCounter> TRTCarHistoryEventsCounter::Registrator(TRTCarHistoryEventsCounter::GetTypeName());
TRTHistoryWatcherState::TFactory::TRegistrator<TRTCarHistoryEventsCounterState> TRTCarHistoryEventsCounterState::Registrator(TRTCarHistoryEventsCounter::GetTypeName());

IRTRegularBackgroundProcess::TFactory::TRegistrator<TRTUserHistoryEventsCounter> TRTUserHistoryEventsCounter::Registrator(TRTUserHistoryEventsCounter::GetTypeName());
TRTHistoryWatcherState::TFactory::TRegistrator<TRTUserHistoryEventsCounterState> TRTUserHistoryEventsCounterState::Registrator(TRTUserHistoryEventsCounter::GetTypeName());


TString TRTCarHistoryEventsCounterState::GetType() const {
    return TRTCarHistoryEventsCounter::GetTypeName();
}

TString TRTUserHistoryEventsCounterState::GetType() const {
    return TRTUserHistoryEventsCounter::GetTypeName();
}

template <class TBaseRegularBackgroundProcess>
bool TRTHistoryEventsCounter<TBaseRegularBackgroundProcess>::CheckFilter(const TCarTagHistoryEvent& ev, const IRTBackgroundProcess::TExecutionContext& /*context*/) const {
    if (!ActionsFilter.empty() && !ActionsFilter.contains(ev.GetHistoryAction())) {
        return false;
    }
    if (!TagsFilter.empty() && !TagsFilter.contains(ev->GetName())) {
        return false;
    }

    if (!UsersFilter.empty() && !UsersFilter.contains(ev.GetHistoryUserId())) {
        return false;
    }
    if (!OffersFilter.empty()) {
        auto chargableTag = ev.GetTagAs<TChargableTag>();
        if (!chargableTag) {
            return false;
        }
        auto offer = chargableTag->GetOffer();
        if (!offer) {
            return false;
        }
        if (!OffersFilter.contains(offer->GetBehaviourConstructorId()) && !OffersFilter.contains(offer->GetPriceConstructorId())) {
            return false;
        }
    }
    return true;
}

template <class TBaseRegularBackgroundProcess>
TExpectedState TRTHistoryEventsCounter<TBaseRegularBackgroundProcess>::Process(TAtomicSharedPtr<IRTBackgroundProcessState> stateExt, const IRTBackgroundProcess::TExecutionContext& context) const {
    const NDrive::IServer& server = context.GetServerAs<NDrive::IServer>();
    const TRTHistoryWatcherState* state = dynamic_cast<const TRTHistoryWatcherState*>(stateExt.Get());
    THolder<TRTHistoryWatcherState> result(BuildState());
    const auto& tagManager = GetEntityTagsManager(server);
    auto session = tagManager.BuildSession(/*readOnly=*/true);
    auto optionalLockedMaxEventId = tagManager.GetLockedMaxEventId(session);
    if (!optionalLockedMaxEventId) {
        return MakeUnexpected("cannot GetLockedMaxEventId: " + session.GetStringReport());
    }

    const ui64 maxEventId = *optionalLockedMaxEventId;
    const ui64 historyIdCursor = state ? state->GetLastEventId() : (StartEventId ? StartEventId : maxEventId);
    result->SetLastEventId(historyIdCursor);

    auto timestampRange = (EventTimestampLag) ? TRange<TInstant>{TInstant::Zero(), Now() - EventTimestampLag} : TRange<TInstant>{};
    ui64 limit = Limit;
    auto optionalEvents = tagManager.GetEvents(historyIdCursor + 1, timestampRange, session, limit);
    if (!optionalEvents) {
        return MakeUnexpected("cannot GetEventsSince: " + session.GetStringReport());
    }
    if (!session.Rollback()) {
        WARNING_LOG << TBase::GetRobotId() << ": cannot rollback tx: " << session.GetStringReport() << Endl;
    }

    auto messageProvider = IMessageProvider::Construct(
        MessageProviderName,
        MessageProviderConfig,
        &server,
        TBase::GetRTProcessName(),
        TBase::GetRobotUserId(),
        TBase::StartInstant,
        GetEntityType()
    );
    if (!messageProvider) {
        AddSignal(::ToString(EEventsCounterSignal::ContextInitError));
        return result.Release();
    }

    auto notifier = server.GetNotifier(NotifierName);
    NDrive::INotifier::TContext notifierContext;
    notifierContext.SetServer(&server);

    IMessageProvider::TMessages messages;

    auto currentEventId = historyIdCursor;
    TSet<TString> uniqueObjects;
    for (auto&& i : *optionalEvents) {
        if (i.GetHistoryEventId() > maxEventId) {
            continue;
        }
        if (!CheckFilter(i, context)) {
            currentEventId = std::max(currentEventId, i.GetHistoryEventId());
            continue;
        }

        const TString& performer = i->GetPerformer();
        if (!!performer && i.GetHistoryAction() == EObjectHistoryAction::Remove) {
            AddSignal(i->GetName(), "finished");
        }

        if (!uniqueObjects.emplace(messageProvider->GetUniqueName(i)).second) {
            break;
        }

        TMessagesCollector errors;

        if (notifier) try {
            auto fetchedMessages = messageProvider->Fetch(i, errors);
            for (auto&& fetchedMessage : fetchedMessages) {
                if (!fetchedMessage) {
                    ERROR_LOG << TBase::GetRobotId() << ": event " << i.GetHistoryEventId() << " created null message" << Endl;
                    AddSignal(::ToString(EEventsCounterSignal::FetchedMessageNull));
                    continue;
                }
                messages.push_back(fetchedMessage);
                INFO_LOG << TBase::GetRobotId() << ": event " << i.GetHistoryEventId() << " created a message " << fetchedMessage->SerializeToJson().GetStringRobust() << Endl;
                AddSignal(::ToString(EEventsCounterSignal::FetchedMessageValid));
            }
        } catch (const std::exception& e) {
            ERROR_LOG << TBase::GetRobotId() << ": event " << i.GetHistoryEventId() << " triggered an exception: " << FormatExc(e) << Endl;
            AddSignal(::ToString(EEventsCounterSignal::FetchedMessageException));
            break;
        }

        if (errors.HasMessages()) {
            ERROR_LOG << "Errors occurred during message fetching: " << errors.GetStringReport() << Endl;
            AddSignal(::ToString(EEventsCounterSignal::FetchedMessageError), errors.GetMessagesCount());
            if (UseEventLog) {
                NJson::TJsonValue errorReport = NJson::TMapBuilder("process_name", messageProvider->GetProcessName())
                                                                  ("message_provider", messageProvider->GetType())
                                                                  ("history_event_id", i.GetHistoryEventId())
                                                                  ("history_action", ::ToString(i.GetHistoryAction()))
                                                                  ("tag_id", i.GetTagId())
                                                                  ("tag", i->GetName())
                                                                  ("error", errors.GetReport());
                NDrive::TEventLog::Log("HistoryEventsCounterFetchError", errorReport);
            }
        }

        currentEventId = std::max(currentEventId, i.GetHistoryEventId());
        AddSignal(i->GetName(), ::ToString(i.GetHistoryAction()));
    }

    if (notifier && !messages.empty()) {
        auto results = notifier->MultiLinesNotify("Зафиксированы события: ", messages, notifierContext);
        for (auto&& result: results) {
            if (!result) {
                continue;
            }

            if (result->HasErrors()) {
                ERROR_LOG << "Errors occurred during message dispatching: " << result->SerializeToJson().GetStringRobust() << Endl;
                AddSignal(::ToString(EEventsCounterSignal::DispatchedMessageError));
                if (UseEventLog) {
                    NJson::TJsonValue errorReport = NJson::TMapBuilder("process_name", messageProvider->GetProcessName())
                                                                      ("message_provider", messageProvider->GetType())
                                                                      ("error", result->SerializeToJson());
                    NDrive::TEventLog::Log("HistoryEventsCounterDispatchError", errorReport);
                }
            }

            TMessagesCollector errors;
            if (!messageProvider->HandleResult(result, errors)) {
                ERROR_LOG << "Errors occurred during handing message result: " << errors.GetStringReport() << Endl;
                AddSignal(::ToString(EEventsCounterSignal::ResultHandlingError));
                if (UseEventLog) {
                    NJson::TJsonValue errorReport = NJson::TMapBuilder("process_name", messageProvider->GetProcessName())
                                                                      ("message_provider", messageProvider->GetType())
                                                                      ("result", result->SerializeToJson())
                                                                      ("handing_error", errors.GetStringReport());
                    NDrive::TEventLog::Log("HistoryEventsCounterResultHandingError", errorReport);
                }
            }
        }
    }

    result->SetLastEventId(currentEventId);
    return result;
}

template<class TBaseRegularBackgroundProcess>
NDrive::TScheme TRTHistoryEventsCounter<TBaseRegularBackgroundProcess>::DoGetScheme(const IServerBase& server) const {
    auto scheme = TBase::DoGetScheme(server);
    const auto impl = server.GetAs<NDrive::IServer>();
    CHECK_WITH_LOG(impl);
    const auto driveAPI = impl ? impl->GetDriveAPI() : nullptr;
    const auto registeredTags = driveAPI ? driveAPI->GetTagsManager().GetTagsMeta().GetRegisteredTags(GetEntityType()) : ITagsMeta::TTagDescriptionsByName();

    scheme.template Add<TFSVariants>("notifier_name", "Нотификатор").SetVariants(server.GetNotifierNames());
    scheme.template Add<TFSBoolean>("use_event_log", "Отправлять ошибки в event log").SetDefault(false);
    scheme.template Add<TFSDuration>("event_timestamp_lag", "Обрабатывать в текущей итерации события с запаздыванием").SetDefault(TDuration::Zero());
    scheme.template Add<TFSStructure>("message_provider_data", "Конфигурация провайдера сообщений").SetStructure(IMessageProvider::GetProvidersScheme(impl));
    scheme.template Add<TFSVariants>("tags_filter", "Фильтр тегов").SetVariants(registeredTags).SetMultiSelect(true);
    scheme.template Add<TFSVariants>("actions_filter", "Фильтр событий").template InitVariants<EObjectHistoryAction>().SetMultiSelect(true);
    scheme.template Add<TFSVariants>("offers_filter", "Фильтр событий (для chargable тегов)").SetVariants(IOfferBuilderAction::GetNames(server.GetAsPtrSafe<NDrive::IServer>())).SetMultiSelect(true);
    scheme.template Add<TFSArray>("users_filter", "Фильтр исполнителей").template SetElement<TFSString>();
    scheme.template Add<TFSNumeric>("start_event_id", "Стартовый евент");
    return scheme;
}

template<class TBaseRegularBackgroundProcess>
bool TRTHistoryEventsCounter<TBaseRegularBackgroundProcess>::DoDeserializeFromJson(const NJson::TJsonValue& jsonInfo) {
    if (jsonInfo.Has("message_provider_data")) {
        MessageProviderConfig = jsonInfo["message_provider_data"];
        if (!NJson::ParseField(MessageProviderConfig["message_provider"], MessageProviderName)) {
            return false;
        }
    } else {
        // backward compatibility, remove only if all history_events_counter bots are saved again
        if (jsonInfo.Has("message_provider_config")) {
            MessageProviderConfig = jsonInfo["message_provider_config"];
        }
        if (!NJson::ParseField(jsonInfo["message_provider_name"], MessageProviderName)) {
            return false;
        }
        MessageProviderConfig.InsertValue("message_provider", MessageProviderName);  // maintain invariant
    }

    if (!TJsonProcessor::ReadContainer(jsonInfo, "tags_filter", TagsFilter)) {
        return false;
    }
    if (!TJsonProcessor::ReadContainer(jsonInfo, "actions_filter", ActionsFilter)) {
        return false;
    }
    if (!TJsonProcessor::ReadContainer(jsonInfo, "users_filter", UsersFilter)) {
        return false;
    }
    if (!TJsonProcessor::ReadContainer(jsonInfo, "offers_filter", OffersFilter)) {
        return false;
    }
    JREAD_STRING_OPT(jsonInfo, "notifier_name", NotifierName);
    if (!TJsonProcessor::Read(jsonInfo, "use_event_log", UseEventLog)) {
        return false;
    }
    if (!TJsonProcessor::Read(jsonInfo, "event_timestamp_lag", EventTimestampLag)) {
        return false;
    }
    JREAD_UINT_OPT(jsonInfo, "start_event_id", StartEventId);
    NJson::TJsonValue baseJson = jsonInfo;
    baseJson.EraseValue("tags_filter");
    return TBase::DoDeserializeFromJson(baseJson);
}

template<class TBaseRegularBackgroundProcess>
NJson::TJsonValue TRTHistoryEventsCounter<TBaseRegularBackgroundProcess>::DoSerializeToJson() const {
    auto result = TBase::DoSerializeToJson();
    TJsonProcessor::WriteContainerArrayStrings(result, "actions_filter", ActionsFilter);
    TJsonProcessor::WriteContainerArray(result, "tags_filter", TagsFilter);
    TJsonProcessor::WriteContainerArray(result, "users_filter", UsersFilter);
    TJsonProcessor::WriteContainerArray(result, "offers_filter", OffersFilter);
    JWRITE(result, "notifier_name", NotifierName);
    JWRITE(result, "use_event_log", UseEventLog);
    TJsonProcessor::WriteDurationString(result, "event_timestamp_lag", EventTimestampLag);
    JWRITE(result, "message_provider_data", MessageProviderConfig);
    JWRITE(result, "start_event_id", StartEventId);

    // backward compatibility, remove after drive-37 released
    JWRITE(result, "message_provider_name", MessageProviderName);
    JWRITE(result, "message_provider_config", MessageProviderConfig);

    return result;
}

template class TRTHistoryEventsCounter<IRTCarsProcess>;
template class TRTHistoryEventsCounter<IRTRegularBackgroundProcess>;

bool TRTCarHistoryEventsCounter::CheckFilter(const TCarTagHistoryEvent& ev, const IRTBackgroundProcess::TExecutionContext& context) const {
    auto carContext = dynamic_cast<const TCarsProcessContext*>(&context);
    if (carContext && !carContext->GetFilteredCarIds().contains(ev.GetObjectId(ev))) {
        return false;
    }
    return TBase::CheckFilter(ev, context);
}
