#pragma once

#include <drive/backend/rt_background/history_events_counter/message_providers/base.h>

#include <drive/backend/rt_background/common/config.h>
#include <drive/backend/rt_background/common/state.h>

#include <drive/backend/background/manager/regular.h>
#include <drive/backend/proto/background.pb.h>

#include <rtline/library/time_restriction/time_restriction.h>
#include <rtline/library/unistat/cache.h>
#include <rtline/util/types/accessor.h>

enum class EEventsCounterSignal {
    ContextInitError /* "context_init_error" */,

    FetchedMessageNull /* "fetched_message_null" */,
    FetchedMessageValid /* "fetched_message_valid" */,
    FetchedMessageException /* "fetched_message_exception" */,
    FetchedMessageError /* "fetched_message_error" */,

    DispatchedMessageError /* "dispatched_message_error" */,

    ResultHandlingError /* "result_handling_error" */,
};

class TRTCarHistoryEventsCounterState: public TRTHistoryWatcherState {
public:
    virtual TString GetType() const override;

private:
    static TFactory::TRegistrator<TRTCarHistoryEventsCounterState> Registrator;
};

class TRTUserHistoryEventsCounterState: public TRTHistoryWatcherState {
public:
    virtual TString GetType() const override;

private:
    static TFactory::TRegistrator<TRTUserHistoryEventsCounterState> Registrator;
};

template <class TBaseRegularBackgroundProcess>
class TRTHistoryEventsCounter: public TBaseRegularBackgroundProcess {
private:
    using TBase = TBaseRegularBackgroundProcess;

public:
    using TBase::TBase;
    virtual NDrive::TScheme DoGetScheme(const IServerBase& server) const override;
    virtual bool DoDeserializeFromJson(const NJson::TJsonValue& jsonInfo) override;
    virtual NJson::TJsonValue DoSerializeToJson() const override;

protected:
    TExpectedState Process(TAtomicSharedPtr<IRTBackgroundProcessState> state, const IRTBackgroundProcess::TExecutionContext& context) const;
    virtual bool CheckFilter(const TCarTagHistoryEvent& ev, const IRTBackgroundProcess::TExecutionContext& context) const;

    void AddSignal(const TString& metricName, const double value = 1.0) const {
        AddSignal("", metricName, value);
    }

    void AddSignal(const TString& signalName, const TString& metricName, const double value = 1.0) const {
        TUnistatSignalsCache::SignalAdd(TBase::GetRTProcessName() + ((signalName) ? "-" + signalName : ""), metricName, value);
    }

private:
    virtual const IEntityTagsManager& GetEntityTagsManager(const NDrive::IServer& server) const = 0;
    virtual NEntityTagsManager::EEntityType GetEntityType() const = 0;
    virtual TRTHistoryWatcherState* BuildState() const = 0;

private:
    R_READONLY(TSet<TString>, TagsFilter);
    R_READONLY(TSet<TString>, UsersFilter);
    R_READONLY(TSet<TString>, OffersFilter);
    R_READONLY(ui64, Limit, 100000);
    R_READONLY(ui64, StartEventId, 0);
    R_READONLY(TSet<EObjectHistoryAction>, ActionsFilter);
    R_READONLY(TString, NotifierName);
    R_READONLY(bool, UseEventLog, false);
    R_READONLY(TDuration, EventTimestampLag);
    R_READONLY(TString, MessageProviderName, IMessageProvider::DefaultTypeName);
    R_READONLY(NJson::TJsonValue, MessageProviderConfig, NJson::JSON_MAP);
};

class TRTCarHistoryEventsCounter: public TRTHistoryEventsCounter<IRTCarsProcess> {
    using TBase = TRTHistoryEventsCounter<IRTCarsProcess>;

public:
    virtual TString GetType() const override {
        return GetTypeName();
    }

    static TString GetTypeName() {
        return "history_events_counter";
    }

protected:
    virtual TExpectedState DoExecuteFiltered(TAtomicSharedPtr<IRTBackgroundProcessState> state, const NDrive::IServer& /*server*/, TTagsModificationContext& context) const override {
        return Process(state, context);
    }

private:
    static TBase::TFactory::TRegistrator<TRTCarHistoryEventsCounter> Registrator;

    NEntityTagsManager::EEntityType GetEntityType() const override {
        return NEntityTagsManager::EEntityType::Car;
    }

    virtual const IEntityTagsManager& GetEntityTagsManager(const NDrive::IServer& server) const override {
        return server.GetDriveAPI()->GetTagsManager().GetDeviceTags();
    }

    virtual TRTHistoryWatcherState* BuildState() const override {
        return new TRTCarHistoryEventsCounterState();
    }

    virtual bool CheckFilter(const TCarTagHistoryEvent& ev, const IRTBackgroundProcess::TExecutionContext& context) const override;
};

class TRTUserHistoryEventsCounter: public TRTHistoryEventsCounter<IRTRegularBackgroundProcess> {
public:
    virtual TString GetType() const override {
        return GetTypeName();
    }

    static TString GetTypeName() {
        return "user_history_events_counter";
    }

protected:
    virtual TExpectedState DoExecute(TAtomicSharedPtr<IRTBackgroundProcessState> state, const TExecutionContext& context) const override {
        return Process(state, context);
    }

private:
    static TFactory::TRegistrator<TRTUserHistoryEventsCounter> Registrator;

    NEntityTagsManager::EEntityType GetEntityType() const override {
        return NEntityTagsManager::EEntityType::User;
    }

    virtual const IEntityTagsManager& GetEntityTagsManager(const NDrive::IServer& server) const override {
        return server.GetDriveAPI()->GetTagsManager().GetUserTags();
    }

    virtual TRTHistoryWatcherState* BuildState() const override {
        return new TRTUserHistoryEventsCounterState();
    }
};
