#pragma once

#include <drive/backend/abstract/notifier.h>
#include <drive/backend/offers/abstract.h>

#include <drive/library/cpp/scheme/scheme.h>

#include <library/cpp/object_factory/object_factory.h>

#include <rtline/library/unistat/cache.h>
#include <rtline/util/json_processing.h>
#include <rtline/util/types/accessor.h>

namespace NDrive {
    class IServer;
}

class IMessageProviderConfig {
public:
    using TPtr = TAtomicSharedPtr<IMessageProviderConfig>;
    using TFactory = NObjectFactory::TObjectFactory<IMessageProviderConfig, TString>;

    template <typename TDerived>
    static TAtomicSharedPtr<TDerived> Construct(const NJson::TJsonValue& data) {
        TMessagesCollector errors;
        auto config = MakeAtomicShared<TDerived>();
        if (!config->DeserializeFromJson(data, errors)) {
            ERROR_LOG << "Error constructing config: " << errors.GetStringReport() << Endl;
            return nullptr;
        }
        return config;
    }

    virtual ~IMessageProviderConfig() = default;

    virtual NDrive::TScheme GetScheme(const NDrive::IServer* /* server */) const = 0;

    virtual bool DeserializeFromJson(const NJson::TJsonValue& /* config */, TMessagesCollector& /* errors */) = 0;
    virtual NJson::TJsonValue SerializeToJson() const = 0;
};

class IMessageProvider {
public:
    using TPtr = TAtomicSharedPtr<IMessageProvider>;
    using TFactory = NObjectFactory::TObjectFactory<IMessageProvider, TString>;

    using TMessage = NDrive::INotifier::TMessage;
    using TMessages = TVector<TAtomicSharedPtr<TMessage>>;
    using TNotifierContext = NDrive::INotifier::TContext;
    using TNotifierResultPtr = NDrive::INotifier::TResultPtr;

public:
    IMessageProvider() = default;
    virtual ~IMessageProvider() = default;

    const NDrive::IServer* GetServer() const {
        return Server;
    }

    virtual TString GetType() const = 0;

    static TPtr Construct(
        const TString& templateName,
        const NJson::TJsonValue& config,
        const NDrive::IServer* server,
        const TString& processName,
        const TString& robotUserId,
        TInstant startInstant,
        NEntityTagsManager::EEntityType entityType
    );

    static TSet<TString> GetRegisteredProviders();
    static NDrive::TScheme GetProvidersScheme(const NDrive::IServer* server);

    virtual TMessages Fetch(const TCarTagHistoryEvent& event, TMessagesCollector& errors) const = 0;
    virtual TString GetUniqueName(const TCarTagHistoryEvent& event) const  {
        return ToString(event.GetHistoryEventId());
    }

    virtual bool HandleResult(TNotifierResultPtr resultPtr, TMessagesCollector& errors) const;

public:
    static const TString DefaultTypeName;

protected:
    void AddSignal(const TString& fetcherTypeName, const TString& metricName, const double value = 1.0) const {
        TUnistatSignalsCache::SignalAdd(ProcessName + "-" + fetcherTypeName + "_fetcher", metricName, value);
    }

    virtual bool InitContext(const NDrive::IServer* server, const TString& processName, const TInstant& startInstant, NEntityTagsManager::EEntityType entityType, const NJson::TJsonValue& config);

private:
    const NDrive::IServer* Server = nullptr;

    R_READONLY(TString, ProcessName);
    R_READONLY(TString, RobotUserId);
    R_READONLY(TInstant, StartInstant);
    R_READONLY(NEntityTagsManager::EEntityType, EntityType);
};
