#include "default.h"

#include "constants.h"

#include <drive/backend/database/drive_api.h>

#include <rtline/library/json/parse.h>

TDefaultMessageProvider::TRegistrator TDefaultMessageProvider::Registrator;
TDefaultMessageProviderConfig::TRegistrator TDefaultMessageProviderConfig::Registrator(TDefaultMessageProvider::GetTypeName());

TReportGenerator::TReportGenerator(const NDrive::IServer& server, const TInstant actuality)
    : CarInfo(*server.GetDriveAPI()->GetCarsData(), actuality)
    , UserInfo(*server.GetDriveAPI()->GetUsersData(), actuality)
{
}

template <>
TMaybe<typename TCarsDB::TRecordType> TReportGenerator::GetObject(const TCarsDB& fetcher, const TString& id, TInstant actuality) {
    return fetcher.GetObject(id, actuality);
}

template <>
TMaybe<typename TUsersDB::TRecordType> TReportGenerator::GetObject(const TUsersDB& fetcher, const TString& id, TInstant actuality) {
    Y_UNUSED(actuality);
    return fetcher.GetCachedObject(id);
}

const TString& TReportGenerator::GetObjectReport(NEntityTagsManager::EEntityType type, const TString& objectId) const {
    switch (type) {
        case NEntityTagsManager::EEntityType::Car:
            return GetCarReport(objectId);
        case NEntityTagsManager::EEntityType::User:
            return GetUserReport(objectId);
        default:
            Y_UNREACHABLE();
    }
}

const TString& TReportGenerator::GetCarReport(const TString& carId) const {
    return CarInfo.GetReport(carId);
}

const TString& TReportGenerator::GetUserReport(const TString& userId) const {
    return UserInfo.GetReport(userId);
}

NDrive::TScheme TDefaultMessageProviderConfig::GetScheme(const NDrive::IServer* /* server */) const {
    NDrive::TScheme scheme;
    scheme.Add<TFSBoolean>("add_user_info", "Добавлять информацию о пользователе").SetDefault(true);
    scheme.Add<TFSBoolean>("add_object_info", "Добавлять информацию об объекте").SetDefault(false);
    scheme.Add<TFSBoolean>("add_comment", "Добавлять комментарий").SetDefault(false);
    return scheme;
}

bool TDefaultMessageProviderConfig::DeserializeFromJson(const NJson::TJsonValue& config, TMessagesCollector& errors) {
    return NJson::ParseField(config["add_user_info"], AddUserInfo, errors) &&
           NJson::ParseField(config["add_object_info"], AddObjectInfo, errors) &&
           NJson::ParseField(config["add_comment"], AddComment, errors);
}

NJson::TJsonValue TDefaultMessageProviderConfig::SerializeToJson() const {
    NJson::TJsonValue result;
    NJson::InsertField(result, "add_user_info", AddUserInfo);
    NJson::InsertField(result, "add_object_info", AddObjectInfo);
    NJson::InsertField(result, "add_comment", AddComment);
    return result;
}

TString TDefaultMessageProvider::GetTypeName() {
    return "default";  // IMessageProvider::DefaultTypeName
}

TString TDefaultMessageProvider::GetType() const {
    return GetTypeName();
}

bool TDefaultMessageProvider::InitContext(const NDrive::IServer* server, const TString& processName, const TInstant& startInstant, NEntityTagsManager::EEntityType entityType, const NJson::TJsonValue& config) {
    if (!TBase::InitContext(server, processName, startInstant, entityType, config)) {
        return false;
    }
    ReportGenerator = MakeHolder<TReportGenerator>(*server, startInstant);
    Config = TConfig::Construct<TConfig>(config);
    if (!Config) {
        ERROR_LOG << "Cannot construct fetcher" << Endl;
        AddSignal(GetType(), ::ToString(EFetcherSignal::InvalidConfig));
        return false;
    }
    return true;
}

TDefaultMessageProvider::TMessages TDefaultMessageProvider::Fetch(const TCarTagHistoryEvent& event, TMessagesCollector& /* errors */) const {
    TStringStream ss;
    ss << event.GetHistoryInstant().FormatLocalTime("%H:%M:%S") << ": ";
    ss << "<b>" << event->GetName() << "</b>" << ":" << event.GetHistoryAction();

    if (Config->IsAddUserInfo()) {
        const TString& performer = event->GetPerformer();
        TString userId = performer ? performer : event.GetHistoryUserId();
        ss << "\n" << ReportGenerator->GetUserReport(userId);
    }

    if (Config->IsAddObjectInfo()) {
        ss << "\n" << ReportGenerator->GetObjectReport(GetEntityType(), event.GetObjectId(event));
    }

    if (Config->IsAddComment() && event->GetComment()) {
        ss << "\n" << event->GetComment();
    }
    return { MakeAtomicShared<TMessage>(ss.Str()) };
}
