#pragma once

#include "base.h"

#include <drive/backend/cars/car.h>

#include <drive/library/cpp/scheme/scheme.h>

#include <library/cpp/object_factory/object_factory.h>

class TReportGenerator {
    template <class TEntities>
    static TMaybe<typename TEntities::TRecordType> GetObject(const TEntities& fetcher, const TString& id, TInstant actuality);

    template <class TEntities>
    class TEntitiesWithCache {
    public:
        TEntitiesWithCache(const TEntities& fetcher, const TInstant actuality)
            : Fetcher(fetcher)
            , Actuality(actuality)
        {
        }

        const TString& GetReport(const TString& id) const {
            auto iter = CachedReports.find(id);
            if (iter != CachedReports.end()) {
                return iter->second;
            }
            TString report = id;
            auto idData = TReportGenerator::GetObject<TEntities>(Fetcher, id, Actuality);
            if (idData) {
                report = idData->GetHRReport();
            }
            return CachedReports.emplace(id, std::move(report)).first->second;
        }

    private:
        const TEntities& Fetcher;
        mutable TMap<TString, TString> CachedReports;
        const TInstant Actuality;
    };

public:
    TReportGenerator(const NDrive::IServer& server, const TInstant actuality);

    const TString& GetObjectReport(NEntityTagsManager::EEntityType type, const TString& objectId) const;
    const TString& GetCarReport(const TString& carId) const;
    const TString& GetUserReport(const TString& userId) const;

private:
    TEntitiesWithCache<TCarsDB> CarInfo;
    TEntitiesWithCache<TUsersDB> UserInfo;
};

class TDefaultMessageProviderConfig: public IMessageProviderConfig {
    using TBase = IMessageProviderConfig;

public:
    virtual NDrive::TScheme GetScheme(const NDrive::IServer* server) const override;
    virtual bool DeserializeFromJson(const NJson::TJsonValue& config, TMessagesCollector& errors) override;
    virtual NJson::TJsonValue SerializeToJson() const override;

private:
    using TRegistrator = TFactory::TRegistrator<TDefaultMessageProviderConfig>;
    static TRegistrator Registrator;

    R_READONLY(bool, AddUserInfo, true);
    R_READONLY(bool, AddObjectInfo, false);
    R_READONLY(bool, AddComment, false);
};

class TDefaultMessageProvider: public IMessageProvider {
    using TBase = IMessageProvider;
    using TConfig = TDefaultMessageProviderConfig;

public:
    static TString GetTypeName();
    virtual TString GetType() const override;

    virtual TMessages Fetch(const TCarTagHistoryEvent& event, TMessagesCollector& errors) const override;

private:
    virtual bool InitContext(const NDrive::IServer* server, const TString& processName, const TInstant& startInstant, NEntityTagsManager::EEntityType entityType, const NJson::TJsonValue& config) override;


private:
    using TRegistrator = TFactory::TRegistrator<TDefaultMessageProvider>;
    static TRegistrator Registrator;

    THolder<TReportGenerator> ReportGenerator;
    R_READONLY(TAtomicSharedPtr<TConfig>, Config);
};
