#include "images_snapshot.h"

#include "constants.h"

#include <drive/backend/cars/car.h>
#include <drive/backend/compiled_riding/manager.h>
#include <drive/backend/database/drive_api.h>
#include <drive/backend/device_snapshot/snapshots/image.h>
#include <drive/backend/images/image.h>

#include <drive/library/cpp/mds/client.h>
#include <drive/library/cpp/user_events_api/client.h>

#include <rtline/util/algorithm/container.h>

TImagesSnapshotMessageProvider::TRegistrator TImagesSnapshotMessageProvider::Registrator;
TImagesSnapshotConfig::TRegistrator TImagesSnapshotConfig::Registrator(TImagesSnapshotMessageProvider::GetTypeName());

NDrive::TScheme TImagesSnapshotConfig::GetScheme(const NDrive::IServer* server) const {
    Y_UNUSED(server);
    return NDrive::TScheme();
}

NJson::TJsonValue TImagesSnapshotConfig::SerializeToJson() const {
    return NJson::JSON_MAP;
}

bool TImagesSnapshotConfig::DeserializeFromJson(const NJson::TJsonValue& config, TMessagesCollector& errors) {
    Y_UNUSED(config);
    Y_UNUSED(errors);
    return true;
}

TString TImagesSnapshotMessageProvider::GetTypeName() {
    return "images_snapshot";
}

TString TImagesSnapshotMessageProvider::GetType() const {
    return GetTypeName();
}

bool TImagesSnapshotMessageProvider::InitContext(const NDrive::IServer* server, const TString& processName, const TInstant& startInstant, NEntityTagsManager::EEntityType entityType, const NJson::TJsonValue& config) {
    if (!TBase::InitContext(server, processName, startInstant, entityType, config)) {
        return false;
    }
    Config = TConfig::Construct<TConfig>(config);
    if (!Config) {
        ERROR_LOG << "Cannot construct fetcher" << Endl;
        AddSignal(GetType(), ::ToString(EFetcherSignal::InvalidConfig));
        return false;
    }
    return true;
}

TImagesSnapshotMessageProvider::TMessages TImagesSnapshotMessageProvider::Fetch(const TCarTagHistoryEvent& event, TMessagesCollector& errors) const {
    Y_UNUSED(errors);
    auto server = GetServer();
    auto snapshot = event->GetObjectSnapshotAs<TImagesSnapshot>();
    if (!snapshot) {
        return {};
    }
    TImagesSnapshotMessageProvider::TMessages result;

    for (auto&& image : snapshot->GetImages()) {
        NJson::TJsonValue recordNode;
        recordNode.InsertValue("object_id", event.GetObjectId());
        recordNode.InsertValue("tag_id", event.GetTagId());
        recordNode.InsertValue("user_id", event.GetHistoryUserId());
        auto photoUrl = server->GetDriveAPI()->HasMDSClient() ? TCommonImageData::BuildUrl(image.Path, server->GetDriveAPI()->GetMDSClient()) : "";
        recordNode.InsertValue("photo_url", photoUrl);
        recordNode.InsertValue("origin", image.Origin);
        recordNode.InsertValue("comment", snapshot->HasComment() ? snapshot->GetCommentUnsafe() : "");

        NDrive::TNotifierMessage::TPtr message = MakeAtomicShared<TMessage>(event->GetName(), event.GetObjectId() + " " + photoUrl);
        message->SetAdditionalInfo(recordNode);
        result.push_back(message);
    }

    return result;
}
