#include "st_base.h"

#include "constants.h"

#include <drive/backend/context_fetcher/car_tags_history_event.h>
#include <drive/backend/billing/manager.h>
#include <drive/backend/data/billing_tags.h>
#include <drive/backend/data/device_tags.h>
#include <drive/backend/tags/history.h>
#include <drive/backend/tags/tags_manager.h>
#include <drive/library/cpp/raw_text/datetime.h>

#include <rtline/library/json/adapters.h>
#include <rtline/library/json/cast.h>
#include <rtline/library/json/parse.h>
#include <rtline/util/algorithm/ptr.h>

#include <util/generic/algorithm.h>
#include <util/string/cast.h>
#include <util/string/join.h>
#include <util/string/split.h>
#include <util/system/backtrace.h>

namespace NJson {
    template <>
    TJsonValue ToJson(const TConditionalStatus& object) {
        return object.SerializeToJson();
    }

    template <>
    bool TryFromJson(const TJsonValue& value, TConditionalStatus& result) {
        return result.DeserializeFromJson(value);
    }
}

bool TConditionMatcher::Match(const TRawConditions& rawConditions) const {
    auto valueToMatchPtr = rawConditions.FindPtr(GetName());
    return (!!valueToMatchPtr) ? DoMatch(*valueToMatchPtr) : (!IsRequired());
}

TEqualBooleanFieldConditionMatcher::TEqualBooleanFieldConditionMatcher(const TString& name, const bool needle)
    : TBase(name, true)
    , Needle(needle)
{
}

bool TEqualBooleanFieldConditionMatcher::DoMatch(const NJson::TJsonValue& value) const {
    return (value.IsBoolean()) ? value.GetBoolean() == GetNeedle() : false;
}

const TString TConditionalStatus::AnyStatusPlaceholder = "";

bool TConditionalStatus::Match(const TConditionMatchers& matchers) const {
    for (const auto& matcher : matchers) {
        if (!matcher->Match(Conditions)) {
            return false;
        }
    }
    return true;
}

NJson::TJsonValue TConditionalStatus::SerializeToJson() const {
    NJson::TJsonValue data(NJson::JSON_MAP);
    data["destination"] = Destination;
    for (const auto& [name, value] : Conditions) {
        data[name] = value;
    }
    return data;
}

bool TConditionalStatus::DeserializeFromJson(const NJson::TJsonValue& data) {
    if (data.IsMap()) {
        for (const auto& [name, value] : data.GetMap()) {
            if (name == "destination") {
                Destination = value.GetString();
            } else {
                Conditions[name] = value;
            }
        }
    } else if (data.IsString()) {
        Destination = data.GetString();
    } else {
        return false;
    }
    return (!!Destination);
}

bool TConditionalStatus::FindDestinationStatus(const TConditionMatchers& matchers, const TConditionStatuses& statuses, TString& destination) {
    size_t found = 0;
    for (const auto& status : statuses) {
        if (status.Match(matchers)) {
            destination = status.GetDestination();
            found++;
        }
    }
    return (1 == found);
}

NDrive::TScheme TStartrekBaseMessageConfig::GetScheme(const NDrive::IServer* /* server */) const {
    NDrive::TScheme scheme;

    scheme.Add<TFSArray>("ticket_queues", "Очереди, с которыми работает фетчер").SetElement<TFSString>();

    scheme.Add<TFSBoolean>("do_fetch_ticket_from_comment", "Искать существующий тикет в комментарии тега").SetDefault(true);

    // set default from the current value as it could be updated from the children
    scheme.Add<TFSBoolean>("do_link_issue_on_tag_add", "Добавлять информацию в тег после создания тикета (при добавлении тега)").SetDefault(LinkIssueOnTagAdd);

    scheme.Add<TFSString>("default_queue_to_make_ticket", "Очередь по умолчанию для создания новых тикетов")
          .SetTooltip("Только если в теге нет существующего\r\nПустая, если не нужно\r\nТолько при добавлении тега");
    scheme.Add<TFSText>("summary_template", "Шаблон заголовка тикета");
    scheme.Add<TFSText>("description_template", "Шаблон описания тикета")
          .SetTooltip("Доступны подстановки: " + JoinSeq(", ", ICarTagHistoryEventContextFetcher::GetRegisteredFetchers()));

    scheme.Add<TFSJson>("add_tag_status_mapping", "Правила изменения статусов при добавлении тега");
    scheme.Add<TFSJson>("remove_tag_status_mapping", "Правила изменения статусов при удалении тега");

    return scheme;
}

NJson::TJsonValue TStartrekBaseMessageConfig::SerializeToJson() const {
    NJson::TJsonValue result;
    NJson::InsertField(result, "ticket_queues", TicketQueues);
    NJson::InsertField(result, "do_fetch_ticket_from_comment", FetchTicketFromComment);
    NJson::InsertField(result, "do_link_issue_on_tag_add", LinkIssueOnTagAdd);
    NJson::InsertNonNull(result, "default_queue_to_make_ticket", DefaultQueueToMakeTicket);
    NJson::InsertField(result, "summary_template", SummaryTemplate);
    NJson::InsertField(result, "description_template", DescriptionTemplate);
    NJson::InsertField(result, "add_tag_status_mapping", NJson::Dictionary(AddTagStatusMapping));
    NJson::InsertField(result, "remove_tag_status_mapping", NJson::Dictionary(RemoveTagStatusMapping));
    return result;
}

bool TStartrekBaseMessageConfig::DeserializeFromJson(const NJson::TJsonValue& config, TMessagesCollector& errors) {
    return NJson::ParseField(config["ticket_queues"], TicketQueues, errors) &&
           NJson::ParseField(config["do_fetch_ticket_from_comment"], FetchTicketFromComment, errors) &&
           NJson::ParseField(config["do_link_issue_on_tag_add"], LinkIssueOnTagAdd, errors) &&
           NJson::ParseField(config["default_queue_to_make_ticket"], DefaultQueueToMakeTicket, errors) &&
           NJson::ParseField(config["summary_template"], SummaryTemplate, errors) &&
           NJson::ParseField(config["description_template"], DescriptionTemplate, errors) &&
           NJson::ParseField(config["add_tag_status_mapping"], NJson::Dictionary(AddTagStatusMapping), errors) &&
           NJson::ParseField(config["remove_tag_status_mapping"], NJson::Dictionary(RemoveTagStatusMapping), errors);
}

const TString IStartrekBaseMessageProvider::TEventTransitInfo::Sentinel = "$";

IStartrekBaseMessageProvider::TEventTransitInfo::TEventTransitInfo(const TString& transitId)
    : TransitId(transitId)
{
}

IStartrekBaseMessageProvider::TEventTransitInfo::TEventTransitInfo(const TCarTagHistoryEvent& event)
    : TransitId(JoinSeq(Sentinel, {::ToString(event.GetHistoryEventId()), ::ToString(event.GetHistoryAction()), event.GetTagId()}))
{
}

ui64 IStartrekBaseMessageProvider::TEventTransitInfo::GetHistoryEventId() const {
    ui64 value;
    return TryFromString(GetTransitIdPart(0), value) ? value : 0ull;
}

EObjectHistoryAction IStartrekBaseMessageProvider::TEventTransitInfo::GetHistoryAction() const {
    EObjectHistoryAction value;
    return TryFromString(GetTransitIdPart(1), value) ? value : EObjectHistoryAction::Unknown;
}

TString IStartrekBaseMessageProvider::TEventTransitInfo::GetTagId() const {
    return GetTransitIdPart(2);
}

TString IStartrekBaseMessageProvider::TEventTransitInfo::GetTransitIdPart(size_t idx) const {
    TVector<TStringBuf> parts = StringSplitter(TransitId).SplitByString(Sentinel);
    return (parts.size() > idx) ? TString{parts[idx]} : "";
}

bool IStartrekBaseMessageProvider::InitContext(const NDrive::IServer* server, const TString& processName, const TInstant& startInstant, NEntityTagsManager::EEntityType entityType, const NJson::TJsonValue& config) {
    if (!TBase::InitContext(server, processName, startInstant, entityType, config)) {
        return false;
    }
    Config = ConstructConfig(config);
    if (!Config) {
        ERROR_LOG << "Cannot construct fetcher" << Endl;
        AddSignal(::ToString(EFetcherSignal::InvalidConfig));
        return false;
    }
    return true;
}

IStartrekBaseMessageProvider::TMessages IStartrekBaseMessageProvider::Fetch(const TCarTagHistoryEvent& event, TMessagesCollector& errors) const {
    TString issue;
    if (!GetIssue(event, issue, errors, /*readOnly=*/false)) {
        return {};
    }

    TMessages messages;
    switch (event.GetHistoryAction()) {
        case EObjectHistoryAction::Add:
            messages = DoFetchAddTag(event, issue, errors);
            break;
        case EObjectHistoryAction::Remove:
            messages = DoFetchRemoveTag(event, issue, errors);
            break;
        case EObjectHistoryAction::TagEvolve:
            messages = DoFetchEvolveTag(event, issue, errors);
            break;
        case EObjectHistoryAction::AddSnapshot:
            messages = DoFetchAddSnapshot(event, issue, errors);
            break;
        case EObjectHistoryAction::UpdateData:
            messages = DoFetchUpdateData(event, issue, errors);
            break;
        default:
            break;
    }

    return messages;
}

TString IStartrekBaseMessageProvider::GetUniqueName(const TCarTagHistoryEvent& event) const {
    TString issue;
    TMessagesCollector errors;
    if (!GetIssue(event, issue, errors, /*readOnly=*/true)) {
        return TBase::GetUniqueName(event);
    }
    return issue;
}

bool IStartrekBaseMessageProvider::HandleResult(TNotifierResultPtr resultPtr, TMessagesCollector& errors) const {
    if (Config->GetLinkIssueOnTagAdd()) {
        const EObjectHistoryAction historyAction = TEventTransitInfo(resultPtr->GetTransitId()).GetHistoryAction();
        return NeedToCreateIssueOnAction(historyAction) ? UpdateTagIssueKey(resultPtr, errors) : true;
    }
    return true;
}

void IStartrekBaseMessageProvider::SetMessageTransitId(const TCarTagHistoryEvent& event, TAtomicSharedPtr<TStartrekMessage> messagePtr) const {
    messagePtr->SetTransitId(TEventTransitInfo(event).GetTransitId());
}

IStartrekBaseMessageProvider::TMessages IStartrekBaseMessageProvider::DoFetchAddTag(const TCarTagHistoryEvent& /* event */, const TString& /* issue */, TMessagesCollector& /* errors */) const {
    return {};
}

IStartrekBaseMessageProvider::TMessages IStartrekBaseMessageProvider::DoFetchEvolveTag(const TCarTagHistoryEvent& /* event */, const TString& /* issue */, TMessagesCollector& /* errors */) const {
    return {};
}

IStartrekBaseMessageProvider::TMessages IStartrekBaseMessageProvider::DoFetchRemoveTag(const TCarTagHistoryEvent& /* event */, const TString& /* issue */, TMessagesCollector& /* errors */) const {
    return {};
}

IStartrekBaseMessageProvider::TMessages IStartrekBaseMessageProvider::DoFetchAddSnapshot(const TCarTagHistoryEvent& /* event */, const TString& /* issue */, TMessagesCollector& /* errors */) const {
    return {};
}

IStartrekBaseMessageProvider::TMessages IStartrekBaseMessageProvider::DoFetchUpdateData(const TCarTagHistoryEvent& /* event */, const TString& /* issue */, TMessagesCollector& /* errors */) const {
    return {};
}

bool IStartrekBaseMessageProvider::DoFetchTicketDescription(const TCarTagHistoryEvent& event, TStartrekTicket& ticket, TMessagesCollector& errors) const {
    TCarTagHistoryEventFetchContext context(GetServer(), event);

    TString summary = ICarTagHistoryEventContextFetcher::ProcessText(Config->GetSummaryTemplate(), context, errors);
    TString description = ICarTagHistoryEventContextFetcher::ProcessText(Config->GetDescriptionTemplate(), context, errors);

    ticket.SetSummary(summary).SetDescription(description);
    return true;
}

bool IStartrekBaseMessageProvider::DoFetchAddTagComment(const TCarTagHistoryEvent& event, TString& comment, TMessagesCollector& errors) const {
    TStringBuilder commentBuilder;

    if (auto paymentInfo = GetPaymentDetails(event, errors)) {
        commentBuilder << "Запрошено списание суммы " << (double)paymentInfo->FullSum / 100 << " рублей ";
    } else {
        commentBuilder << "Добавлен тег ";
    }

    commentBuilder << "(id тега - " << event.GetTagId() << ", имя тега - \"" << event.GetData()->GetName() << "\", имя робота - \"" << GetProcessName() << "\")";
    comment = commentBuilder;
    return true;
}

bool IStartrekBaseMessageProvider::DoFetchRemoveTagComment(const TCarTagHistoryEvent& event, TString& comment, TMessagesCollector& errors) const {
    TStringBuilder commentBuilder;

    if (auto paymentInfo = GetPaymentDetails(event, errors)) {
        if (paymentInfo->Resolution == "canceled") {
            if (paymentInfo->Canceled) {
                commentBuilder << "Списание суммы " << (double)*(paymentInfo->Canceled) / 100 << "(из " << (double)paymentInfo->FullSum / 100 << ") рублей по тегу отменено ";
            } else {
                commentBuilder << "Списание суммы " << (double)paymentInfo->FullSum / 100 << " рублей по тегу отменено ";
            }
        } else if (paymentInfo->Resolution) {
            commentBuilder << "Списание суммы " << (double)paymentInfo->FullSum / 100 << " рублей по тегу успешно завершено ";
        } else {
            commentBuilder << "Удален тег ";
        }
    } else {
        commentBuilder << "Удален тег ";
    }

    commentBuilder << "(id тега - " << event.GetTagId() << ", имя тега - \"" << event.GetData()->GetName() << "\", имя робота - \"" << GetProcessName() << "\")";
    comment = commentBuilder;
    return true;
}

bool IStartrekBaseMessageProvider::DoFetchUpdateTagComment(const TCarTagHistoryEvent& event, const IStartrekBaseMessageProvider::TPaymentInfo& paymentInfo, TString& comment, TMessagesCollector& /*errors*/) const {
    TStringBuilder commentBuilder;

    if (paymentInfo.Diff) {
        commentBuilder << "Списано по тегу " << (double)*paymentInfo.Diff / 100 << Endl << "Всего " << (double)*paymentInfo.Paid / 100 << " из " << (double)paymentInfo.FullSum / 100 << Endl;
    } else {
        commentBuilder << "Изменен тег ";
    }

    commentBuilder << "(id тега - " << event.GetTagId() << ", имя тега - \"" << event.GetData()->GetName() << "\", имя робота - \"" << GetProcessName() << "\")";
    comment = commentBuilder;
    return true;
}

bool IStartrekBaseMessageProvider::NeedToCreateIssueOnAction(const EObjectHistoryAction action) const {
    return action == EObjectHistoryAction::Add;
}

bool IStartrekBaseMessageProvider::GetIssue(const TCarTagHistoryEvent& event, TString& issue, TMessagesCollector& errors, bool readOnly) const {
    const auto& queues = Config->GetTicketQueues();

    auto userProblemTag = event.GetTagAs<TUserProblemTag>();
    if (userProblemTag != nullptr) {
        auto issues = userProblemTag->GetStartrekIssues(queues);
        if (issues) {
            if (issues.size() > 1) {
                errors.AddMessage(__LOCATION__, "multiple issue keys are fetched");
                AddSignal(::ToString(EFetcherSignal::UniqueIssueTagError));
                return false;
            }
            issue = issues.front();
        }
    }

    auto deviceRepairTag = event.GetTagAs<TRepairTagRecord>();
    if (deviceRepairTag != nullptr) {
        issue = deviceRepairTag->GetTicketNumber();
    }

    if (!issue && Config->IsFetchTicketFromComment()) {
        TString comment = event.GetData()->GetComment();
        for (TStringBuf line : StringSplitter(comment).SplitBySet("\r\n").SkipEmpty()) {
            auto possibleIssue = line.substr(line.find_last_of("/") + 1);
            for (const TString& queue : queues) {
                if (possibleIssue.StartsWith(queue + "-")) {
                    issue = possibleIssue;
                    break;
                }
            }
        }
    }

    if (!issue && !readOnly && NeedToCreateIssueOnAction(event.GetHistoryAction()) && Config->GetDefaultQueueToMakeTicket()) {
        auto stClient = GetServer()->GetStartrekClient();
        if (stClient) {
            TStartrekTicket content, resultTicket;
            content
                .SetSummary("Autogenerated issue")
                .SetDescription(TStringBuilder()
                    << "process: " << GetProcessName() << Endl
                    << "tag: " << event->GetName() << Endl
                    << "tag_id: " << event.GetTagId() << Endl
                    << "object_id: " << event.GetObjectId() << Endl
                )
                .SetAdditionalValue("queue", Config->GetDefaultQueueToMakeTicket())
            ;
            NDrive::TEventLog::Log("StartrekCreateNewIssue", NJson::TMapBuilder
                ("content", content.SerializeToJson())
            );
            bool success = stClient->CreateIssue(content, resultTicket, errors);
            NDrive::TEventLog::Log("StartrekCreateNewIssueResult", NJson::TMapBuilder
                ("result", resultTicket.SerializeToJson())
                ("errors", errors.GetReport())
                ("success", success)
            );
            if (!success) {
                errors.AddMessage(__LOCATION__, "error parsing an issue");
                AddSignal(::ToString(EFetcherSignal::DummyTicketCreateError));
                return false;
            }
            if (!resultTicket.GetAdditionalValue(::ToString(TStartrekTicket::ETicketField::Key), issue)) {
                errors.AddMessage(__LOCATION__, "cannot obtain key from autogenerated issue");
                AddSignal(::ToString(EFetcherSignal::DummyTicketInvalidData));
                return false;
            }
            AddSignal(::ToString(EFetcherSignal::DummyTicketCreateSuccess));
        }
    }

    if (!issue) {
        errors.AddMessage(__LOCATION__, "no issue keys found found");
        AddSignal(::ToString(EFetcherSignal::NoIssueError));
        return false;
    }

    return true;
}

TMaybe<IStartrekBaseMessageProvider::TPaymentInfo> IStartrekBaseMessageProvider::GetPaymentDetails(const TCarTagHistoryEvent& event, TMessagesCollector& errors) const {
    auto billingTag = event.GetTagAs<TBillingTag>();
    if (!billingTag) {
        TBackTrace bt;
        bt.Capture();
        errors.AddMessage(__LOCATION__, NJson::ToJson(bt).GetStringRobust());
        errors.AddMessage(__LOCATION__, "Incorrect tag data");
        return {};
    }

    TPaymentInfo info(billingTag->GetResolution(), billingTag->GetAmount());

    if (billingTag->GetAmount() == 0) {  // deprecated; fixed since r5668367 (17 Sep 2019)
        auto fixedSumTag = event.GetTagAs<TFixedSumTag>();
        if (fixedSumTag) {
            auto tagDescription = GetServer()->GetDriveAPI()->GetTagsManager().GetTagsMeta().GetDescriptionByName(fixedSumTag->GetName());
            if (tagDescription) {
                auto fixedSumTagDescription = dynamic_cast<const TFixedBillingTagDescription*>(tagDescription.Get());
                if (fixedSumTagDescription) {
                    info.FullSum = fixedSumTagDescription->GetAmount();
                }
            }
        }
    }

    TVector<TString> sessions;
    for (const auto& op : billingTag->GetOperations()) {
        if (op.HasPaid()) {
            if (!info.Paid) {
                info.Paid = 0;
            }
            (*info.Paid) += op.GetPaidRef();
        }
        if (op.HasDiff()) {
            if (!info.Diff) {
                info.Diff = 0;
            }
            (*info.Diff) += op.GetDiffRef();
        }
        sessions.push_back(op.GetPaymentId());
    }

    if (GetServer()->GetDriveAPI()->HasBillingManager()) {
        const auto& manager = GetServer()->GetDriveAPI()->GetBillingManager();
        auto session = manager.BuildSession(false);
        TMap<TString, TCachedPayments> payments;
        if (!manager.GetPaymentsManager().GetPayments(payments, sessions, session, true)) {
            return info;
        }
        for (const auto& [_, snapshot] : payments) {
            auto timeline = snapshot.GetTimeline();
            info.Payments.insert(info.Payments.end(), timeline.begin(), timeline.end());
            if (snapshot.GetCanceledSum()) {
                if (!info.Canceled) {
                    info.Canceled = 0;
                }
                (*info.Canceled) += snapshot.GetCanceledSum();
            }
        }
    }

    return info;
}

bool IStartrekBaseMessageProvider::GetAppropriateTransition(const TStartrekTicket& ticket, const TStatusMapping& statusMapping, TString& needle, const TConditionMatchers& conditionMatchers) const {
    auto sourceStatus = ticket.GetAdditionalValue<TString>(::ToString(TStartrekTicket::ETicketField::StatusKey)).GetOrElse("");

    TString needleKey;
    if (sourceStatus && statusMapping.contains(sourceStatus)) {
        needleKey = sourceStatus;
    } else if (statusMapping.contains(TConditionalStatus::AnyStatusPlaceholder)) {
        needleKey = TConditionalStatus::AnyStatusPlaceholder;
    } else {
        return false;  // status should not be changed
    }

    TString destinationStatus;
    if (!TConditionalStatus::FindDestinationStatus(conditionMatchers, statusMapping.at(needleKey), destinationStatus)) {
        return false;
    }

    TMessagesCollector errors;
    auto issue = ticket.GetKey();
    return (!!issue) ? GetValidTransition(issue, destinationStatus, needle, errors) : false;
}

bool IStartrekBaseMessageProvider::GetValidTransition(const TString& issue, const TSet<TString>& transitions, TString& needle, TMessagesCollector& errors) const {
    TStartrekClient::TTransitions availableTransitions;
    if (!GetServer()->GetStartrekClient()->GetTransitions(issue, availableTransitions, errors)) {
        errors.AddMessage(__LOCATION__, "Error obtaining allowed transitions");
        AddSignal(::ToString(EFetcherSignal::GetTransitionsError));
        return false;
    }
    for (const auto& t : availableTransitions) {
        if (transitions.contains(t.GetName())) {
            needle = t.GetName();
            return true;
        }
    }
    AddSignal(::ToString(EFetcherSignal::NoValidTransition));
    errors.AddMessage(__LOCATION__, TStringBuilder() << "no one of the next transitions is admissible for issue " << issue << ": " << JoinSeq(", ", transitions));
    return false;
}

bool IStartrekBaseMessageProvider::GetValidTransition(const TString& issue, const TString& statusTo, TString& needle, TMessagesCollector& errors) const {
    TStartrekClient::TTransitions availableTransitions;
    if (!GetServer()->GetStartrekClient()->GetTransitions(issue, availableTransitions, errors)) {
        errors.AddMessage(__LOCATION__, "Error obtaining allowed transitions");
        AddSignal(::ToString(EFetcherSignal::GetTransitionsError));
        return false;
    }
    for (const auto& t : availableTransitions) {
        if (t.GetTargetStatusKey() == statusTo) {
            needle = t.GetName();
            return true;  // select the first one
        }
    }
    AddSignal(::ToString(EFetcherSignal::NoValidTransition));
    errors.AddMessage(__LOCATION__, TStringBuilder() << "no transitions to move issue " << issue << " to status " << statusTo);
    return false;
}

const IEntityTagsManager* IStartrekBaseMessageProvider::GetTagsManagerImpl() const {
    switch (GetEntityType()) {
    case NEntityTagsManager::EEntityType::User:
        return &GetServer()->GetDriveAPI()->GetTagsManager().GetUserTags();
    case NEntityTagsManager::EEntityType::Car:
        return &GetServer()->GetDriveAPI()->GetTagsManager().GetDeviceTags();
    default:
        return nullptr;
    }
}

bool IStartrekBaseMessageProvider::UpdateTagIssueKey(TNotifierResultPtr resultPtr, TMessagesCollector& errors) const {
    if (!resultPtr || resultPtr->HasErrors()) {
        errors.AddMessage(__LOCATION__, "Result is erroneous");
        AddSignal(::ToString(EFetcherSignal::ErroneousNotifyResult));
        return false;
    }

    TAtomicSharedPtr<TStartrekNotifierResult> nativeResultPtr = std::dynamic_pointer_cast<TStartrekNotifierResult>(resultPtr);
    if (!nativeResultPtr) {
        errors.AddMessage(__LOCATION__, "Incorrect notify result type");
        AddSignal(::ToString(EFetcherSignal::ErroneousNotifyResult));
        return false;
    }

    const TStartrekTicket& ticket = nativeResultPtr->GetTicket();
    TString ticketKey = ticket.GetKey();
    if (!ticketKey) {
        errors.AddMessage(__LOCATION__, "Invalid ticket key");
        AddSignal(::ToString(EFetcherSignal::InconsistentNotifyResultContent));
        return false;
    }

    auto transitInfo = TEventTransitInfo(nativeResultPtr->GetTransitId());
    return UpdateTagIssueKey(transitInfo.GetTagId(), ticketKey, errors);
}

bool IStartrekBaseMessageProvider::UpdateTagIssueKey(const TString& tagId, const TString& issueKey, TMessagesCollector& errors) const {
    const auto* tagsManagerImpl = GetTagsManagerImpl();
    const auto* stClient = GetServer()->GetStartrekClient();
    if (!tagsManagerImpl || !stClient) {
        errors.AddMessage(__LOCATION__, "Cannot restore tag - no tags manager / startrek client configured");
        AddSignal(::ToString(EFetcherSignal::TagRestoreError));
        return false;
    }

    IEntityTagsManager::TOptionalTag dbTag;
    {
        auto session = tagsManagerImpl->template BuildTx<NSQL::ReadOnly>();
        dbTag = tagsManagerImpl->RestoreTag(tagId, session);
        if (!dbTag || !*dbTag) {
            errors.AddMessage(__LOCATION__, "Cannot restore tag " + tagId);
            errors.MergeMessages(session.GetMessages(), "session");
            AddSignal(::ToString(EFetcherSignal::TagRestoreError));
            return false;
        }
    }

    if (auto userProblemTag = dbTag->MutableTagAs<TUserProblemTag>()) {
        auto issues = userProblemTag->GetStartrekIssues(Config->GetTicketQueues());
        if (Find(issues, issueKey) == issues.end()) {
            TUserProblemTag::TLink link;
            link.SetUri(stClient->GetTicketUri(issueKey));
            link.SetType(TUserProblemTag::ELinkType::Startrek);
            userProblemTag->MutableLinks().push_back(std::move(link));
        }
    } else if (auto dbMutableTagRecord = dbTag->MutableTagAs<TRepairTagRecord>()) {
        dbMutableTagRecord->SetTicketNumber(issueKey);
    } else {
        auto tagPtr = dbTag->GetData();
        TString comment = tagPtr->GetComment();
        if (!!comment) {
            comment += "\r\n";
        }
        comment += issueKey;
        tagPtr->SetComment(comment);
    }

    {
        auto session = tagsManagerImpl->template BuildTx<NSQL::Writable>();
        if (!tagsManagerImpl->UpdateTagData(*dbTag, GetRobotUserId(), session) || !session.Commit()) {
            errors.AddMessage(__LOCATION__, "Cannot update tag ticket info " + session.GetStringReport());
            AddSignal(::ToString(EFetcherSignal::TagUpdateError));
            return false;
        }
    }

    return true;
}
