#pragma once

#include "base.h"

#include <drive/backend/database/history/event.h>
#include <drive/backend/notifications/startrek/startrek.h>
#include <drive/backend/billing/interfaces/payments.h>

#include <drive/library/cpp/scheme/scheme.h>
#include <drive/library/cpp/startrek/client.h>

#include <library/cpp/object_factory/object_factory.h>

#include <rtline/util/json_processing.h>

#include <util/datetime/base.h>
#include <util/generic/vector.h>

class TConditionMatcher {
public:
    using TPtr = TAtomicSharedPtr<TConditionMatcher>;

    using TRawConditions = TMap<TString, NJson::TJsonValue>;

    explicit TConditionMatcher(const TString& name, const bool isRequired = false)
        : Name(name)
        , Required(isRequired)
    {
    }

    virtual ~TConditionMatcher() = default;

    virtual bool Match(const TRawConditions& rawConditions) const;

private:
    virtual bool DoMatch(const NJson::TJsonValue& value) const = 0;

private:
    R_READONLY(TString, Name);
    R_READONLY(bool, Required, false);
};

class TEqualBooleanFieldConditionMatcher: public TConditionMatcher {
    using TBase = TConditionMatcher;

public:
    TEqualBooleanFieldConditionMatcher(const TString& name, const bool needle);

private:
    virtual bool DoMatch(const NJson::TJsonValue& value) const override;

private:
    R_READONLY(bool, Needle, false);
};

class TConditionalStatus {
public:
    using TRawConditions = TConditionMatcher::TRawConditions;
    using TConditionMatchers = TVector<TConditionMatcher::TPtr>;
    using TConditionStatuses = TVector<TConditionalStatus>;

    static const TString AnyStatusPlaceholder;

    bool Match(const TConditionMatchers& matchers) const;

    NJson::TJsonValue SerializeToJson() const;
    bool DeserializeFromJson(const NJson::TJsonValue& data);

    static bool FindDestinationStatus(const TConditionMatchers& matchers, const TConditionStatuses& statuses, TString& destination);

private:
    R_READONLY(TString, Destination);
    R_READONLY(TRawConditions, Conditions, {});
};

class TStartrekBaseMessageConfig: public IMessageProviderConfig {
    using TBase = IMessageProviderConfig;

public:
    using TConditionStatuses = TVector<TConditionalStatus>;
    using TStatusMapping = TMap<TString, TConditionStatuses>;

    TStartrekBaseMessageConfig(const TVector<TString>& ticketQueues = {})
        : TBase()
        , TicketQueues(ticketQueues)
    {
    }

    virtual NDrive::TScheme GetScheme(const NDrive::IServer* server) const override;
    virtual NJson::TJsonValue SerializeToJson() const override;
    virtual bool DeserializeFromJson(const NJson::TJsonValue& config, TMessagesCollector& errors) override;

    template <class T>
    const T& GetAsSafe() const {
        return *VerifyDynamicCast<const T*>(this);
    }

private:
    R_READONLY(TVector<TString>, TicketQueues);
    R_READONLY(bool, FetchTicketFromComment, true);
    R_READONLY(bool, LinkIssueOnTagAdd, false);
    R_READONLY(TString, DefaultQueueToMakeTicket);

    R_READONLY(TString, SummaryTemplate);
    R_READONLY(TString, DescriptionTemplate);

    R_READONLY(TStatusMapping, AddTagStatusMapping, {});
    R_READONLY(TStatusMapping, RemoveTagStatusMapping, {});
};

class IStartrekBaseMessageProvider: public IMessageProvider {
    using TBase = IMessageProvider;

public:
    class TEventTransitInfo {
    public:
        explicit TEventTransitInfo(const TString& transitId);
        explicit TEventTransitInfo(const TCarTagHistoryEvent& event);

        ui64 GetHistoryEventId() const;
        EObjectHistoryAction GetHistoryAction() const;
        TString GetTagId() const;

    private:
        TString GetTransitIdPart(size_t idx) const;

    private:
        static const TString Sentinel;
        R_READONLY(TString, TransitId);
    };

    struct TPaymentInfo {
        TString Resolution;
        TMaybe<ui64> Paid;
        TMaybe<ui64> Diff;
        TMaybe<ui64> Canceled;
        ui64 FullSum = 0;
        TVector<TPaymentTask> Payments;

        TPaymentInfo(const TString& resolution, ui64 amount)
            : Resolution(resolution)
            , FullSum(amount)
        {}
    };

    using TStartrekTicket = TStartrekClient::TTicket;
    using TStartrekMessage = TStartrekNotifier::TStartrekMessage;

    using TConditionMatchers = TVector<TConditionMatcher::TPtr>;
    using TConditionStatuses = TVector<TConditionalStatus>;
    using TStatusMapping = TStartrekBaseMessageConfig::TStatusMapping;

    virtual TMessages Fetch(const TCarTagHistoryEvent& event, TMessagesCollector& errors) const override;
    virtual TString GetUniqueName(const TCarTagHistoryEvent& event) const override;
    virtual bool HandleResult(TNotifierResultPtr resultPtr, TMessagesCollector& errors) const override;

protected:
    void AddSignal(const TString& metricName, const double value = 1.0) const {
        TBase::AddSignal(GetType(), metricName, value);
    }

    virtual TAtomicSharedPtr<TStartrekBaseMessageConfig> ConstructConfig(const NJson::TJsonValue& data) const = 0;

    void SetMessageTransitId(const TCarTagHistoryEvent& event, TAtomicSharedPtr<TStartrekMessage> messagePtr) const;

    virtual TMessages DoFetchAddTag(const TCarTagHistoryEvent& event, const TString& issue, TMessagesCollector& errors) const;
    virtual TMessages DoFetchRemoveTag(const TCarTagHistoryEvent& event, const TString& issue, TMessagesCollector& errors) const;
    virtual TMessages DoFetchEvolveTag(const TCarTagHistoryEvent& event, const TString& issue, TMessagesCollector& errors) const;
    virtual TMessages DoFetchAddSnapshot(const TCarTagHistoryEvent& event, const TString& issue, TMessagesCollector& errors) const;
    virtual TMessages DoFetchUpdateData(const TCarTagHistoryEvent& event, const TString& issue, TMessagesCollector& errors) const;

    virtual bool NeedToCreateIssueOnAction(const EObjectHistoryAction action) const;

    virtual bool DoFetchTicketDescription(const TCarTagHistoryEvent& event, TStartrekTicket& ticket, TMessagesCollector& errors) const;
    virtual bool DoFetchAddTagComment(const TCarTagHistoryEvent& event, TString& comment, TMessagesCollector& errors) const;
    virtual bool DoFetchRemoveTagComment(const TCarTagHistoryEvent& event, TString& comment, TMessagesCollector& errors) const;
    virtual bool DoFetchUpdateTagComment(const TCarTagHistoryEvent& event, const TPaymentInfo& paymentInfo, TString& comment, TMessagesCollector& errors) const;

    bool GetIssue(const TCarTagHistoryEvent& event, TString& issue, TMessagesCollector& errors, bool readOnly) const;
    TMaybe<TPaymentInfo> GetPaymentDetails(const TCarTagHistoryEvent& event, TMessagesCollector& errors) const;

    bool GetAppropriateTransition(const TStartrekTicket& ticket, const TStatusMapping& statusMapping, TString& needle, const TConditionMatchers& conditionMatchers = {}) const;

    bool GetValidTransition(const TString& issue, const TSet<TString>& transitions, TString& needle, TMessagesCollector& errors) const;
    bool GetValidTransition(const TString& issue, const TString& statusTo, TString& needle, TMessagesCollector& errors) const;

    const IEntityTagsManager* GetTagsManagerImpl() const;
    bool UpdateTagIssueKey(TNotifierResultPtr resultPtr, TMessagesCollector& errors) const;
    bool UpdateTagIssueKey(const TString& tagId, const TString& issueKey, TMessagesCollector& errors) const;

private:
    virtual bool InitContext(const NDrive::IServer* server, const TString& processName, const TInstant& startInstant, NEntityTagsManager::EEntityType entityType, const NJson::TJsonValue& config) override;

private:
    R_READONLY(TAtomicSharedPtr<TStartrekBaseMessageConfig>, Config);
};
