#include "st_drive_evacuation.h"

#include "constants.h"

#include <drive/backend/database/drive_api.h>
#include <drive/backend/tags/history.h>
#include <drive/backend/tags/tags_manager.h>

#include <rtline/library/json/parse.h>
#include <rtline/util/json_processing.h>

TDriveEvacuationMessageProvider::TRegistrator TDriveEvacuationMessageProvider::DriveEvacuationRegistrator;
TDriveEvacuationConfig::TRegistrator TDriveEvacuationConfig::Registrator(TDriveEvacuationMessageProvider::GetTypeName());

const TVector<TString> TDriveEvacuationConfig::Queues = {"DRIVEEVACUATION", "SPBEVACUATION"};
const TDuration TDriveEvacuationConfig::DefaultTransactionTimeout = TDuration::Minutes(30);

NDrive::TScheme TDriveEvacuationConfig::GetScheme(const NDrive::IServer* server) const {
    auto scheme = TBase::GetScheme(server);
    scheme.Add<TFSArray>("add_tag_transitions", "Названия переходов статусов при добавлении тега").SetElement<TFSString>();
    scheme.Add<TFSArray>("remove_tag_transitions", "Названия переходов статусов при удалении тега").SetElement<TFSString>();
    scheme.Add<TFSArray>("long_transaction_tags", "Теги для удаления в случае успеха").SetElement<TFSString>();
    scheme.Add<TFSDuration>("transaction_timeout", "Макс. длительность успешной операции").SetDefault(DefaultTransactionTimeout);
    return scheme;
}

bool TDriveEvacuationConfig::DeserializeFromJson(const NJson::TJsonValue& config, TMessagesCollector& errors) {
    return NJson::ParseField(config["add_tag_transitions"], AddTagTransitions, errors) &&
           NJson::ParseField(config["remove_tag_transitions"], RemoveTagTransitions, errors) &&
           NJson::ParseField(config["long_transaction_tags"], LongTransactionTags, errors) &&
           NJson::ParseField(config["transaction_timeout"], TransactionTimeout, errors) &&
           TBase::DeserializeFromJson(config, errors);
}

NJson::TJsonValue TDriveEvacuationConfig::SerializeToJson() const {
    NJson::TJsonValue result = TBase::SerializeToJson();
    NJson::InsertField(result, "add_tag_transitions", AddTagTransitions);
    NJson::InsertField(result, "remove_tag_transitions", RemoveTagTransitions);
    NJson::InsertField(result, "long_transaction_tags", LongTransactionTags);
    NJson::InsertField(result, "transaction_timeout", TransactionTimeout);
    return result;
}

TString TDriveEvacuationMessageProvider::GetTypeName() {
    return "st_drive_evacuation";
}

TString TDriveEvacuationMessageProvider::GetType() const {
    return GetTypeName();
}

TAtomicSharedPtr<TStartrekBaseMessageConfig> TDriveEvacuationMessageProvider::ConstructConfig(const NJson::TJsonValue& data) const {
    return TConfig::Construct<TConfig>(data);
}

TDriveEvacuationMessageProvider::TMessages TDriveEvacuationMessageProvider::DoFetchAddTag(const TCarTagHistoryEvent& event, const TString& issue, TMessagesCollector& errors) const {
    TMessages messages;

    auto config = GetConfig()->GetAsSafe<TConfig>();

    TStartrekTicket ticket;
    if (!GetServer()->GetStartrekClient()->GetIssueInfo(issue, ticket, errors)) {
        AddSignal(::ToString(EFetcherSignal::GetIssueError));
        return {};
    }

    {
        auto message = MakeAtomicShared<TStartrekMessage>(issue);
        SetMessageTransitId(event, message);

        TString comment;
        if (!DoFetchAddTagComment(event, comment, errors)) {
            AddSignal(::ToString(EFetcherSignal::InvalidTicketCommentMessage));
            return {};
        }
        message->SetComment(comment);

        messages.push_back(message);
        AddSignal(::ToString(EFetcherSignal::TicketCommentMessage));
    }

    {
        // apply update and transition separately to not affect each other if smth's invalid (DRIVESUP-10118)
        TString transition;
        if (GetValidTransition(ticket.GetKey(), config.GetAddTagTransitions(), transition, errors)) {
            auto message = MakeAtomicShared<TStartrekMessage>(issue);
            SetMessageTransitId(event, message);

            message->SetTransition(transition);

            messages.push_back(message);
            AddSignal(::ToString(EFetcherSignal::TicketTransitionMessage));
        } else {
            AddSignal(::ToString(EFetcherSignal::InvalidTicketTransitionMessage));
        }
    }

    if (config.GetLongTransactionTags()) {
        auto message = MakeAtomicShared<TStartrekMessage>(issue);
        SetMessageTransitId(event, message);

        TStartrekTicket update;
        update.AddAdditionalContainerValue(::ToString(TStartrekTicket::EContainerTicketField::Tags), config.GetLongTransactionTags());
        message->SetUpdate(update);

        messages.push_back(message);
        AddSignal(::ToString(EFetcherSignal::TicketUpdateMessage));
    }

    return messages;
}

TDriveEvacuationMessageProvider::TMessages TDriveEvacuationMessageProvider::DoFetchRemoveTag(const TCarTagHistoryEvent& event, const TString& issue, TMessagesCollector& errors) const {
    TMessages messages;

    auto config = GetConfig()->GetAsSafe<TConfig>();

    TStartrekTicket ticket;
    if (!GetServer()->GetStartrekClient()->GetIssueInfo(issue, ticket, errors)) {
        AddSignal(::ToString(EFetcherSignal::GetIssueError));
        return {};
    }

    {
        auto message = MakeAtomicShared<TStartrekMessage>(issue);
        SetMessageTransitId(event, message);

        TString comment;
        if (!DoFetchRemoveTagComment(event, comment, errors)) {
            AddSignal(::ToString(EFetcherSignal::InvalidTicketCommentMessage));
            return {};
        }
        message->SetComment(comment);

        messages.push_back(message);
        AddSignal(::ToString(EFetcherSignal::TicketCommentMessage));
    }

    {
        auto paymentInfo = GetPaymentDetails(event, errors);
        if (!paymentInfo) {
            AddSignal(::ToString(EFetcherSignal::PaymentDetailsTagError));
            return {};
        }

        if (paymentInfo->Resolution == "finished") {
            {
                TStartrekTicket update;

                const double amountPaid = ticket.GetAdditionalValue<double>(::ToString(TConfig::ETicketField::AmountPaid)).GetOrElse(0.0) + paymentInfo->FullSum / 100.0;
                update.SetAdditionalValue(::ToString(TConfig::ETicketField::AmountPaid), amountPaid);

                if (config.GetLongTransactionTags()) {
                    const auto& userTagsManager = GetServer()->GetDriveAPI()->GetTagsManager().GetUserTags();
                    auto session = userTagsManager.BuildSession(true);
                    auto optionalEvents = userTagsManager.GetEventsByObject(event.GetObjectId(event), session, 0, Now() - config.GetTransactionTimeout());
                    if (optionalEvents) {
                        for (auto&& ev : *optionalEvents) {
                            if (EObjectHistoryAction::Add == ev.GetHistoryAction() && ev.GetTagId() == event.GetTagId()) {
                                update.RemoveAdditionalContainerValue(::ToString(TStartrekTicket::EContainerTicketField::Tags), config.GetLongTransactionTags());
                            }
                        }
                    }
                }

                auto message = MakeAtomicShared<TStartrekMessage>(issue);
                SetMessageTransitId(event, message);

                message->SetUpdate(update);

                messages.push_back(message);
                AddSignal(::ToString(EFetcherSignal::TicketUpdateMessage));
            }
            {
                // apply update and transition separately to not affect each other if smth's invalid (DRIVESUP-10118)
                TString transition;
                if (GetValidTransition(ticket.GetKey(), config.GetRemoveTagTransitions(), transition, errors)) {
                    auto message = MakeAtomicShared<TStartrekMessage>(issue);
                    SetMessageTransitId(event, message);

                    message->SetTransition(transition);

                    messages.push_back(message);
                    AddSignal(::ToString(EFetcherSignal::TicketTransitionMessage));
                } else {
                    AddSignal(::ToString(EFetcherSignal::InvalidTicketTransitionMessage));
                }
            }
        }
    }

    return messages;
}
