#include "st_drive_security.h"

#include "constants.h"

#include <rtline/library/json/cast.h>
#include <rtline/library/json/parse.h>

#include <util/generic/ymath.h>

TDriveSecurityMessageProvider::TRegistrator TDriveSecurityMessageProvider::DriveSecurityRegistrator;
TDriveSecurityConfig::TRegistrator TDriveSecurityConfig::Registrator(TDriveSecurityMessageProvider::GetTypeName());

const TVector<TString> TDriveSecurityConfig::Queues = {"DRIVESECURITY"};

NDrive::TScheme TDriveSecurityConfig::GetScheme(const NDrive::IServer* server) const {
    auto scheme = TBase::GetScheme(server);
    scheme.Add<TFSArray>("successful_payment_summonees", "Логины сотрудников для призыва в комментарий (при удалении тега)").SetElement<TFSString>();
    return scheme;
}

NJson::TJsonValue TDriveSecurityConfig::SerializeToJson() const {
    NJson::TJsonValue result = TBase::SerializeToJson();
    NJson::InsertField(result, "successful_payment_summonees", SuccessfulPaymentSummonees);
    return result;
}

bool TDriveSecurityConfig::DeserializeFromJson(const NJson::TJsonValue& config, TMessagesCollector& errors) {
    return NJson::ParseField(config["successful_payment_summonees"], SuccessfulPaymentSummonees, errors) &&
           TBase::DeserializeFromJson(config, errors);
}

TString TDriveSecurityMessageProvider::GetTypeName() {
    return "st_drive_security";
}

TString TDriveSecurityMessageProvider::GetType() const {
    return GetTypeName();
}

TAtomicSharedPtr<TStartrekBaseMessageConfig> TDriveSecurityMessageProvider::ConstructConfig(const NJson::TJsonValue& data) const {
    return TConfig::Construct<TConfig>(data);
}

TDriveSecurityMessageProvider::TMessages TDriveSecurityMessageProvider::DoFetchAddTag(const TCarTagHistoryEvent& event, const TString& issue, TMessagesCollector& errors) const {
    TMessages messages;

    auto config = GetConfig()->GetAsSafe<TConfig>();

    {
        auto message = MakeAtomicShared<TStartrekMessage>(issue);
        SetMessageTransitId(event, message);

        TString comment;
        if (!DoFetchAddTagComment(event, comment, errors)) {
            AddSignal(::ToString(EFetcherSignal::InvalidTicketCommentMessage));
            return {};
        }
        message->SetComment(comment);

        messages.push_back(message);
        AddSignal(::ToString(EFetcherSignal::TicketCommentMessage));
    }

    {
        auto message = MakeAtomicShared<TStartrekMessage>(issue);
        SetMessageTransitId(event, message);

        TStartrekTicket currentTicket;
        if (!GetServer()->GetStartrekClient()->GetIssueInfo(issue, currentTicket, errors)) {
            AddSignal(::ToString(EFetcherSignal::GetIssueError));
            return {};
        }

        TString transition;
        if (GetAppropriateTransition(currentTicket, config.GetAddTagStatusMapping(), transition)) {
            message->SetTransition(transition);
            messages.push_back(message);
            AddSignal(::ToString(EFetcherSignal::TicketTransitionMessage));
        }
    }

    return messages;
}

TDriveSecurityMessageProvider::TMessages TDriveSecurityMessageProvider::DoFetchRemoveTag(const TCarTagHistoryEvent& event, const TString& issue, TMessagesCollector& errors) const {
    TMessages messages;

    auto config = GetConfig()->GetAsSafe<TConfig>();

    {
        auto message = MakeAtomicShared<TStartrekMessage>(issue);
        SetMessageTransitId(event, message);

        TString commentText;
        if (!DoFetchRemoveTagComment(event, commentText, errors)) {
            AddSignal(::ToString(EFetcherSignal::InvalidTicketCommentMessage));
            return {};
        }
        message->SetComment(TStartrekClient::TComment(commentText, config.GetSuccessfulPaymentSummonees()));

        messages.push_back(message);
        AddSignal(::ToString(EFetcherSignal::TicketCommentMessage));
    }

    {
        auto paymentInfo = GetPaymentDetails(event, errors);
        if (!paymentInfo) {
            AddSignal(::ToString(EFetcherSignal::PaymentDetailsTagError));
            return {};
        }

        if (paymentInfo->Resolution == "finished") {
            TStartrekTicket currentTicket;
            if (!GetServer()->GetStartrekClient()->GetIssueInfo(issue, currentTicket, errors)) {
                AddSignal(::ToString(EFetcherSignal::GetIssueError));
                return {};
            }

            const double totalPaid = currentTicket.GetAdditionalValue<double>(::ToString(TConfig::ETicketField::TotalPaid)).GetOrElse(0.0);
            const double totalToPay = currentTicket.GetAdditionalValue<double>(::ToString(TConfig::ETicketField::TotalToPay)).GetOrElse(0.0);

            {
                // apply update and transition separately to not affect each other if smth's invalid (DRIVESUP-10118)

                TConditionMatchers conditionMatchers = { MakeAtomicShared<TEqualBooleanFieldConditionMatcher>("all_paid", FuzzyEquals(1 + totalToPay, 1 + totalPaid)) };

                TString transition;
                if (GetAppropriateTransition(currentTicket, config.GetRemoveTagStatusMapping(), transition, conditionMatchers)) {
                    auto message = MakeAtomicShared<TStartrekMessage>(issue);
                    SetMessageTransitId(event, message);

                    message->SetTransition(transition);

                    messages.push_back(message);
                    AddSignal(::ToString(EFetcherSignal::TicketTransitionMessage));
                }
            }
        }
    }

    return messages;
}

TDriveSecurityMessageProvider::TMessages TDriveSecurityMessageProvider::DoFetchUpdateData(const TCarTagHistoryEvent& event, const TString& issue, TMessagesCollector& errors) const {
    TMessages messages;

    auto config = GetConfig()->GetAsSafe<TConfig>();
    auto paymentInfo = GetPaymentDetails(event, errors);
    if (!paymentInfo) {
        AddSignal(::ToString(EFetcherSignal::PaymentDetailsTagError));
        return {};
    }
    if (paymentInfo->Diff) {
        {
            auto message = MakeAtomicShared<TStartrekMessage>(issue);
            SetMessageTransitId(event, message);

            TString commentText;
            if (!DoFetchUpdateTagComment(event, *paymentInfo, commentText, errors)) {
                AddSignal(::ToString(EFetcherSignal::InvalidTicketCommentMessage));
                return {};
            }
            message->SetComment(TStartrekClient::TComment(commentText, config.GetSuccessfulPaymentSummonees()));

            messages.push_back(message);
            AddSignal(::ToString(EFetcherSignal::TicketCommentMessage));
        }

        {
            TStartrekTicket currentTicket;
            if (!GetServer()->GetStartrekClient()->GetIssueInfo(issue, currentTicket, errors)) {
                AddSignal(::ToString(EFetcherSignal::GetIssueError));
                return {};
            }

            const double totalPaid = currentTicket.GetAdditionalValue<double>(::ToString(TConfig::ETicketField::TotalPaid)).GetOrElse(0.0) + (double)*(paymentInfo->Diff) / 100;

            {
                auto message = MakeAtomicShared<TStartrekMessage>(issue);
                SetMessageTransitId(event, message);

                TStartrekTicket update;
                update.SetAdditionalValue(::ToString(TConfig::ETicketField::TotalPaid), totalPaid);
                message->SetUpdate(update);

                messages.push_back(message);
                AddSignal(::ToString(EFetcherSignal::TicketUpdateMessage));
            }
        }
    }

    return messages;
}
