#include "st_drive_tech_generic.h"

#include "constants.h"

#include <drive/backend/context_fetcher/car_tags_history_event.h>
#include <drive/backend/database/drive_api.h>
#include <drive/backend/tags/history.h>
#include <drive/backend/tags/tags_manager.h>

#include <rtline/library/json/cast.h>
#include <rtline/util/json_processing.h>

TStartrekDriveTechGenericMessageProvider::TRegistrator TStartrekDriveTechGenericMessageProvider::StartrekDriveTechGenericRegistrator;
TStartrekDriveTechGenericMessageConfig::TRegistrator TStartrekDriveTechGenericMessageConfig::Registrator(TStartrekDriveTechGenericMessageProvider::GetTypeName());

TStartrekDriveTechGenericMessageConfig::TStartrekDriveTechGenericMessageConfig()
{
    SetProtectedLinkIssueOnTagAdd(true);
}

NDrive::TScheme TStartrekDriveTechGenericMessageConfig::GetScheme(const NDrive::IServer* server) const {
    auto scheme = TBase::GetScheme(server);
    const TString availablePlaceholders = JoinSeq(", ", ICarTagHistoryEventContextFetcher::GetRegisteredFetchers());
    scheme.Add<TFSText>("additional_attributes_template", "Шаблон json для обновления тикета (при добавлении тега)")
          .SetTooltip("Доступны подстановки: " + availablePlaceholders);
    scheme.Add<TFSText>("comment_template", "Шаблон комментария (при добавлении тега)")
          .SetTooltip("Доступны подстановки: " + availablePlaceholders);
    scheme.Add<TFSText>("remove_tag_update_template", "Шаблон json для обновления тикета при удалении тега")
          .SetTooltip("Доступны подстановки: " + availablePlaceholders);
    return scheme;
}

NJson::TJsonValue TStartrekDriveTechGenericMessageConfig::SerializeToJson() const {
    NJson::TJsonValue result = TBase::SerializeToJson();
    NJson::InsertField(result, "additional_attributes_template", AdditionalAttributesTemplate);
    NJson::InsertField(result, "comment_template", CommentTemplate);
    NJson::InsertField(result, "remove_tag_update_template", RemoveTagUpdateTemplate);
    return result;
}

bool TStartrekDriveTechGenericMessageConfig::DeserializeFromJson(const NJson::TJsonValue& config, TMessagesCollector& errors) {
    return NJson::ParseField(config["additional_attributes_template"], AdditionalAttributesTemplate, errors) &&
           NJson::ParseField(config["comment_template"], CommentTemplate, errors) &&
           NJson::ParseField(config["remove_tag_update_template"], RemoveTagUpdateTemplate, errors) &&
           TBase::DeserializeFromJson(config, errors);
}

TString TStartrekDriveTechGenericMessageProvider::GetTypeName() {
    return "st_drive_tech_generic";
}

TString TStartrekDriveTechGenericMessageProvider::GetType() const {
    return GetTypeName();
}

bool TStartrekDriveTechGenericMessageProvider::NeedToCreateIssueOnAction(const EObjectHistoryAction action) const {
    return TBase::NeedToCreateIssueOnAction(action) || action == EObjectHistoryAction::TagEvolve;
}

TAtomicSharedPtr<TStartrekBaseMessageConfig> TStartrekDriveTechGenericMessageProvider::ConstructConfig(const NJson::TJsonValue& data) const {
    return TConfig::Construct<TConfig>(data);
}

TStartrekDriveTechGenericMessageProvider::TMessages TStartrekDriveTechGenericMessageProvider::DoFetchEvolveTag(const TCarTagHistoryEvent& event, const TString& issue, TMessagesCollector& errors) const {
    return DoFetchAddTag(event, issue, errors);
}

TStartrekDriveTechGenericMessageProvider::TMessages TStartrekDriveTechGenericMessageProvider::DoFetchAddTag(const TCarTagHistoryEvent& event, const TString& issue, TMessagesCollector& errors) const {
    TMessages messages;

    auto config = GetConfig()->GetAsSafe<TConfig>();

    {
        auto message = MakeAtomicShared<TStartrekMessage>(issue);
        SetMessageTransitId(event, message);

        TStartrekTicket update;

        {
            TCarTagHistoryEventFetchContext context(GetServer(), event);

            TString summary = ICarTagHistoryEventContextFetcher::ProcessText(config.GetSummaryTemplate(), context, errors);
            update.SetSummary(summary);

            TString description = ICarTagHistoryEventContextFetcher::ProcessText(config.GetDescriptionTemplate(), context, errors);
            update.SetDescription(description);
        }

        {
            TCarTagHistoryEventFetchContext context(GetServer(), event, "<", ">");  // change placeholder markers due to json content

            TString additionalAttributes = ICarTagHistoryEventContextFetcher::ProcessText(config.GetAdditionalAttributesTemplate(), context, errors);

            NJson::TJsonValue parsedAdditionalAttributes;
            if (!NJson::ReadJsonTree(additionalAttributes, &parsedAdditionalAttributes)) {
                errors.AddMessage(__LOCATION__, "Cannot parse additional attributes template");
                errors.AddMessage("AdditionalAttributes", additionalAttributes);
                AddSignal(::ToString(EFetcherSignal::InvalidAdditionalAttributesTemplate));
                return {};
            }

            for (auto&& [key, value] : parsedAdditionalAttributes.GetMap()) {
                update.SetAdditionalValue(key, value);
            }
        }

        message->SetUpdate(update);

        messages.push_back(message);
        AddSignal(::ToString(EFetcherSignal::TicketUpdateMessage));
    }

    if (config.GetCommentTemplate()) {
        auto message = MakeAtomicShared<TStartrekMessage>(issue);
        SetMessageTransitId(event, message);

        TCarTagHistoryEventFetchContext context(GetServer(), event);

        TString comment = ICarTagHistoryEventContextFetcher::ProcessText(config.GetCommentTemplate(), context, errors);
        message->SetComment(comment);

        messages.push_back(message);
        AddSignal(::ToString(EFetcherSignal::TicketCommentMessage));
    }

    return messages;
}

TStartrekDriveTechGenericMessageProvider::TMessages TStartrekDriveTechGenericMessageProvider::DoFetchRemoveTag(const TCarTagHistoryEvent& event, const TString& issue, TMessagesCollector& errors) const {
    TMessages messages;

    auto config = GetConfig()->GetAsSafe<TConfig>();

    {
        auto message = MakeAtomicShared<TStartrekMessage>(issue);
        SetMessageTransitId(event, message);

        TString comment;
        if (!DoFetchRemoveTagComment(event, comment, errors)) {
            AddSignal(::ToString(EFetcherSignal::InvalidTicketCommentMessage));
            return {};
        }
        message->SetComment(comment);

        messages.push_back(message);
        AddSignal(::ToString(EFetcherSignal::TicketCommentMessage));
    }

    if (config.GetRemoveTagUpdateTemplate()) {
        auto message = MakeAtomicShared<TStartrekMessage>(issue);
        TCarTagHistoryEventFetchContext context(GetServer(), event, "<", ">");  // change placeholder markers due to json content

        TString additionalAttributes = ICarTagHistoryEventContextFetcher::ProcessText(config.GetRemoveTagUpdateTemplate(), context, errors);

        NJson::TJsonValue parsedAdditionalAttributes;
        if (!NJson::ReadJsonTree(additionalAttributes, &parsedAdditionalAttributes)) {
            errors.AddMessage(__LOCATION__, "Cannot parse additional attributes template");
            AddSignal(::ToString(EFetcherSignal::InvalidAdditionalAttributesTemplate));
            return {};
        }

        TStartrekTicket update;
        for (auto&& [key, value] : parsedAdditionalAttributes.GetMap()) {
            update.SetAdditionalValue(key, value);
        }
        message->SetUpdate(update);

        messages.push_back(message);
        AddSignal(::ToString(EFetcherSignal::TicketUpdateMessage));
    }

    {
        TStartrekTicket currentTicket;
        if (!GetServer()->GetStartrekClient()->GetIssueInfo(issue, currentTicket, errors)) {
            AddSignal(::ToString(EFetcherSignal::GetIssueError));
            return {};
        }

        TString transition;
        if (GetAppropriateTransition(currentTicket, config.GetRemoveTagStatusMapping(), transition, {})) {
            auto message = MakeAtomicShared<TStartrekMessage>(issue);
            SetMessageTransitId(event, message);

            message->SetTransition(transition);

            messages.push_back(message);
            AddSignal(::ToString(EFetcherSignal::TicketTransitionMessage));
        } else {
            auto status = currentTicket.GetAdditionalValue<TString>(::ToString(TStartrekTicket::ETicketField::StatusKey)).GetOrElse("unknown");
            auto key = currentTicket.GetAdditionalValue<TString>(::ToString(TStartrekTicket::ETicketField::Key)).GetOrElse("unknown");
            errors.AddMessage(__LOCATION__, TStringBuilder() << "Transition to status " << status << " is not found for ticket " << key);
            AddSignal(::ToString(EFetcherSignal::InvalidTicketTransitionMessage));
        }
    }

    return messages;
}
