#include "st_upload_attachments.h"

#include "constants.h"

#include <drive/backend/context_fetcher/car_tags_history_event.h>
#include <drive/backend/database/drive_api.h>
#include <drive/backend/device_snapshot/snapshots/image.h>
#include <drive/backend/notifications/startrek/startrek.h>

#include <drive/library/cpp/mds/client.h>
#include <drive/library/cpp/scheme/scheme.h>
#include <drive/library/cpp/startrek/client.h>
#include <drive/library/cpp/startrek/entity.h>

#include <rtline/library/json/parse.h>

#include <util/string/builder.h>
#include <util/string/split.h>

TStartrekUploadAttachmentsMessageProvider::TRegistrator TStartrekUploadAttachmentsMessageProvider::Registrator;
TStartrekUploadAttachmentsConfig::TRegistrator TStartrekUploadAttachmentsConfig::Registrator(TStartrekUploadAttachmentsMessageProvider::GetTypeName());

const TString TStartrekUploadAttachmentsConfig::DefaultBucketName = "carsharing-tags";
const TString TStartrekUploadAttachmentsConfig::DefaultCommentTemplate =
    "Добавлен(-ы) {attachments_count} файл(-а/-ов) (id тега - {tag.id}, имя тега - \"{tag.name}\", имя робота - \"{process_name}\")";
const TString TStartrekUploadAttachmentsConfig::DefaultUploadErrorCommentTemplate =
    "Не получилось автоматически выгрузить фотографии из хранилища ({attachment_name}). Сохраните фото со (страницы тегов по машине)[{car.url}] и прикрепите в этот тикет";

NDrive::TScheme TStartrekUploadAttachmentsConfig::GetScheme(const NDrive::IServer* server) const {
    auto scheme = TBase::GetScheme(server);

    scheme.Add<TFSString>("bucket_name", "Имя бакета S3, где хранятся фото").SetDefault(DefaultBucketName);

    const TString availablePlaceholders = JoinSeq(", ", ICarTagHistoryEventContextFetcher::GetRegisteredFetchers());
    scheme.Add<TFSText>("comment_template", "Шаблон комментария при добавлении фото (в общий список, если пустой)")
          .SetDefault(DefaultCommentTemplate)
          .SetTooltip("Доступны подстановки: attachments_count, process_name, " + availablePlaceholders);
    scheme.Add<TFSText>("upload_error_comment_template", "Шаблон комментария при неуспехе добавления фото")
          .SetDefault(DefaultUploadErrorCommentTemplate)
          .SetTooltip("Доступны подстановки: attachment_name, process_name, " + availablePlaceholders);

    return scheme;
}

NJson::TJsonValue TStartrekUploadAttachmentsConfig::SerializeToJson() const {
    NJson::TJsonValue result = TBase::SerializeToJson();
    NJson::InsertField(result, "bucket_name", BucketName);
    NJson::InsertField(result, "comment_template", CommentTemplate);
    NJson::InsertField(result, "upload_error_comment_template", UploadErrorCommentTemplate);
    return result;
}

bool TStartrekUploadAttachmentsConfig::DeserializeFromJson(const NJson::TJsonValue& config, TMessagesCollector& errors) {
    return NJson::ParseField(config["bucket_name"], BucketName) &&
           NJson::ParseField(config["comment_template"], CommentTemplate) &&
           NJson::ParseField(config["upload_error_comment_template"], UploadErrorCommentTemplate) &&
           TBase::DeserializeFromJson(config, errors);
}

TString TStartrekUploadAttachmentsMessageProvider::GetTypeName() {
    return "st_upload_attachments";
}

TString TStartrekUploadAttachmentsMessageProvider::GetType() const {
    return GetTypeName();
}

TAtomicSharedPtr<TStartrekBaseMessageConfig> TStartrekUploadAttachmentsMessageProvider::ConstructConfig(const NJson::TJsonValue& data) const {
    return TConfig::Construct<TConfig>(data);
}

TStartrekUploadAttachmentsMessageProvider::TMessages TStartrekUploadAttachmentsMessageProvider::DoFetchAddSnapshot(const TCarTagHistoryEvent& event, const TString& issue, TMessagesCollector& errors) const {
    TMessages messages;

    auto generalMessage = MakeAtomicShared<TStartrekMessage>(issue);
    SetMessageTransitId(event, generalMessage);

    auto config = GetConfig()->GetAsSafe<TConfig>();

    TCarTagHistoryEventFetchContext context(GetServer(), event);
    context.SetDynamicContext({
        { "process_name", GetProcessName() },
    });

    const TS3Client::TBucket* bucket =
          GetServer() && GetServer()->GetDriveAPI() && GetServer()->GetDriveAPI()->HasMDSClient()
        ? GetServer()->GetDriveAPI()->GetMDSClient().GetBucket(config.GetBucketName())
        : nullptr;

    auto snapshot = event->GetObjectSnapshotAs<TImagesSnapshot>();
    if (snapshot) {
        for (const TImagesSnapshot::TImage& image: snapshot->GetImages()) {
            TVector<TStringBuf> pathParts = StringSplitter(image.Path).Split('/');
            TString name = TString{pathParts.back()};

            TString content;
            if (!bucket || bucket->GetFile(image.Path, content, errors) / 100 != 2) {  // sync
                if (!bucket) {
                    errors.AddMessage("DoFetchAddSnapshot", "fail to get bucket '" + config.GetBucketName() + "'");
                }
                auto errorMessage = MakeAtomicShared<TStartrekMessage>(issue);
                SetMessageTransitId(event, errorMessage);

                context.MutableDynamicContext().emplace("attachment_name", name);

                TString comment = ICarTagHistoryEventContextFetcher::ProcessText(config.GetUploadErrorCommentTemplate(), context, errors);
                errorMessage->SetComment(comment);

                context.MutableDynamicContext().erase("attachment_name");

                AddSignal(::ToString(EFetcherSignal::InvalidTicketUploadAttachmentsMessage));
                messages.push_back(errorMessage);
            } else {
                generalMessage->MutableAttachments().emplace(name, content);
            }
        }
    }

    if (generalMessage->GetAttachments()) {
        context.MutableDynamicContext().emplace("attachments_count", ::ToString(generalMessage->GetAttachments().size()));
        TString comment = ICarTagHistoryEventContextFetcher::ProcessText(config.GetCommentTemplate(), context, errors);
        generalMessage->SetComment(comment);

        AddSignal(::ToString(EFetcherSignal::TicketUploadAttachmentsMessage));
        messages.push_back(generalMessage);
    }

    return messages;
}
