#include "user_info.h"

#include "constants.h"

#include <drive/backend/cars/car.h>
#include <drive/backend/compiled_riding/manager.h>
#include <drive/backend/database/drive_api.h>

#include <drive/library/cpp/user_events_api/client.h>

#include <rtline/util/algorithm/container.h>

TUserInfoMessageProvider::TRegistrator TUserInfoMessageProvider::Registrator;
TUserInfoConfig::TRegistrator TUserInfoConfig::Registrator(TUserInfoMessageProvider::GetTypeName());

NDrive::TScheme TUserInfoConfig::GetScheme(const NDrive::IServer* /* server */) const {
    NDrive::TScheme scheme;
    scheme.Add<TFSArray>("area_ids", "Фильтр area id").SetElement<TFSString>();
    scheme.Add<TFSArray>("device_ids", "Фильтр device id").SetElement<TFSString>();
    scheme.Add<TFSString>("user_agent_pattern", "Фильтр user agent (regular expression)");
    scheme.Add<TFSString>("max_sessions_count", "Макс. количество ненулевых поездок");
    scheme.Add<TFSBoolean>("fresh_accounts", "Только новые пользователи (первая поездка)").SetDefault(false);
    return scheme;
}

NJson::TJsonValue TUserInfoConfig::SerializeToJson() const {
    NJson::TJsonValue result;
    NJson::InsertField(result, "user_agent_pattern", UserAgentPattern);
    NJson::InsertNonNull(result, "max_sessions_count", MaxSessionsCount);
    NJson::InsertField(result, "fresh_accounts", FreshAccounts);
    NJson::InsertField(result, "device_ids", DeviceIds);
    NJson::InsertField(result, "area_ids", AreaIds);
    return result;
}

bool TUserInfoConfig::DeserializeFromJson(const NJson::TJsonValue& config, TMessagesCollector& errors) {
    return NJson::ParseField(config["user_agent_pattern"], UserAgentPattern, errors) &&
           NJson::ParseField(config["max_sessions_count"], MaxSessionsCount, errors) &&
           NJson::ParseField(config["fresh_accounts"], FreshAccounts, errors) &&
           NJson::ParseField(config["device_ids"], DeviceIds, errors) &&
           NJson::ParseField(config["area_ids"], AreaIds, errors);
}

TString TUserInfoMessageProvider::GetTypeName() {
    return "user_info";
}

TString TUserInfoMessageProvider::GetType() const {
    return GetTypeName();
}

bool TUserInfoMessageProvider::InitContext(const NDrive::IServer* server, const TString& processName, const TInstant& startInstant, NEntityTagsManager::EEntityType entityType, const NJson::TJsonValue& config) {
    if (!TBase::InitContext(server, processName, startInstant, entityType, config)) {
        return false;
    }

    Config = TConfig::Construct<TConfig>(config);
    if (!Config) {
        ERROR_LOG << "Cannot construct fetcher" << Endl;
        AddSignal(GetType(), ::ToString(EFetcherSignal::InvalidConfig));
        return false;
    }

    if (Config->GetUserAgentPattern()) try {
        UserAgentRegEx.ConstructInPlace(Config->GetUserAgentPattern());
    } catch (const std::exception& e) {
        ERROR_LOG << "Cannot compile UserAgentPattern " << Config->GetUserAgentPattern() << ": " << FormatExc(e) << Endl;
        AddSignal(GetType(), ::ToString(EFetcherSignal::InvalidConfig));
        return false;
    }

    return true;
}

TUserInfoMessageProvider::TMessages TUserInfoMessageProvider::Fetch(const TCarTagHistoryEvent& event, TMessagesCollector& errors) const {
    Y_UNUSED(errors);
    auto server = GetServer();
    auto driveApi = Yensured(server)->GetDriveAPI();
    auto userEventsApi = Yensured(server)->GetUserEventsApi();

    auto asyncRealtimeUserInfo = Yensured(userEventsApi)->GetRealtimeUserInfo(event.GetHistoryUserId(), true, true);
    auto realtimeUserInfo = asyncRealtimeUserInfo.ExtractValueSync();

    NDrive::TAreaIds matchedAreaIds;
    if (!Config->GetAreaIds().empty()) {
        auto areaIds = Yensured(driveApi)->GetAreaIds(realtimeUserInfo.GetLocation());
        if (IsIntersectionEmpty(Config->GetAreaIds(), areaIds)) {
            return {};
        }
        for (auto&& areaId : Config->GetAreaIds()) {
            if (areaIds.contains(areaId)) {
                matchedAreaIds.insert(areaId);
            }
        }
    }

    if (!Config->GetDeviceIds().empty()) {
        const auto& deviceId = realtimeUserInfo.GetDeviceId();
        if (!deviceId.empty() && !Config->GetDeviceIds().contains(deviceId)) {
            return {};
        }
    }

    if (UserAgentRegEx) {
        const auto& userAgent = realtimeUserInfo.GetUserAgent();
        if (!userAgent.empty() && !UserAgentRegEx->Match(userAgent.c_str())) {
            return {};
        }
    }

    TMaybe<ui64> rides;
    if (Config->HasMaxSessionsCount()) {
        const auto& compiledSessionManager = driveApi->GetMinimalCompiledRides();
        auto tx = compiledSessionManager.BuildSession(true);
        auto ydbTx = GetServer()->GetDriveAPI()->BuildYdbTx<NSQL::ReadOnly>("user_info_message_provider", server);
        auto optionalCompiledSessions = compiledSessionManager.GetUser<TMinimalCompiledRiding>(event.GetHistoryUserId(), tx, ydbTx);
        if (optionalCompiledSessions) {
            for (auto&& ev : *optionalCompiledSessions) {
                if (ev.GetSumPrice() > 0) {
                    rides = rides.GetOrElse(0) + 1;
                }
            }
        } else {
            rides = 0;
        }
        if (rides.GetOrElse(0) > Config->GetMaxSessionsCountRef()) {
            return {};
        }
    }

    TStringBuilder message;
    auto car = Yensured(Yensured(driveApi)->GetCarsData())->GetObject(event.GetObjectId());
    auto user = Yensured(Yensured(driveApi)->GetUsersData())->GetCachedObject(event.GetHistoryUserId());
    if (Config->GetFreshAccounts()) {
        if (user && !user->IsFirstRiding()) {
            return {};
        }
    }
    if (car) {
        message << "Car: " << car->GetHRReport() << Endl;
    }
    if (user) {
        message << "User: " << user->GetHRReport() << Endl;
    }
    {
        message << "When: " << event.GetHistoryInstant().ToIsoStringLocal() << Endl;
        message << "DeviceId: " << realtimeUserInfo.GetDeviceId() << Endl;
        message << "UserAgent: " << realtimeUserInfo.GetUserAgent() << Endl;
        message << "User location: " << realtimeUserInfo.GetLocation().Y << ' ' << realtimeUserInfo.GetLocation().X << Endl;
    }
    if (!matchedAreaIds.empty()) {
        message << "User areas: " << JoinStrings(matchedAreaIds.begin(), matchedAreaIds.end(), ",") << Endl;
    }
    if (rides) {
        message << "User rides count: " << rides << Endl;
    }
    return { MakeAtomicShared<TMessage>(message) };
}
