#include "process.h"

#include <drive/backend/rt_background/manager/state.h>

#include <drive/backend/database/drive_api.h>

#include <rtline/library/unistat/cache.h>

IRTRegularBackgroundProcess::TFactory::TRegistrator<TRTHistoryReplicator> TRTHistoryReplicator::Registrator(TRTHistoryReplicator::GetTypeName());
TRTHistoryReplicatorState::TFactory::TRegistrator<TRTHistoryReplicatorState> TRTHistoryReplicatorState::Registrator(TRTHistoryReplicator::GetTypeName());

TString TRTHistoryReplicatorState::GetType() const {
    return TRTHistoryReplicator::GetTypeName();
}

NDrive::TScheme TRTHistoryReplicator::DoGetScheme(const IServerBase& server) const {
    NDrive::TScheme scheme = TBase::DoGetScheme(server);
    scheme.Add<TFSVariants>("destination_db", "В какую базу копировать").SetVariants(server.GetDatabaseNames()).SetRequired(true);
    scheme.Add<TFSString>("destination_history_table", "В какую таблицу копировать историю").SetRequired(true);
    scheme.Add<TFSString>("source_entity_table", "Из какой таблицы копировать данные сущности");
    scheme.Add<TFSString>("entity_id_column", "Столбец с уникальным id сущности");
    scheme.Add<TFSString>("destination_entity_table", "В какую таблицу копировать данные сущности");
    scheme.Add<TFSString>("history_action_column", "Столбец действия истории").SetDefault("history_action");
    scheme.Add<TFSVariants>("remove_history_actions").InitVariants<EObjectHistoryAction>().SetMultiSelect(true).SetDefault(ToString(EObjectHistoryAction::Remove));
    return scheme;
}

bool TRTHistoryReplicator::DoDeserializeFromJson(const NJson::TJsonValue& jsonInfo) {
    return
        NJson::ParseField(jsonInfo, "destination_db", DestinationDBName, true) &&
        NJson::ParseField(jsonInfo, "destination_history_table", DestinationHistoryTableName, true) &&
        NJson::ParseField(jsonInfo, "destination_entity_table", DestinationEntityTableName) &&
        NJson::ParseField(jsonInfo, "source_entity_table", SourceEntityTableName) &&
        NJson::ParseField(jsonInfo, "entity_id_column", EntityIdColumn) &&
        NJson::ParseField(jsonInfo, "history_action_column", HistoryActionColumnName) &&
        NJson::ParseField(jsonInfo, "remove_history_actions", RemoveHistoryActions) &&
        TBase::DoDeserializeFromJson(jsonInfo);
}

NJson::TJsonValue TRTHistoryReplicator::DoSerializeToJson() const {
    NJson::TJsonValue result = TBase::DoSerializeToJson();
    NJson::InsertField(result, "destination_db", DestinationDBName);
    NJson::InsertField(result, "destination_history_table", DestinationHistoryTableName);
    NJson::InsertField(result, "destination_entity_table", DestinationEntityTableName);
    NJson::InsertField(result, "source_entity_table", SourceEntityTableName);
    NJson::InsertField(result, "entity_id_column", EntityIdColumn);
    NJson::InsertField(result, "history_action_column", HistoryActionColumnName);
    NJson::InsertField(result, "remove_history_actions", RemoveHistoryActions);
    return result;
}

TString GenerateInCondition(const TString& column, const TSet<TString> values) {
    TStringStream condition;
    condition << column << " IN ('" << JoinStrings(values.begin(), values.end(), "', '") << "')";
    return condition.Str();
}

bool TRTHistoryReplicator::ProcessRecords(const TRecordsSet& records, const NDrive::IServer& server, ui64& lastEventId, TMessagesCollector& errors) const {
    if (records.GetRecords().empty()) {
        return true;
    }

    auto sourceTable = GetTable();
    if (!sourceTable) {
        errors.AddMessage("TRTHistoryReplicator", "cannot get SequentialTable " + GetTableName());
        return false;
    }
    auto sourceDatabase = sourceTable->GetDatabasePtr();

    NStorage::IDatabase::TPtr destinationDatabase = DestinationDBName ? server.GetDatabase(DestinationDBName) : server.GetDriveAPI()->GetDatabasePtr();
    if (!destinationDatabase) {
        errors.AddMessage("TRTHistoryReplicator", "cannot get destination DB " + DestinationDBName);
        return false;
    }

    auto transaction = destinationDatabase->CreateTransaction(false);
    auto historyTable = destinationDatabase->GetTable(DestinationHistoryTableName);
    if (!historyTable) {
        errors.AddMessage("TRTHistoryReplicator", "cannot get destination history table " + DestinationHistoryTableName);
        return false;
    }
    NStorage::IQueryResult::TPtr tResult = historyTable->AddRows(records, transaction, "");
    if (!tResult || !tResult->IsSucceed()) {
        errors.AddMessage("TRTHistoryReplicator", "cannot add rows for history records: " + transaction->GetErrors().GetStringReport());
        return false;
    }

    TSet<TString> toUpsert;
    TSet<TString> toRemove;
    for (auto&& record : records) {
        if (auto id = record.Get(EntityIdColumn)) {
            EObjectHistoryAction historyAction;
            if (record.TryGet(HistoryActionColumnName, historyAction) && RemoveHistoryActions.contains(historyAction)) {
                toRemove.emplace(id);
            } else {
                toUpsert.emplace(id);
            }
        }
        ui64 eventId;
        if (record.TryGet(sourceTable->GetEventIdFieldName(), eventId)) {
            lastEventId = Max<ui64>(lastEventId, eventId);
        }
    }

    if (EntityIdColumn && SourceEntityTableName && DestinationEntityTableName) {
        auto sourceEntityTable = sourceDatabase->GetTable(SourceEntityTableName);
        if (!sourceEntityTable) {
            errors.AddMessage("TRTHistoryReplicator", "cannot get source entity table " + SourceEntityTableName);
            return false;
        }
        auto destinationEntityTable = destinationDatabase->GetTable(DestinationEntityTableName);
        if (!destinationEntityTable) {
            errors.AddMessage("TRTHistoryReplicator", "cannot get destination entity table " + DestinationEntityTableName);
            return false;
        }

        if (!toUpsert.empty()) {
            TRecordsSet entityRecords;
            auto readSourceTransaction = sourceDatabase->CreateTransaction(true);
            auto getRowsResult = sourceEntityTable->GetRows(GenerateInCondition(EntityIdColumn, toUpsert), entityRecords, readSourceTransaction);
            if (!getRowsResult || !getRowsResult->IsSucceed()) {
                errors.AddMessage("TRTHistoryReplicator", "entity records GetRows failed " + EntityIdColumn + ": " + readSourceTransaction->GetErrors().GetStringReport());
                return false;
            }

            auto upsertResult = destinationEntityTable->UpsertRows(entityRecords, EntityIdColumn, transaction);
            if (!upsertResult || !upsertResult->IsSucceed()) {
                errors.AddMessage("TRTHistoryReplicator", "entity records upsert failed " + EntityIdColumn + ": " + transaction->GetErrors().GetStringReport());
                return false;
            }
        }

        if (!toRemove.empty()) {
            auto removeResult = destinationEntityTable->RemoveRow(GenerateInCondition(EntityIdColumn, toRemove), transaction);
            if (!removeResult || !removeResult->IsSucceed()) {
                errors.AddMessage("TRTHistoryReplicator", "entity records remove failed " + EntityIdColumn + ": " + transaction->GetErrors().GetStringReport());
                return false;
            }
        }
    }

    if (!transaction->Commit()) {
        errors.AddMessage("", "cannot commit transaction " + transaction->GetErrors().GetStringReport());
        return false;
    }
    return true;
}
