#include "config.h"

#include <drive/backend/abstract/base.h>
#include <drive/backend/data/chargable.h>
#include <drive/backend/device_snapshot/manager.h>
#include <drive/backend/tags/tags.h>

#include <util/string/join.h>
#include <util/string/split.h>
#include <util/string/vector.h>

namespace NRTIdleCars {

    TRTIdleCarsWatcher::TFactory::TRegistrator<TRTIdleCarsWatcher> TRTIdleCarsWatcher::Registrator(TRTIdleCarsWatcher::GetTypeName());

    const TChecker& TRTIdleCarsWatcher::GetChecker(const TString& checkerName) const {
        auto it = Checkers.find(checkerName);
        CHECK_WITH_LOG(it != Checkers.end());
        return it->second;
    }

    TExpectedState TRTIdleCarsWatcher::DoExecuteFiltered(TAtomicSharedPtr<IRTBackgroundProcessState> /*state*/, const NDrive::IServer& server, TTagsModificationContext& context) const {
        TSet<TString> usefulCars;
        {
            auto session = server.GetDriveAPI()->template BuildTx<NSQL::Writable>();
            TRecordsSet records;
            auto result = session->Exec("SELECT object_id FROM car_tags_history WHERE tag='old_state_riding' AND history_timestamp > " + ToString((ModelingNow() - GetDowntimeCriticalDuration()).Seconds()), &records);

            if (!result || !result->IsSucceed()) {
                ERROR_LOG << "cannot take cars with downtime(new): " << session.GetStringReport() << Endl;
                return MakeUnexpected<TString>({});
            }
            for (auto&& i : records) {
                usefulCars.emplace(i.Get("object_id"));
            }
        }

        TMap<TString, std::multimap<TString, TString>> objectsByChecker;
        TSet<TString> tagNames;
        for (auto&& checker : GetCheckers()) {
            for (auto&& i : checker.second.GetTagNames()) {
                tagNames.emplace(i);
            }
            objectsByChecker.emplace(checker.second.GetName(), std::multimap<TString, TString>());
        }


        const TSet<TString> filteredCars = context.GetFilteredCarIds();

        TVector<TTaggedDevice> taggedDevices;
        for (ui32 i = 0; i < 10; ++i) {
            if (server.GetDriveAPI()->GetTagsManager().GetDeviceTags().GetObjectsFromCache(filteredCars, tagNames, taggedDevices, Now())) {
                break;
            }
        }


        TSet<TString> ids;
        auto builder = server.GetDriveAPI()->GetTagsManager().GetDeviceTags().GetHistoryManager().GetSessionsBuilder("billing", Now());
        if (!builder) {
            ERROR_LOG << "cannot control idle time in cars";
        }

        for (auto&& i : taggedDevices) {
            if (usefulCars.contains(i.GetId())) {
                continue;
            }

            for (auto&& checker : GetCheckers()) {
                const TSet<TString> tagIds = checker.second.Accept(i);
                if (tagIds.size()) {
                    objectsByChecker[checker.second.GetName()].emplace(JoinSeq(",", tagIds), i.GetId());
                    ids.emplace(i.GetId());
                }
            }
        }

        TDevicesSnapshot snapshots = server.GetSnapshotsManager().GetSnapshots(ids);
        ELocalization local = ELocalization::Rus;
        for (auto&& i : objectsByChecker) {
            TStringStream ssHeader = (!GetChecker(i.first).GetTitle() ? GetChecker(i.first).GetName() : GetChecker(i.first).GetTitle());
            ssHeader << Endl;
            if (i.second.empty()) {
                ssHeader << "Не обнаружены простаивающие (" + server.GetLocalization()->FormatDuration(local, GetDowntimeCriticalDuration()) + ") машины";
            } else {
                ssHeader << "Простаивающие (" + server.GetLocalization()->FormatDuration(local, GetDowntimeCriticalDuration()) + ") машины (" + ToString(i.second.size()) + "): " << Endl;
            }
            TVector<TString> reportLines;
            for (auto&& obj : i.second) {
                TStringStream ss;
                TDuration idleDuration = TDuration::Max();
                if (!!builder) {
                    TVector<::IEventsSession<TCarTagHistoryEvent>::TPtr> sessions = builder->GetObjectSessions(obj.second);
                    for (auto it = sessions.rbegin(); it != sessions.rend(); ++it) {
                        const TBillingSession* bSession = dynamic_cast<const TBillingSession*>(it->Get());
                        if (bSession) {
                            TDuration freeDuration;
                            TDuration pricedDuration;
                            if (bSession->GetFreeAndPricedDurations(freeDuration, pricedDuration) && pricedDuration != TDuration::Zero()) {
                                idleDuration = Now() - bSession->GetLastTS();
                                break;
                            }
                        }
                    }
                }

                const auto* car = context.GetFetchedCarsData(obj.second);
                const auto* snapshot = snapshots.GetSnapshot(obj.second);
                NDrive::TLocation location;
                TString strPosition;
                if (snapshot && snapshot->GetLocation(location)) {
                    strPosition = Sprintf(" <a href=\"http://maps.yandex.ru/?text=%f,%f\">позиция</a>", location.Latitude, location.Longitude);
                }
                if (!car) {
                    ss << obj.second;
                } else {
                    ss << car->GetHRReport();
                }
                if (idleDuration != TDuration::Max()) {
                    ss << "(" << server.GetLocalization()->FormatDuration(local, idleDuration) << ")";
                }
                if (!!strPosition) {
                    ss << " (" << strPosition << ")";
                }
                //            ss << " (" << obj.first << ")";
                reportLines.emplace_back(ss.Str());
            }

            if (!i.second.empty()) {
                NDrive::INotifier::MultiLinesNotify(server.GetNotifier(GetChecker(i.first).GetNotifierName()), ssHeader.Str(), reportLines);
            } else {
                NDrive::INotifier::Notify(server.GetNotifier(GetChecker(i.first).GetNotifierName()), ssHeader.Str());
            }
        }

        return new IRTBackgroundProcessState();
    }

    NDrive::TScheme TChecker::GetScheme(const NDrive::IServer& server) {
        NDrive::TScheme result;
        result.Add<TFSString>("name", "id");
        result.Add<TFSString>("title", "public id");
        result.Add<TFSVariants>("notifier", "notifier").SetVariants(server.GetNotifierNames());
        auto tags = server.GetDriveAPI()->GetTagsManager().GetTagsMeta().GetRegisteredTags(NEntityTagsManager::EEntityType::Car);
        result.Add<TFSVariants>("tag_names").SetVariants(NContainer::Keys(tags)).SetMultiSelect(true);
        return result;
    }

    TSet<TString> TChecker::Accept(const TTaggedObject& obj) const {
        i32 priorityMaxInterest = -Max<i32>();
        i32 priorityMax = -Max<i32>();
        TSet<TString> result;
        for (auto&& i : obj.GetTags()) {
            if (TagNames.contains(i->GetName())) {
                if (priorityMaxInterest < i->GetTagPriority(0)) {
                    result.emplace(i->GetName());
                    priorityMaxInterest = i->GetTagPriority(0);
                }
            }
            priorityMax = Max<i32>(priorityMax, i->GetTagPriority(0));
        }
        return (result.size() && (priorityMax <= priorityMaxInterest)) ? result : TSet<TString>();
    }

}
