#pragma once
#include <drive/backend/rt_background/common/config.h>

#include <drive/backend/background/manager/regular.h>

#include <rtline/library/time_restriction/time_restriction.h>
#include <rtline/util/types/accessor.h>

class TTaggedObject;

namespace NRTIdleCars {

    class TChecker {
        R_READONLY(TString, Name);
        R_READONLY(TString, Title);
        R_READONLY(TString, NotifierName);
        R_READONLY(TSet<TString>, TagNames);
    public:

        TChecker() = default;

        TChecker(const TString& name)
            : Name(name) {

        }
        NJson::TJsonValue SerializeToJson() const {
            NJson::TJsonValue result = NJson::JSON_MAP;
            JWRITE(result, "name", Name);
            JWRITE(result, "title", Title);
            JWRITE(result, "notifier", NotifierName);
            TJsonProcessor::WriteContainerArray(result, "tag_names", TagNames);
            return result;
        }

        bool DeserializeFromJson(const NJson::TJsonValue& item) {
            JREAD_STRING_OPT(item, "name", Name);
            JREAD_STRING_OPT(item, "title", Title);
            JREAD_STRING_OPT(item, "notifier", NotifierName);
            TJsonProcessor::ReadContainer(item, "tag_names", TagNames);
            return true;
        }

        static NDrive::TScheme GetScheme(const NDrive::IServer& server);

        TSet<TString> Accept(const TTaggedObject& obj) const;
    };

    class TRTIdleCarsWatcher: public IRTCarsProcess {
        R_FIELD(TDuration, DowntimeCriticalDuration, TDuration::Hours(20));
    private:
        using TBase = IRTCarsProcess;
        static TFactory::TRegistrator<TRTIdleCarsWatcher> Registrator;
        TMap<TString, TChecker> Checkers;
    public:
        using TBase::TBase;

        const TMap<TString, TChecker>& GetCheckers() const {
            return Checkers;
        }

        virtual TString GetType() const override {
            return GetTypeName();
        }

        static TString GetTypeName() {
            return "idle_cars_control";
        }

        const TChecker& GetChecker(const TString& checkerName) const;

        virtual NDrive::TScheme DoGetScheme(const IServerBase& server) const override {
            NDrive::TScheme result = TBase::DoGetScheme(server);
            result.Add<TFSDuration>("critical_duration").SetDefault(TDuration::Hours(20));
            result.Add<TFSArray>("checkers").SetElement(TChecker::GetScheme(server.GetAsSafe<NDrive::IServer>()));
            return result;
        }

        virtual bool DoDeserializeFromJson(const NJson::TJsonValue& jsonInfo) override {
            if (!TBase::DoDeserializeFromJson(jsonInfo)) {
                return false;
            }
            TJsonProcessor::Read(jsonInfo, "critical_duration", DowntimeCriticalDuration);
            const NJson::TJsonValue::TArray* arrCheckersJson;
            if (jsonInfo["checkers"].GetArrayPointer(&arrCheckersJson)) {
                for (auto&& i : *arrCheckersJson) {
                    TChecker checker;
                    if (checker.DeserializeFromJson(i)) {
                        Checkers.emplace(checker.GetName(), std::move(checker));
                    }
                }
            }
            return true;
        }

        virtual TExpectedState DoExecuteFiltered(TAtomicSharedPtr<IRTBackgroundProcessState> state, const NDrive::IServer& server, TTagsModificationContext& context) const override;

        virtual NJson::TJsonValue DoSerializeToJson() const override {
            NJson::TJsonValue result = TBase::DoSerializeToJson();
            TJsonProcessor::WriteDurationString(result, "critical_duration", DowntimeCriticalDuration);
            NJson::TJsonValue& checkersJson = result.InsertValue("checkers", NJson::JSON_ARRAY);
            for (auto&& i : Checkers) {
                checkersJson.AppendValue(i.second.SerializeToJson());
            }
            return result;
        }
    };

}
