#include "config.h"

#include <drive/backend/abstract/base.h>
#include <drive/backend/device_snapshot/manager.h>
#include <drive/backend/tags/tags.h>

#include <util/string/join.h>
#include <util/string/split.h>
#include <util/string/vector.h>

namespace NRTIdleTags {

    TRTIdleTagsWatcher::TFactory::TRegistrator<TRTIdleTagsWatcher> TRTIdleTagsWatcher::Registrator(TRTIdleTagsWatcher::GetTypeName());

    class TTagCheckInfo {
        R_FIELD(TString, TagName);
        R_FIELD(TDuration, Duration, TDuration::Zero());
    public:
        TTagCheckInfo(const TString& tagName, const TDuration& duration)
            : TagName(tagName)
            , Duration(duration) {

        }

        bool operator<(const TTagCheckInfo& item) const {
            return Duration < item.GetDuration();
        };
    };

    class TCheckInfo {
        R_FIELD(TString, ObjectId);
        R_FIELD(TSet<TTagCheckInfo>, TagsInfo);
    public:
        TCheckInfo(const TString& objectId, TSet<TTagCheckInfo>&& tags)
            : ObjectId(objectId)
            , TagsInfo(std::move(tags)) {

        }
        TString GetTagsInfoReport(const bool noTagName, const IServerBase& server) const {
            TStringStream ss;
            for (auto&& i : TagsInfo) {
                if (!ss.Empty()) {
                    ss << ", ";
                }
                if (i.GetDuration() != TDuration::Zero()) {
                    if (noTagName) {
                        ss << server.GetLocalization()->FormatDuration(ELocalization::Rus, i.GetDuration());
                    } else {
                        ss << i.GetTagName() << " " << server.GetLocalization()->FormatDuration(ELocalization::Rus, i.GetDuration());
                    }
                } else {
                    ss << i.GetTagName();
                }
            }
            return ss.Str();
        }

        bool operator<(const TCheckInfo& item) const {
            if (TagsInfo.rbegin() == TagsInfo.rend()) {
                return true;
            }
            if (item.GetTagsInfo().rbegin() == item.GetTagsInfo().rend()) {
                return false;
            }
            return *TagsInfo.rbegin() < *item.GetTagsInfo().rbegin();
        };
    };

    TExpectedState TRTIdleTagsWatcher::DoExecuteFiltered(TAtomicSharedPtr<IRTBackgroundProcessState> /*state*/, const NDrive::IServer& server, TTagsModificationContext& context) const {
        TMap<TString, TSet<TCheckInfo>> objectsByChecker;
        TSet<TString> tagNames;
        for (auto&& checker : GetCheckers()) {
            for (auto&& i : checker.second.GetTagNames()) {
                tagNames.emplace(i);
            }
            objectsByChecker.emplace(checker.second.GetName(), TSet<TCheckInfo>());
        }

        const auto& deviceTagManager = server.GetDriveAPI()->GetTagsManager().GetDeviceTags();
        auto session = deviceTagManager.BuildSession(true);

        TSet<TString> usefulTags;
        {
            TRecordsSet records;
            TString condition = "SELECT DISTINCT tag_id FROM car_tags_history WHERE history_timestamp > " + ToString((ModelingNow() - GetDowntimeCriticalDuration()).Seconds()) + " AND tag IN ('" + JoinSeq("','", tagNames) + "')";
            if (!GetStartActions().empty()) {
                condition += " AND history_action IN('remove','evolve','" + JoinSeq("', '", GetStartActions()) + "')";
            }

            auto result = session->Exec(condition, &records);

            if (!result || !result->IsSucceed()) {
                ERROR_LOG << "cannot take cars with downtime(new): " << session.GetStringReport() << Endl;
                return MakeUnexpected<TString>({});
            }
            for (auto&& i : records) {
                usefulTags.emplace(i.Get("tag_id"));
            }
        }

        const TSet<TString> filteredCars = context.GetFilteredCarIds();

        TVector<TTaggedDevice> taggedDevices;
        for (ui32 i = 0; i < 10; ++i) {
            if (server.GetDriveAPI()->GetTagsManager().GetDeviceTags().GetObjectsFromCache(filteredCars, tagNames, taggedDevices, Now())) {
                break;
            }
        }

        TSet<TString> ids;

        if (!server.GetDriveAPI()->GetTagsManager().GetDeviceTags().GetHistoryManager().Update(Now())) {
            return MakeUnexpected<TString>("cannot update DeviceTagsHistoryManager");
        }

        for (auto&& i : taggedDevices) {
            for (auto&& checker : GetCheckers()) {
                TVector<TDBTag> tags = checker.second.Accept(i);
                tags.erase(std::remove_if(tags.begin(), tags.end(), [&usefulTags](const TDBTag& tag) ->bool {return usefulTags.contains(tag.GetTagId()); }), tags.end());
                if (tags.size()) {
                    TSet<TTagCheckInfo> tagNamesLocal;
                    for (auto&& tag : tags) {
                        auto optionalEvents = deviceTagManager.GetEventsByTag(tag.GetTagId(), session);
                        if (!optionalEvents) {
                            return MakeUnexpected("cannot GetEventsByObject: " + session.GetStringReport());
                        }
                        TMaybe<TCarTagHistoryEvent> lastEvent;
                        if (!optionalEvents->empty()) {
                            if (GetStartActions().empty()) {
                                lastEvent = optionalEvents->back();
                            } else {
                                for (auto it = optionalEvents->rbegin(); it != optionalEvents->rend(); ++it) {
                                    if (GetStartActions().contains(ToString(it->GetHistoryAction()))) {
                                        lastEvent = *it;
                                        break;
                                    }
                                }
                            }
                        }

                        if (!!lastEvent) {
                            tagNamesLocal.emplace(tag->GetName(), Now() - lastEvent->GetHistoryInstant());
                        } else {
                            tagNamesLocal.emplace(tag->GetName(), TDuration::Max());
                        }
                    }
                    objectsByChecker[checker.second.GetName()].emplace(i.GetId(), std::move(tagNamesLocal));
                    ids.emplace(i.GetId());
                }
            }
        }

        TDevicesSnapshot snapshots = server.GetSnapshotsManager().GetSnapshots(ids);

        for (auto&& i : objectsByChecker) {
            TStringStream ssHeader = (!GetChecker(i.first).GetTitle() ? GetChecker(i.first).GetName() : GetChecker(i.first).GetTitle());
            ssHeader << Endl;
            if (i.second.empty()) {
                ssHeader << "Не обнаружены не исполняемые (" << server.GetLocalization()->FormatDuration(ELocalization::Rus, GetDowntimeCriticalDuration()) << ") задания";
            } else {
                ssHeader << "Не исполняемые (" << server.GetLocalization()->FormatDuration(ELocalization::Rus, GetDowntimeCriticalDuration()) << ") задания для (" << i.second.size() << " объектов): ";
            }
            ssHeader << Endl;
            TVector<TString> reportLines;
            if (GetChecker(i.first).GetFullReport()) {
                for (auto itObj = i.second.rbegin(); itObj != i.second.rend(); ++itObj) {
                    const auto& obj = *itObj;
                    TStringStream ss;
                    const auto* car = context.GetFetchedCarsData(obj.GetObjectId());
                    const auto* snapshot = snapshots.GetSnapshot(obj.GetObjectId());
                    NDrive::TLocation location;
                    TString strPosition;
                    if (snapshot && snapshot->GetLocation(location)) {
                        strPosition = Sprintf(" <a href=\"http://maps.yandex.ru/?text=%f,%f\">позиция</a>", location.Latitude, location.Longitude);
                    }
                    if (!car) {
                        ss << obj.GetObjectId();
                    } else {
                        ss << car->GetHRReport();
                    }
                    if (!!strPosition) {
                        ss << " (" << strPosition << ")";
                    }
                    ss << " (" << obj.GetTagsInfoReport(GetChecker(i.first).GetTagNames().size() == 1, server) << ")";
                    reportLines.emplace_back(ss.Str());
                }
            }

            if (!reportLines.empty()) {
                NDrive::INotifier::MultiLinesNotify(server.GetNotifier(GetChecker(i.first).GetNotifierName()), ssHeader.Str(), reportLines);
            } else {
                NDrive::INotifier::Notify(server.GetNotifier(GetChecker(i.first).GetNotifierName()), ssHeader.Str());
            }
        }

        return new IRTBackgroundProcessState();
    }

    TVector<TDBTag> TChecker::Accept(const TTaggedObject& obj) const {
        i32 priorityMaxInterest = -Max<i32>();
        i32 priorityMax = -Max<i32>();
        TVector<TDBTag> result;
        for (auto&& i : obj.GetTags()) {
            if (GetTagNames().contains(i->GetName())) {
                if (priorityMaxInterest < i->GetTagPriority(0)) {
                    result.emplace_back(i);
                    priorityMaxInterest = i->GetTagPriority(0);
                }
            }
            priorityMax = Max<i32>(priorityMax, i->GetTagPriority(0));
        }
        return (result.size() && (priorityMax <= priorityMaxInterest)) ? result : TVector<TDBTag>();
    }

    const TChecker& TRTIdleTagsWatcher::GetChecker(const TString& checkerName) const {
        auto it = Checkers.find(checkerName);
        CHECK_WITH_LOG(it != Checkers.end());
        return it->second;
    }
}
