#pragma once
#include <drive/backend/rt_background/idle_cars/config.h>

#include <drive/backend/background/manager/regular.h>

#include <rtline/library/time_restriction/time_restriction.h>
#include <rtline/util/types/accessor.h>

class TTaggedObject;
class TDBTag;
namespace NRTIdleTags {

    class TChecker: public NRTIdleCars::TChecker {
    private:
        using TBase = NRTIdleCars::TChecker;
        R_READONLY(bool, FullReport, true);
    public:

        TChecker() = default;

        TChecker(const TString& name)
            : TBase(name) {

        }

        TVector<TDBTag> Accept(const TTaggedObject& obj) const;
        NJson::TJsonValue SerializeToJson() const {
            NJson::TJsonValue result = TBase::SerializeToJson();
            JWRITE(result, "full_report", FullReport);
            return result;
        }

        bool DeserializeFromJson(const NJson::TJsonValue& item) {
            if (!TBase::DeserializeFromJson(item)) {
                return false;
            }
            JREAD_BOOL_OPT(item, "full_report", FullReport);
            return true;
        }

        static NDrive::TScheme GetScheme(const NDrive::IServer& server) {
            NDrive::TScheme result = TBase::GetScheme(server);
            result.Add<TFSBoolean>("full_report").SetDefault(true);
            return result;
        }
    };

    class TRTIdleTagsWatcher: public IRTCarsProcess {
        R_FIELD(TDuration, DowntimeCriticalDuration, TDuration::Hours(20));
        R_FIELD(TSet<TString>, StartActions);
    private:
        using TBase = IRTCarsProcess;
        static TFactory::TRegistrator<TRTIdleTagsWatcher> Registrator;
        TMap<TString, TChecker> Checkers;
    public:
        using TBase::TBase;

        const TMap<TString, TChecker>& GetCheckers() const {
            return Checkers;
        }

        virtual TString GetType() const override {
            return GetTypeName();
        }

        static TString GetTypeName() {
            return "idle_tags_control";
        }

        const TChecker& GetChecker(const TString& checkerName) const;

        virtual NDrive::TScheme DoGetScheme(const IServerBase& server) const override {
            NDrive::TScheme result = TBase::DoGetScheme(server);
            result.Add<TFSDuration>("critical_duration").SetDefault(TDuration::Hours(20));
            result.Add<TFSVariants>("start_actions").InitVariants<EObjectHistoryAction>().SetMultiSelect(true);
            result.Add<TFSArray>("checkers").SetElement(TChecker::GetScheme(server.GetAsSafe<NDrive::IServer>()));
            return result;
        }

        virtual bool DoDeserializeFromJson(const NJson::TJsonValue& jsonInfo) override {
            if (!TBase::DoDeserializeFromJson(jsonInfo)) {
                return false;
            }
            TJsonProcessor::Read(jsonInfo, "critical_duration", DowntimeCriticalDuration);
            if (!TJsonProcessor::ReadContainer(jsonInfo, "start_actions", StartActions)) {
                return false;
            }
            const NJson::TJsonValue::TArray* arrCheckersJson;
            if (jsonInfo["checkers"].GetArrayPointer(&arrCheckersJson)) {
                for (auto&& i : *arrCheckersJson) {
                    TChecker checker;
                    if (checker.DeserializeFromJson(i)) {
                        Checkers.emplace(checker.GetName(), std::move(checker));
                    }
                }
            }
            return true;
        }

        virtual TExpectedState DoExecuteFiltered(TAtomicSharedPtr<IRTBackgroundProcessState> state, const NDrive::IServer& server, TTagsModificationContext& context) const override;

        virtual NJson::TJsonValue DoSerializeToJson() const override {
            NJson::TJsonValue result = TBase::DoSerializeToJson();
            TJsonProcessor::WriteDurationString(result, "critical_duration", DowntimeCriticalDuration);
            TJsonProcessor::WriteContainerArray(result, "start_actions", StartActions);
            NJson::TJsonValue& checkersJson = result.InsertValue("checkers", NJson::JSON_ARRAY);
            for (auto&& i : Checkers) {
                checkersJson.AppendValue(i.second.SerializeToJson());
            }
            return result;
        }
    };

}
