#include "renins_claim_catalogue_updater.h"

#include <drive/backend/database/drive_api.h>
#include <drive/backend/incident/renins_claims/catalogue_entity_kasko.h>
#include <drive/backend/incident/renins_claims/catalogue_entity_osago.h>
#include <drive/backend/incident/renins_claims/client.h>

#include <util/string/builder.h>

TString TRTReninsClaimCatalogueUpdater::GetTypeName() {
    return "incident_renins_claim_catalogue_updater";
}

IRTBackgroundProcess::TFactory::TRegistrator<TRTReninsClaimCatalogueUpdater> TRTReninsClaimCatalogueUpdater::Registrator(TRTReninsClaimCatalogueUpdater::GetTypeName());

TRTReninsClaimCatalogueUpdater::TRTReninsClaimCatalogueUpdater()
    : TBase()
{
    InitNotifyHandlers();
}

NDrive::TScheme TRTReninsClaimCatalogueUpdater::DoGetScheme(const IServerBase& server) const {
    auto scheme = TBase::DoGetScheme(server);
    return scheme;
}

bool TRTReninsClaimCatalogueUpdater::DoDeserializeFromJson(const NJson::TJsonValue& data) {
    return TBase::DoDeserializeFromJson(data);
}

NJson::TJsonValue TRTReninsClaimCatalogueUpdater::DoSerializeToJson() const {
    auto result = TBase::DoSerializeToJson();
    return result;
}

bool TRTReninsClaimCatalogueUpdater::DoStart(const TRTBackgroundProcessContainer& container) {
    if (!TBase::DoStart(container)) {
        return false;
    }
    if (!NotifyHandlers) {
        return false;
    }
    NotifyHandlers->SetSignalName(GetRTProcessName());
    return true;
}

void TRTReninsClaimCatalogueUpdater::InitNotifyHandlers() {
    if (!NotifyHandlers) {
        NotifyHandlers = MakeAtomicShared<TNotifyHandlers>(GetType());
    }
}

TExpectedState TRTReninsClaimCatalogueUpdater::DoExecute(TAtomicSharedPtr<IRTBackgroundProcessState> /* state */, const TExecutionContext& context) const {
    const NDrive::IServer& server = context.GetServerAs<NDrive::IServer>();

    if (!Yensured(server.GetDriveAPI())->HasReninsClaimClient()) {
        return MakeUnexpected<TString>("Renins claim client unavailable");
    }

    const NDrive::NRenins::TReninsClaimClient& client = server.GetDriveAPI()->GetReninsClaimClient();

    TVector<TAtomicSharedPtr<NDrive::NRenins::ICatalogueEntry>> entriesToUpdate;

    // only kasko ones now
    {
        TSet<NDrive::NRenins::EKaskoCatalogueType> kaskoCatalogueTypes;
        NDrive::NRenins::IKaskoCatalogueEntry::TFactory::GetRegisteredKeys(kaskoCatalogueTypes);
        for (auto&& catalogueType: kaskoCatalogueTypes) {
            TAtomicSharedPtr<NDrive::NRenins::IKaskoCatalogueEntry> entryPtr = NDrive::NRenins::IKaskoCatalogueEntry::TFactory::Construct(catalogueType);
            if (entryPtr && entryPtr->IsRequestable()) {
                entriesToUpdate.push_back(entryPtr);
            }
        }
    }

    for (auto&& entryPtr: entriesToUpdate) {
        TMessagesCollector errors;
        if (!client.GetCatalogueValuesSync(*entryPtr, errors)) {
            NotifyHandlers->Handle(EStatus::ObtainError, server);
            return MakeUnexpected<TString>(
                TStringBuilder() << "Error obtaining catalogue value: "
                << "type - " << entryPtr->GetType()
                << ", claim type - " << entryPtr->GetClaimType()
                << ", errors - " << errors.GetStringReport()
            );
        }

        if (!entryPtr->StoreSettingValue(&server, GetRobotUserId())) {
            NotifyHandlers->Handle(EStatus::UpdateError, server);
            return MakeUnexpected<TString>(
                TStringBuilder() << "Error updating catalogue value: "
                << "type - " << entryPtr->GetType() << ", claim type - " << entryPtr->GetClaimType()
            );
        }

        NotifyHandlers->Handle(EStatus::Success, server);
    }

    {
        NDrive::NRenins::TKaskoSTOAAddressEntry entry;
        TMessagesCollector errors;
        Y_ENSURE(client.GetSTOAAddressesSync(entry, errors), "cannot GetSTOAAddresses: " << errors.GetStringReport());
        Y_ENSURE(entry.StoreSettingValue(&server, GetRobotUserId()));
    }

    if (!server.GetSettings().SetValue(NDrive::NRenins::KaskoSettingPrefix + ".catalogue_values_last_update", ::ToString(Now().Seconds()), GetRobotUserId())) {
        return MakeUnexpected<TString>("Cannot set catalogue values last update time");
    }

    return MakeAtomicShared<IRTBackgroundProcessState>();
}
