#include "issue_date.h"

#include <drive/backend/data/leasing/leasing.h>
#include <drive/backend/database/drive_api.h>
#include <drive/backend/rt_background/manager/state.h>
#include <drive/backend/tags/tags_manager.h>

namespace NDrivematics {
    IRTRegularBackgroundProcess::TFactory::TRegistrator<TIssueDateThresholdChecker> TIssueDateThresholdChecker::Registrator(TIssueDateThresholdChecker::GetTypeName());

    TString TIssueDateThresholdChecker::GetTypeName() {
        return "issue_date_threshold_checker";
    }

    TString TIssueDateThresholdChecker::GetType() const {
        return GetTypeName();
    }

    NDrive::TScheme TIssueDateThresholdChecker::DoGetScheme(const IServerBase& server) const {
        NDrive::TScheme result = TBase::DoGetScheme(server);
        result.Add<TFSNumeric>("issue_date_threshold", "Порог в днях, начиная с которого считается, что машина новая").SetRequired(true);
        return result;
    }

    bool TIssueDateThresholdChecker::DoDeserializeFromJson(const NJson::TJsonValue& value) {
        return TBase::DoDeserializeFromJson(value)
            && NJson::ParseField(value["issue_date_threshold"], IssueDateThreshold, true);
    }

    NJson::TJsonValue TIssueDateThresholdChecker::DoSerializeToJson() const {
        NJson::TJsonValue result = TBase::DoSerializeToJson();
        result["issue_date_threshold"] = IssueDateThreshold;
        return result;
    }

    TExpectedState TIssueDateThresholdChecker::DoExecute(TAtomicSharedPtr<IRTBackgroundProcessState> state, const TExecutionContext& context) const {
        const auto& server = context.GetServerAs<NDrive::IServer>();
        const TDriveAPI& api = *Yensured(server.GetDriveAPI());
        const auto& deviceTagsManager = api.GetTagsManager().GetDeviceTags();
        const auto& tagsMeta = api.GetTagsManager().GetTagsMeta();
        auto session = api.BuildTx<NSQL::Writable>();
        TTaggedObjectsSnapshot carsDevicesSnapshot;
        Y_ENSURE(api.GetTagsManager().GetDeviceTags().GetObjectsFromCacheByIds({}, carsDevicesSnapshot, TInstant::Zero()), "can't restore cars from cache");
        for (const auto& [carId, taggedObject] : carsDevicesSnapshot) {
            auto leasingStatsDBTags = taggedObject.GetTagsByClass<TLeasingStatsTag>();
            if (leasingStatsDBTags.size() != 1) {
                continue;
            }
            auto issueDate = Yensured(leasingStatsDBTags[0].GetTagAs<TLeasingStatsTag>())->GetIssueDate();
            if (!issueDate) {
                continue;
            }
            ui32 daysDiff = Now().Days() - issueDate;
            bool shouldMarkWithTag = (daysDiff < IssueDateThreshold);
            auto maybeTag = taggedObject.GetTag(FreshIssueDateTagName);
            if (shouldMarkWithTag) {
                if (!maybeTag) {
                    auto tag = tagsMeta.CreateTag(FreshIssueDateTagName);
                    Y_ENSURE(tag, "cannot create tag " << FreshIssueDateTagName);
                    Y_ENSURE(deviceTagsManager.AddTag(tag, GetRobotUserId(), carId, &server, session), "can't add tag " << FreshIssueDateTagName << ": " << session.GetStringReport());
                }
            } else {
                if (maybeTag) {
                    Y_ENSURE(deviceTagsManager.RemoveTagSimple(*maybeTag, GetRobotUserId(), session, false), "can't remove tag: " << session.GetStringReport());
                }
            }
        }
        Y_ENSURE(session.Commit(), "cannot Commit: " << session.GetStringReport());
        return state;
    }

}
