#include "supply_hours.h"

#include <drive/backend/data/leasing/leasing.h>
#include <drive/backend/database/drive_api.h>
#include <drive/backend/rt_background/manager/state.h>
#include <drive/backend/tags/tags_manager.h>

namespace NDrivematics {
    IRTRegularBackgroundProcess::TFactory::TRegistrator<TSupplyHoursThresholdChecker> TSupplyHoursThresholdChecker::Registrator(TSupplyHoursThresholdChecker::GetTypeName());

    TString TSupplyHoursThresholdChecker::GetTypeName() {
        return "supply_hours_threshold_checker";
    }

    TString TSupplyHoursThresholdChecker::GetType() const {
        return GetTypeName();
    }

    NDrive::TScheme TSupplyHoursThresholdChecker::DoGetScheme(const IServerBase& server) const {
        NDrive::TScheme result = TBase::DoGetScheme(server);
        result.Add<TFSNumeric>("supply_hours_threshold", "Порог, начиная с которого считается, что машина не выполняет SH").SetRequired(true);
        return result;
    }

    bool TSupplyHoursThresholdChecker::DoDeserializeFromJson(const NJson::TJsonValue& value) {
        return TBase::DoDeserializeFromJson(value)
            && NJson::ParseField(value["supply_hours_threshold"], SupplyHoursThreshold, true);
    }

    NJson::TJsonValue TSupplyHoursThresholdChecker::DoSerializeToJson() const {
        NJson::TJsonValue result = TBase::DoSerializeToJson();
        result["supply_hours_threshold"] = SupplyHoursThreshold;
        return result;
    }


    TExpectedState TSupplyHoursThresholdChecker::DoExecute(TAtomicSharedPtr<IRTBackgroundProcessState> state, const TExecutionContext& context) const {
        const auto& server = context.GetServerAs<NDrive::IServer>();
        const TDriveAPI& api = *Yensured(server.GetDriveAPI());
        const auto& deviceTagsManager = api.GetTagsManager().GetDeviceTags();
        const auto& tagsMeta = api.GetTagsManager().GetTagsMeta();
        auto session = api.BuildTx<NSQL::Writable>();
        TTaggedObjectsSnapshot carsDevicesSnapshot;
        Y_ENSURE(api.GetTagsManager().GetDeviceTags().GetObjectsFromCacheByIds({}, carsDevicesSnapshot, TInstant::Zero()), "can't restore cars from cache");
        for (const auto& [carId, taggedObject] : carsDevicesSnapshot) {
            auto leasingStatsDBTags = taggedObject.GetTagsByClass<TLeasingStatsTag>();
            if (leasingStatsDBTags.size() != 1) {
                continue;
            }
            auto aggregatedStats = Yensured(leasingStatsDBTags[0].GetTagAs<TLeasingStatsTag>())->AggregateStats(Min<ui32>(), Max<ui32>());
            if (!aggregatedStats) {
                continue;
            }
            bool shouldMarkWithTag = (aggregatedStats->GetSupplyHours() < SupplyHoursThreshold);
            auto maybeTag = taggedObject.GetTag(InsufficientSupplyHoursTagName);
            if (shouldMarkWithTag) {
                if (!maybeTag) {
                    auto tag = tagsMeta.CreateTag(InsufficientSupplyHoursTagName);
                    Y_ENSURE(tag, "cannot create tag " << InsufficientSupplyHoursTagName);
                    Y_ENSURE(deviceTagsManager.AddTag(tag, GetRobotUserId(), carId, &server, session), "can't add tag " << InsufficientSupplyHoursTagName << ": " << session.GetStringReport());
                }
            } else {
                if (maybeTag) {
                    Y_ENSURE(deviceTagsManager.RemoveTagSimple(*maybeTag, GetRobotUserId(), session, false), "can't remove tag: " << session.GetStringReport());
                }
            }
        }
        Y_ENSURE(session.Commit(), "cannot Commit: " << session.GetStringReport());
        return state;
    }
}
