#include "before_delivery_process.h"

#include "helper.h"

#include <drive/backend/data/chargable.h>
#include <drive/backend/data/long_term.h>
#include <drive/backend/data/support_tags.h>
#include <drive/backend/offers/actions/long_term.h>
#include <drive/backend/roles/manager.h>

TExpectedState TLongTermBeforeDeliveryProcess::DoExecute(TAtomicSharedPtr<IRTBackgroundProcessState> state, const TExecutionContext& context) const {
    Y_UNUSED(state);

    const auto& server = context.GetServerAs<NDrive::IServer>();
    const auto& api = *Yensured(server.GetDriveAPI());
    const auto& userTagManager = api.GetTagsManager().GetUserTags();

    TDBTags offerHolders;
    {
        auto tx = userTagManager.BuildTx<NSQL::ReadOnly>();
        auto optionalOfferHolders = GetOfferHolders(server, tx);
        if (!optionalOfferHolders) {
            tx.Check();
        }
        offerHolders = std::move(*optionalOfferHolders);
    }

    auto readTx = userTagManager.BuildTx<NSQL::ReadOnly>();

    auto now = ModelingNow();
    auto since = now - NotificationInterval;
    auto optionalEvents = userTagManager.GetEvents(TRange<TInstant>(since), readTx, IEntityTagsManager::TQueryOptions()
        .SetTags({ NotificationTagName })
    );
    if (!optionalEvents) {
        return MakeUnexpected<TString>("cannot GetEventsByObject: " + readTx.GetStringReport());
    }

    TSet<TString> notifiedUsers;
    for (auto event: *optionalEvents) {
        notifiedUsers.insert(event.GetObjectId());
    }

    TVector<TDBTag> tasks;

    for (auto&& offerHolder : offerHolders) {
        if (notifiedUsers.contains(offerHolder.GetObjectId())) {
            continue;
        }

        if (!TagNames.contains(offerHolder->GetName())) {
            continue;
        }

        auto offerHolderTag = Yensured(offerHolder.GetTagAs<TOfferHolderTag>());

        auto threshold = now + NotificationInterval;
        auto deliveryDay = GetScheduledAt(*offerHolderTag);
        if (threshold < deliveryDay) {
            INFO_LOG << GetRobotId() << ": tag " << offerHolder.GetTagId() << " is not due yet" << Endl;
            continue;
        }

        if (!DoCheckRunCondition(server, *offerHolderTag, readTx)) {
            continue;
        }
        tasks.push_back(std::move(offerHolder));
    }

    Y_ENSURE(readTx.Rollback());

    for (auto&& offerHolder : tasks) try {
        auto tx = userTagManager.BuildTx<NSQL::Writable>();

        if (!DoRun(server, offerHolder, tx)) {
            continue;
        }

        if (!tx.Commit()) {
            NDrive::TEventLog::Log(GetRobotId(), NJson::TMapBuilder
                ("message", "cannot commit for tag")
                ("tag_id", offerHolder.GetTagId())
                ("error", tx.GetStringReport())
            );
            continue;
        }
    } catch (...) {
        NDrive::TEventLog::Log(GetRobotId(), NJson::TMapBuilder
            ("message", "cannot process tag")
            ("tag_id", offerHolder.GetTagId())
            ("exception", CurrentExceptionInfo(true).GetStringRobust())
        );
    }

    return MakeAtomicShared<IRTBackgroundProcessState>();
}

bool TLongTermBeforeDeliveryProcess::DoRun(const NDrive::IServer& server, const TDBTag& offerHolder, NDrive::TEntitySession& tx) const {
    const auto& api = *Yensured(server.GetDriveAPI());
    const auto& userTagManager = api.GetTagsManager().GetUserTags();
    const auto localization = server.GetLocalization();
    auto offerHolderTag = Yensured(offerHolder.GetTagAs<TOfferHolderTag>());
    auto offer = Yensured(offerHolderTag->GetOffer());
    if (NotificationTagName) {
        auto tag = api.GetTagsManager().GetTagsMeta().CreateTag(NotificationTagName);
        FillNotificationTag(tag, *offerHolderTag, localization);
        auto added = userTagManager.AddTag(tag, GetRobotUserId(), offer->GetUserId(), &server, tx);
        if (!added) {
            NDrive::TEventLog::Log(GetRobotId(), NJson::TMapBuilder
                ("message", "cannot AddTag for user")
                ("user_id", offer->GetUserId())
                ("error", tx.GetStringReport())
            );
            return false;
        }
    }
    return true;
}

void TLongTermBeforeDeliveryProcess::FillNotificationTag(ITag::TPtr tag, const TOfferHolderTag& offerHolderTag, const ILocalization* localization) const {
    if (auto notificationTag = std::dynamic_pointer_cast<TUserMessageTagBase>(tag)) {
        auto longTermOffer = std::dynamic_pointer_cast<TLongTermOffer>(offerHolderTag.GetOffer());
        auto locale = Yensured(longTermOffer)->GetLocale();
        auto firstPayment = GetFirstPayment(*longTermOffer);
        auto valueString = localization ? localization->FormatPrice(locale, firstPayment) : ToString(0.01 * firstPayment);
        notificationTag->AddMacro("_PaymentValue_", valueString);
    }
}

TOptionalDBTags TLongTermBeforeDeliveryProcess::GetOfferHolders(const NDrive::IServer& server, NDrive::TEntitySession& tx) const {
    return TLongTermOfferHolderTag::RestoreOfferHolderTags({}, server, tx);
}

TInstant TLongTermBeforeDeliveryProcess::GetScheduledAt(const TOfferHolderTag& offerHolderTag) const {
    auto longTermOffer = std::dynamic_pointer_cast<TLongTermOffer>(offerHolderTag.GetOffer());
    return TInstant::Days(Yensured(longTermOffer)->GetSince().Days());
}

NDrive::TScheme TLongTermBeforeDeliveryProcess::DoGetScheme(const IServerBase& server) const {
    const auto impl = server.GetAs<NDrive::IServer>();
    const auto api = impl ? impl->GetDriveAPI() : nullptr;
    const auto holderTagNames = api
        ? api->GetTagsManager().GetTagsMeta().GetRegisteredTagNames({ TLongTermOfferHolderTag::Type() })
        : TSet<TString>();
    const auto notificationTagNames = api
        ? api->GetTagsManager().GetTagsMeta().GetRegisteredTagNames({ TUserPushTag::TypeName })
        : TSet<TString>();
    NDrive::TScheme scheme = TBase::DoGetScheme(server);
    scheme.Add<TFSDuration>("notification_interval").SetDefault(NotificationInterval);
    scheme.Add<TFSVariants>("tag_names").SetVariants(holderTagNames).SetMultiSelect(true);
    scheme.Add<TFSVariants>("notification_tag_name").SetVariants(notificationTagNames);
    return scheme;
}

bool TLongTermBeforeDeliveryProcess::DoDeserializeFromJson(const NJson::TJsonValue& value) {
    if (!TBase::DoDeserializeFromJson(value)) {
        return false;
    }
    return
        NJson::ParseField(value["notification_interval"], NotificationInterval) &&
        NJson::ParseField(value["tag_names"], TagNames) &&
        NJson::ParseField(value["notification_tag_name"], NotificationTagName) &&
        true;
}

NJson::TJsonValue TLongTermBeforeDeliveryProcess::DoSerializeToJson() const {
    NJson::TJsonValue result = TBase::DoSerializeToJson();
    result["notification_interval"] = NJson::ToJson(NJson::Hr(NotificationInterval));
    result["tag_names"] = NJson::ToJson(TagNames);
    result["notification_tag_name"] = NotificationTagName;
    return result;
}
