#include "process.h"

#include <drive/backend/data/chargable.h>
#include <drive/backend/data/long_term.h>
#include <drive/backend/data/notifications_tags.h>
#include <drive/backend/data/support_tags.h>
#include <drive/backend/device_snapshot/manager.h>
#include <drive/backend/offers/actions/long_term.h>
#include <drive/backend/roles/manager.h>

struct TAutoassignDeviceInfo {
    TString Id;
    i32 Tier;
    TInstant AvailableUntil = TInstant::Max();
    double MileageDistanceCoeff = 0;
};

double ComputeMileageDistanceCoeff(double mileage, double distance) {
    return (mileage + 10000) * distance;
}

TExpectedState TLongTermAutoassignProcess::DoExecute(TAtomicSharedPtr<IRTBackgroundProcessState> state, const TExecutionContext& context) const {
    Y_UNUSED(state);
    const auto& server = context.GetServerAs<NDrive::IServer>();
    const auto api = Yensured(server.GetDriveAPI());
    const auto& userTagManager = api->GetTagsManager().GetUserTags();

    auto permissions = api->GetUserPermissions(GetRobotUserId());
    if (!permissions) {
        ERROR_LOG << GetRobotId() << ": cannot create permissions for " << GetRobotUserId() << Endl;
        return nullptr;
    }

    TDBTags offerHolders;
    {
        auto tx = userTagManager.BuildTx<NSQL::ReadOnly>();
        auto optionalOfferHolders = TLongTermOfferHolderTag::RestoreOfferHolderTags({}, server, tx);
        if (!optionalOfferHolders) {
            tx.Check();
        }
        offerHolders = std::move(*optionalOfferHolders);
    }

    TSet<TString> busy;
    for (auto&& offerHolder : offerHolders) {
        auto longTermOfferHolderTag = offerHolder.GetTagAs<TLongTermOfferHolderTag>();
        if (!longTermOfferHolderTag) {
            ERROR_LOG << GetRobotId() << ": cannot cast tag " << offerHolder.GetTagId() << Endl;
            continue;
        }
        auto offer = longTermOfferHolderTag->GetOffer();
        auto longTermOffer = std::dynamic_pointer_cast<TLongTermOffer>(offer);
        if (!longTermOffer) {
            ERROR_LOG << GetRobotId() << ": cannot cast offer from tag " << offerHolder.GetTagId() << Endl;
            continue;
        }
        if (longTermOffer->GetObjectId()) {
            busy.insert(longTermOffer->GetObjectId());
        }
    }
    INFO_LOG << GetRobotId() << ": " << busy.size() << " busy" << Endl;

    auto selectCandidate = [&](const TVector<TTaggedObject>& candidatesTier1,
                               const TVector<TTaggedObject>& candidatesTier2,
                               const TLongTermOfferBuilder& builder,
                               const TLongTermOffer& offer) -> TString {
        std::vector<TAutoassignDeviceInfo> candidates;
        for (auto &candidate : candidatesTier1) {
            candidates.push_back({candidate.GetId(), 1});
        }
        for (auto &candidate : candidatesTier2) {
            candidates.push_back({candidate.GetId(), 2});
        }
        erase_if(candidates, [&](const auto& candidate) {
            return !candidate.Id || busy.contains(candidate.Id);
        });
        if (builder.IsAvailableUntilByCarInfo()) {
            for (auto &candidate : candidates) {
                candidate.AvailableUntil = TLongTermOfferBuilder::GetCarLongTermUntil(candidate.Id, server);
                if (!candidate.AvailableUntil) {
                    candidate.AvailableUntil = TInstant::Max();
                }
            }
            erase_if(candidates, [&](const auto& candidate) {
                return candidate.AvailableUntil < offer.GetUntil();
            });
        }
        for (auto&& candidate : candidates) {
            auto snapshot = server.GetSnapshotsManager().GetSnapshot(candidate.Id);
            double coeff = 0;
            auto maybeMileage = snapshot.GetMileage();
            auto maybeLocation = snapshot.GetLocation();
            if (maybeMileage && maybeLocation && offer.HasDeliveryLocation()) {
                auto distance = maybeLocation->GetCoord().GetLengthTo(offer.GetDeliveryLocationUnsafe());
                coeff = ComputeMileageDistanceCoeff(*maybeMileage, distance);
            } else {
                if (!maybeMileage) {
                    NDrive::TEventLog::Log("SelectCandidate", NJson::TMapBuilder
                                              ("error", "no_device_mileage")
                                              ("candidate_id", candidate.Id));
                }
                if (!maybeLocation) {
                    NDrive::TEventLog::Log("SelectCandidate", NJson::TMapBuilder
                                              ("error", "no_device_location")
                                              ("candidate_id", candidate.Id));
                }
                if (!offer.HasDeliveryLocation()) {
                    NDrive::TEventLog::Log("SelectCandidate", NJson::TMapBuilder
                                              ("error", "no_offer_delivery_location")
                                              ("offer_id", offer.GetOfferId()));
                }
            }
            candidate.MileageDistanceCoeff = coeff;
        }
        SortBy(candidates, [&](const auto& c) {
            return std::make_tuple(c.AvailableUntil, c.Tier, c.MileageDistanceCoeff);
        });
        if (!candidates.empty()) {
            return candidates.front().Id;
        }
        return {};
    };

    for (auto&& offerHolder : offerHolders) try {
        auto longTermOfferHolderTag = offerHolder.GetTagAs<TLongTermOfferHolderTag>();
        auto offer = Yensured(longTermOfferHolderTag)->GetOffer();
        auto longTermOffer = std::dynamic_pointer_cast<TLongTermOffer>(Yensured(offer));
        if (longTermOffer->GetObjectId()) {
            DEBUG_LOG << GetRobotId() << ": tag " << offerHolder.GetTagId() << " already has object_id " << longTermOffer->GetObjectId() << Endl;
            continue;
        }
        if (!OfferNames.contains(longTermOffer->GetBehaviourConstructorId())) {
            continue;
        }
        auto now = Now();
        auto threshold = now + AssignmentInterval;
        if (threshold < longTermOffer->GetSince()) {
            INFO_LOG << GetRobotId() << ": tag " << offerHolder.GetTagId() << " is not due yet" << Endl;
            continue;
        }
        auto builder = api->GetRolesManager()->GetAction(longTermOffer->GetBehaviourConstructorId());
        auto longTermOfferBuilder = Yensured(builder)->GetAs<TLongTermOfferBuilder>();
        auto candidates = longTermOfferBuilder->FindCandidates(server);
        auto candidateId = selectCandidate(candidates.Tier1, candidates.Tier2, *longTermOfferBuilder, *longTermOffer);
        if (!candidateId) {
            ERROR_LOG << GetRobotId() << ": cannot select candidate for tag " << offerHolder.GetTagId() << Endl;
            continue;
        }
        if (DryRun) {
            INFO_LOG << GetRobotId() << ": selected candidate " << candidateId << " for tag " << offerHolder.GetTagId() << Endl;
        } else {
            auto tx = userTagManager.BuildTx<NSQL::Writable | NSQL::RepeatableRead>();
            auto optionalTag = userTagManager.RestoreTag(offerHolder.GetTagId(), tx);
            if (!optionalTag) {
                tx.Check();
            }
            if (!TLongTermOfferHolderTag::AssignCarId(*optionalTag, candidateId, TargetTagName, *permissions, server, tx)) {
                tx.Check();
            }
            if (!tx.Commit()) {
                tx.Check();
            }
        }
        busy.insert(candidateId);
    } catch (...) {
        ERROR_LOG << GetRobotId() << ": cannot process tag " << offerHolder.GetTagId() << ": " << CurrentExceptionInfo(true).GetStringRobust() << Endl;
    }
    return MakeAtomicShared<IRTBackgroundProcessState>();
}

NDrive::TScheme TLongTermAutoassignProcess::DoGetScheme(const IServerBase& server) const {
    const auto impl = server.GetAs<NDrive::IServer>();
    const auto api = impl ? impl->GetDriveAPI() : nullptr;
    const auto rolesManager = api ? api->GetRolesManager() : nullptr;
    const auto offerNames = rolesManager
        ? rolesManager->GetActionsDB().GetActionNamesWithType<TLongTermOfferBuilder>(/*reportDeprecated=*/false)
        : TVector<TString>();
    const auto tagNames = api
        ? api->GetTagsManager().GetTagsMeta().GetRegisteredTagNames({ TLongTermOfferHolderTag::Type(), TSupportOutgoingCommunicationTag::TypeName })
        : TSet<TString>();
    NDrive::TScheme scheme = TBase::DoGetScheme(server);
    scheme.Add<TFSDuration>("assignment_interval").SetDefault(AssignmentInterval);
    scheme.Add<TFSBoolean>("dry_run").SetDefault(DryRun);
    scheme.Add<TFSVariants>("offer_names").SetVariants(offerNames).SetMultiSelect(true);
    scheme.Add<TFSVariants>("target_tag_name").SetVariants(tagNames);
    return scheme;
}

bool TLongTermAutoassignProcess::DoDeserializeFromJson(const NJson::TJsonValue& value) {
    if (!TBase::DoDeserializeFromJson(value)) {
        return false;
    }
    return
        NJson::ParseField(value["assignment_interval"], AssignmentInterval) &&
        NJson::ParseField(value["dry_run"], DryRun) &&
        NJson::ParseField(value["offer_names"], OfferNames) &&
        NJson::ParseField(value["target_tag_name"], TargetTagName) &&
        true;
}

NJson::TJsonValue TLongTermAutoassignProcess::DoSerializeToJson() const {
    NJson::TJsonValue result = TBase::DoSerializeToJson();
    result["assignment_interval"] = NJson::ToJson(NJson::Hr(AssignmentInterval));
    result["dry_run"] = DryRun;
    result["offer_names"] = NJson::ToJson(OfferNames);
    result["target_tag_name"] = TargetTagName;
    return result;
}

TExpectedState TLongTermPaymentPushProcess::DoExecute(TAtomicSharedPtr<IRTBackgroundProcessState> state, const TExecutionContext& context) const {
    Y_UNUSED(state);
    const auto& server = context.GetServerAs<NDrive::IServer>();
    const auto api = Yensured(server.GetDriveAPI());
    const auto localization = server.GetLocalization();
    const auto& userTagManager = api->GetTagsManager().GetUserTags();
    const auto sessionBuilder = api->GetTagsManager().GetDeviceTags().GetSessionsBuilder("billing");
    const auto sessions = Yensured(sessionBuilder)->GetSessionsActual();
    for (auto& session : sessions) {
        if (!session) {
            continue;
        }
        if (session->GetClosed()) {
            continue;
        }
        auto billingSession = std::dynamic_pointer_cast<TBillingSession>(session);
        if (!billingSession) {
            ERROR_LOG << GetRobotId() << ": cannot cast " << session->GetSessionId() << " to BillingSession" << Endl;
            continue;
        }
        auto currentOffer = billingSession->GetCurrentOffer();
        if (!currentOffer) {
            ERROR_LOG << GetRobotId() << ": cannot find CurrentOffer in " << session->GetSessionId() << Endl;
            continue;
        }
        auto longTermOffer = std::dynamic_pointer_cast<TLongTermOffer>(currentOffer);
        if (!longTermOffer) {
            continue;
        }
        if (!OfferNames.empty() && !OfferNames.contains(longTermOffer->GetBehaviourConstructorId())) {
            continue;
        }
        auto billingCompilation = billingSession->GetCompilationAs<TBillingSession::TBillingCompilation>();
        if (!billingCompilation) {
            ERROR_LOG << GetRobotId() << ": cannot construct TBillingCompilation in " << session->GetSessionId() << Endl;
            continue;
        }
        auto packOfferState = billingCompilation->GetCurrentOfferStateAs<TPackOfferState>();
        if (!packOfferState) {
            ERROR_LOG << GetRobotId() << ": cannot construct TPackOfferState in " << session->GetSessionId() << Endl;
            continue;
        }
        auto schedule = longTermOffer->GetOrBuildPaymentSchedule();
        auto now = Now();
        auto nextQuantum = schedule.GetNextQuantum(now, packOfferState->GetMileage());
        if (!nextQuantum) {
            continue;
        }
        if (!nextQuantum->ShouldBePaid(now + WarningInterval, packOfferState->GetMileage() + WarningMileage)) {
            continue;
        }
        if (UserTagName) {
            auto since = now - WarningInterval;
            auto tx = userTagManager.BuildSession(true);
            auto optionalEvents = userTagManager.GetEventsByObject(currentOffer->GetUserId(), tx, 0, since, IEntityTagsManager::TQueryOptions(1)
                .SetTags({ UserTagName })
            );
            if (!optionalEvents) {
                ERROR_LOG << GetRobotId() << ": cannot GetEventsByObject for " << session->GetSessionId() << ": " << tx.GetStringReport() << Endl;
                continue;
            }
            if (!optionalEvents->empty()) {
                continue;
            }
        }
        if (UserTagName) {
            auto tag = api->GetTagsManager().GetTagsMeta().CreateTag(UserTagName);
            auto notificationTag = std::dynamic_pointer_cast<TUserMessageTagBase>(tag);
            if (notificationTag) {
                auto locale = longTermOffer->GetLocale();
                auto value = longTermOffer->GetPublicDiscountedPrice(nextQuantum->Value);
                auto valueString = localization ? localization->FormatPrice(locale, value) : ToString(0.01 * value);
                notificationTag->AddMacro("_PaymentValue_", valueString);
            }

            auto tx = userTagManager.BuildSession();
            auto added = userTagManager.AddTag(tag, GetRobotUserId(), currentOffer->GetUserId(), &server, tx);
            if (!added) {
                ERROR_LOG << GetRobotId() << ": cannot AddTag for " << session->GetSessionId() << ": " << tx.GetStringReport() << Endl;
                continue;
            }
            if (!tx.Commit()) {
                ERROR_LOG << GetRobotId() << ": cannot Commit for " << session->GetSessionId() << ": " << tx.GetStringReport() << Endl;
                continue;
            }
        }
        INFO_LOG << GetRobotId() << ": eligble session " << session->GetSessionId() << Endl;
    }
    return MakeAtomicShared<IRTBackgroundProcessState>();
}

NDrive::TScheme TLongTermPaymentPushProcess::DoGetScheme(const IServerBase& server) const {
    const auto impl = server.GetAs<NDrive::IServer>();
    const auto api = impl ? impl->GetDriveAPI() : nullptr;
    const auto rolesManager = api ? api->GetRolesManager() : nullptr;
    const auto offerNames = rolesManager
        ? rolesManager->GetActionsDB().GetActionNamesWithType<TLongTermOfferBuilder>(/*reportDeprecated=*/false)
        : TVector<TString>();
    NDrive::TScheme scheme = TBase::DoGetScheme(server);
    scheme.Add<TFSVariants>("offer_names", "Названия офферов").SetVariants(offerNames).SetMultiSelect(true);
    scheme.Add<TFSString>("user_tag_name", "Тег нотификации");
    scheme.Add<TFSDuration>("warning_interval", "Интервал нотификации").SetDefault(WarningInterval);
    scheme.Add<TFSNumeric>("warning_mileage", "Пробег до следующего списания для нотификации").SetDefault(WarningMileage);
    return scheme;
}

bool TLongTermPaymentPushProcess::DoDeserializeFromJson(const NJson::TJsonValue& value) {
    if (!TBase::DoDeserializeFromJson(value)) {
        return false;
    }
    return
        NJson::ParseField(value["offer_names"], OfferNames) &&
        NJson::ParseField(value["user_tag_name"], UserTagName) &&
        NJson::ParseField(value["warning_interval"], WarningInterval) &&
        NJson::ParseField(value["warning_mileage"], WarningMileage);
}

NJson::TJsonValue TLongTermPaymentPushProcess::DoSerializeToJson() const {
    NJson::TJsonValue result = TBase::DoSerializeToJson();
    result["offer_names"] = NJson::ToJson(OfferNames);
    result["user_tag_name"] = NJson::ToJson(NJson::Nullable(UserTagName));
    result["warning_interval"] = NJson::ToJson(NJson::Hr(WarningInterval));
    result["warning_mileage"] = NJson::ToJson(WarningMileage);
    return result;
}

TLongTermAutoassignProcess::TFactory::TRegistrator<TLongTermAutoassignProcess> TLongTermAutoassignProcess::Registrator(TLongTermAutoassignProcess::GetTypeName());
TLongTermPaymentPushProcess::TFactory::TRegistrator<TLongTermPaymentPushProcess> TLongTermPaymentPushProcess::Registrator(TLongTermPaymentPushProcess::GetTypeName());
