#include "config.h"

#include <drive/backend/rt_background/manager/state.h>

#include <drive/backend/data/notifications_tags.h>
#include <drive/backend/database/drive_api.h>
#include <drive/backend/device_snapshot/snapshots/tag.h>
#include <drive/backend/logging/events.h>
#include <drive/backend/tags/tag_description.h>
#include <drive/backend/tags/tags_manager.h>

#include <rtline/library/json/builder.h>

#include <util/generic/utility.h>

class TSendInfo {
    R_READONLY(TTagDescription::TConstPtr, Description);
    R_FIELD(TVector<TDBTag>, Tags);

public:
    explicit TSendInfo(TTagDescription::TConstPtr description)
        : Description(description)
    {
    }
};

TRTUserMailSenderProcess::TFactory::TRegistrator<TRTUserMailSenderProcess> TRTUserMailSenderProcess::Registrator("user_mail_sender");

TString TRTUserMailSenderProcess::GetType() const {
    return "user_mail_sender";
}

NDrive::TScheme TRTUserMailSenderProcess::DoGetScheme(const IServerBase& server) const {
    NDrive::TScheme scheme = TBase::DoGetScheme(server);
    scheme.Add<TFSVariants>("notifier", "Название агента посылки сообщений").SetVariants(server.GetNotifierNames());
    scheme.Add<TFSNumeric>("pack_size", "Размер пакета сообщений").SetDefault(1000);
    return scheme;
}

NJson::TJsonValue TRTUserMailSenderProcess::DoSerializeToJson() const {
    NJson::TJsonValue result = TBase::DoSerializeToJson();
    TJsonProcessor::Write(result, "notifier", Notifier);
    TJsonProcessor::Write(result, "pack_size", PackSize);
    return result;
}

bool TRTUserMailSenderProcess::DoDeserializeFromJson(const NJson::TJsonValue& jsonInfo) {
    if (!TBase::DoDeserializeFromJson(jsonInfo)) {
        return false;
    }

    JREAD_STRING(jsonInfo, "notifier", Notifier);
    JREAD_INT_OPT(jsonInfo, "pack_size", PackSize);
    return true;
}

TExpectedState TRTUserMailSenderProcess::DoExecute(TAtomicSharedPtr<IRTBackgroundProcessState> /*state*/, const TExecutionContext& context) const {
    const NDrive::IServer* server = &context.GetServerAs<NDrive::IServer>();
    NDrive::INotifier::TPtr notifier = server->GetNotifier(GetNotifier());
    if (!notifier) {
        ERROR_LOG << "Incorrect notifier for TMailSender" << Endl;
        return MakeUnexpected<TString>({});
    }

    TMap<TString, TSendInfo> tagsSendInfo;
    TVector<TDBTag> tags;
    TMap<TString, TVector<TString>> usersByTagName;
    ui32 incorrectTags = 0;
    TSet<TString> userIds;
    const TInstant now = ModelingNow();
    TVector<TDBTag> oldTagsForRemove;
    TSet<TString> unsubscribedUserIds, undefinedSubscriptionUserIds;
    {
        auto session = server->GetDriveAPI()->template BuildTx<NSQL::ReadOnly>();
        ITagsMeta::TTagDescriptions td = server->GetDriveAPI()->GetTagsManager().GetTagsMeta().GetTagsByType(TUserMailTag::TypeName);
        TSet<TString> tagNames;
        for (auto&& i : td) {
            tagNames.emplace(i->GetName());
            tagsSendInfo.emplace(i->GetName(), TSendInfo(i));
        }
        auto optionalTags = server->GetDriveAPI()->GetTagsManager().GetUserTags().RestoreTags(TSet<TString>(), MakeVector(tagNames), session, GetPackSize());
        if (!optionalTags) {
            return MakeUnexpected<TString>("cannot restore tags: " + session.GetStringReport());
        }
        tags = std::move(*optionalTags);

        for (auto&& i : tags) {
            const TUserMailTag* mailNotificationTag = i.GetTagAs<TUserMailTag>();
            if (!mailNotificationTag) {
                continue;
            }

            if (mailNotificationTag->GetSendInstant() > now) {
                continue;
            }

            if (mailNotificationTag->GetDeadline() < now) {
                i->SetObjectSnapshot(MakeAtomicShared<TTagSnapshot>(TTagSnapshot::EActionReason::NotificationTooOld));
                oldTagsForRemove.push_back(i);
                continue;
            }

            auto it = tagsSendInfo.find(i->GetName());
            if (it == tagsSendInfo.end()) {
                ++incorrectTags;
            } else {
                it->second.MutableTags().emplace_back(i);
                userIds.emplace(i.GetObjectId());
                if (!unsubscribedUserIds.contains(i.GetObjectId())) {
                    TUserAppSettings userAppSettings(i.GetObjectId());
                    auto subscription = userAppSettings.GetEmailSubscription(session);
                    if (!subscription && subscription.GetError() != TUserSettingStatus::NotFound) {
                        ERROR_LOG << "Can't get subscription info for user " << i.GetObjectId() << ", error: " << session.GetStringReport() << Endl;
                        undefinedSubscriptionUserIds.emplace(i.GetObjectId());
                        session = server->GetDriveAPI()->template BuildTx<NSQL::ReadOnly>();
                    } else if (subscription && IsFalse(subscription.GetRef())) {
                        unsubscribedUserIds.emplace(i.GetObjectId());
                    }
                }
            }
        }
    }

    if (incorrectTags) {
        WARNING_LOG << "Incorrect tags for send mail: " << incorrectTags << Endl;
    }
    RemoveTagsGently(server, oldTagsForRemove);

    auto locale = ELocalization::Rus;
    auto gUsers = server->GetDriveAPI()->GetUsersData()->FetchInfo(userIds);
    for (auto&& [_, tagInfo] : tagsSendInfo) {
        auto mailTagDesc = dynamic_cast<const TUserMailTag::TDescription*>(tagInfo.GetDescription().Get());
        if (!mailTagDesc) {
            return nullptr;
        }
        bool ignoreSubscription = mailTagDesc->GetCommunicationChannelSettings().GetIgnoreSubscriptionStatus();

        TVector<TDBTag> tagsForRemove;
        for (auto&& tag : tagInfo.MutableTags()) {
            if (tagsForRemove.size() >= GetPackSize()) {
                break;
            }

            if (!ignoreSubscription && undefinedSubscriptionUserIds.contains(tag.GetObjectId())) {
                INFO_LOG << "Skipped tag " << mailTagDesc->GetName() << " for " << tag.GetObjectId() << " because subscription status is unknown" << Endl;
                continue;
            }
            if (!ignoreSubscription && unsubscribedUserIds.contains(tag.GetObjectId())) {
                tag->SetObjectSnapshot(MakeAtomicShared<TTagSnapshot>(TTagSnapshot::EActionReason::UserUnsubscribedFromCommunitation));
                tagsForRemove.emplace_back(tag);
                INFO_LOG << "Skipped tag " << mailTagDesc->GetName() << " for " << tag.GetObjectId() << " because user was unsubscribed" << Endl;
                continue;
            }
            const auto* userData = gUsers.GetResultPtr(tag.GetObjectId());
            if (!userData) {
                tag->SetObjectSnapshot(MakeAtomicShared<TTagSnapshot>(TTagSnapshot::EActionReason::WrongUser));
                tagsForRemove.emplace_back(tag);
                ERROR_LOG << "Incorrect user for tag " << tag.GetTagId() << Endl;
                continue;
            }
            if (mailTagDesc->GetEnabled()) {
                const TUserMailTag* tagData = tag.GetTagAs<TUserMailTag>();
                if (!tagData) {
                    return MakeUnexpected<TString>("Incorrect tag in TMailSender " + tag.GetTagId());
                }
                auto message = tagData->BuildMessage(locale, *server, *mailTagDesc, *userData);
                if (!message) {
                    NDrive::TEventLog::Log("MailSenderSkip", NJson::TMapBuilder
                        ("tag", tagData->GetName())
                        ("reason", message.GetError())
                    );
                    continue;
                }

                auto resp = notifier->Notify(*message, NDrive::INotifier::TContext().SetRecipients({*userData}));
                if (!resp || !resp->HasErrors()) {
                    tag->SetObjectSnapshot(MakeAtomicShared<TTagSnapshot>(TTagSnapshot::EActionReason::NotificationSent));
                    tagsForRemove.emplace_back(tag);
                }
            } else {
                NDrive::TEventLog::Log("MailSenderSkip", NJson::TMapBuilder
                    ("user_id", tag.GetObjectId())
                    ("reason", "disabled notifier")
                );
            }
        }
        RemoveTagsGently(server, tagsForRemove);
    }

    return new IRTBackgroundProcessState();
}

void TRTUserMailSenderProcess::RemoveTagsGently(
        const NDrive::IServer* server, const TVector<TDBTag>& tagsForRemove) const {
    const auto& start = tagsForRemove.begin();
    size_t i = 0;
    while (i < tagsForRemove.size()) {
        size_t endIndex = Min(tagsForRemove.size(), i + GetPackSize());
        TVector<TDBTag> sublist = TVector<TDBTag>(start + i, start + endIndex);
        for (ui32 att = 0; att < 7; ++att) {
            auto session = server->GetDriveAPI()->template BuildTx<NSQL::Writable>();
            if (!server->GetDriveAPI()->GetTagsManager().GetUserTags().RemoveTagsSimple(sublist, GetRobotUserId(), session, true) || !session.Commit()) {
                WARNING_LOG << att << " Attempt for remove user mail tags: " << session.GetStringReport() << Endl;
                continue;
            } else {
                break;
            }
        }
        i = endIndex;
    }
}
