#include "process.h"

#include <drive/backend/data/dictionary_tags.h>
#include <drive/backend/database/drive_api.h>
#include <drive/backend/tags/tags_manager.h>

#include <drive/library/cpp/sendr/client.h>
#include <library/cpp/regex/pcre/regexp.h>

IRTRegularBackgroundProcess::TFactory::TRegistrator<TRTMailSubscriptionWatcher> TRTMailSubscriptionWatcher::Registrator(TRTMailSubscriptionWatcher::GetTypeName());
TRTInstantWatcherState::TFactory::TRegistrator<TRTMailSubscriptionWatcherState> TRTMailSubscriptionWatcherState::Registrator(TRTMailSubscriptionWatcher::GetTypeName());

TString TRTMailSubscriptionWatcherState::GetType() const {
    return TRTMailSubscriptionWatcher::GetTypeName();
}

TSet<TString> GetEmails(const NDrive::IServer& server, const TSet<TString>& userIds, const TRegExMatch& regexMatch) {
    TSet<TString> result;
    if (userIds.empty()) {
        return result;
    }
    auto users = server.GetDriveAPI()->GetUsersData()->FetchInfo(userIds);
    for (auto&& id : userIds) {
        const auto* userData = users.GetResultPtr(id);
        if (userData && userData->GetEmail()) {
            if (regexMatch.IsCompiled()) {
                if (regexMatch.Match(userData->GetEmail().data())) {
                    result.emplace(userData->GetEmail());
                } else {
                    NDrive::TEventLog::Log("MailSubscriptionWatcher", NJson::TMapBuilder
                        ("user_id", id)
                        ("email", userData->GetEmail())
                        ("info", "Skipped not matching to regex")
                    );
                }
            } else {
                result.emplace(userData->GetEmail());
            }
        }
    }
    return result;
}

TExpectedState TRTMailSubscriptionWatcher::DoExecute(TAtomicSharedPtr<IRTBackgroundProcessState> state, const TExecutionContext& context) const {
    const TRTMailSubscriptionWatcherState* rState = dynamic_cast<const TRTMailSubscriptionWatcherState*>(state.Get());
    const NDrive::IServer& server = context.GetServerAs<NDrive::IServer>();
    TOptionalTagHistoryEvents events;
    const ui64 sinceId = rState ? rState->GetLastEventId() : StartEventId;
    {
        const auto& userTagsManager = server.GetDriveAPI()->GetTagsManager().GetUserTags();
        auto session = userTagsManager.BuildSession(true);
        auto lockedEventId = userTagsManager.GetLockedMaxEventId(session);
        if (!lockedEventId) {
            return MakeUnexpected<TString>("could not GetLockedMaxEventId");
        }
        const ui64 untilId = *lockedEventId + 1;
        auto queryOptions = TTagEventsManager::TQueryOptions(Limit);
        queryOptions.SetTags({ GetTagName() });
        queryOptions.SetOrderBy({ "history_event_id" });
        events = userTagsManager.GetEvents({sinceId, untilId}, session, queryOptions);
        if (!events) {
            return MakeUnexpected<TString>(session.GetStringReport());
        }
    }

    TAtomicSharedPtr<TRTMailSubscriptionWatcherState> result = new TRTMailSubscriptionWatcherState();
    if (events->empty()) {
        result->SetLastEventId(sinceId);
        return result;
    }

    TMaybe<bool> defaultValue;
    if (auto tagDescription = server.GetDriveAPI()->GetTagsManager().GetTagsMeta().GetDescriptionByName(GetTagName())) {
        auto description = dynamic_cast<const TDictionaryTagDescription*>(tagDescription.Get());
        if (description) {
            auto defaultString = description->GetDefaults(GetFieldName());
            if (defaultString) {
                defaultValue = IsTrue(*defaultString);
            }
        } else {
            return MakeUnexpected<TString>("tag " + GetTagName() + " is not disctionary tag");
        }
    }

    TSet<TString> subscribedIds, unsubscribedIds;
    for (auto&& ev : Reversed(*events)) {
        if (subscribedIds.contains(ev.GetObjectId()) || unsubscribedIds.contains(ev.GetObjectId())) {
            continue;
        }
        const TUserDictionaryTag* settingsData = ev.GetTagAs<TUserDictionaryTag>();
        if (!settingsData) {
            return MakeUnexpected<TString>("tag " + GetTagName() + " is not disctionary tag");
        }
        auto emailStatus = settingsData->GetField(GetFieldName());
        if (emailStatus.Defined() || defaultValue.Defined()) {
            bool isSubscribed = emailStatus.Defined() ? IsTrue(emailStatus.GetRef()) : *defaultValue;
            if (isSubscribed) {
                subscribedIds.emplace(ev.GetObjectId());
            } else {
                unsubscribedIds.emplace(ev.GetObjectId());
            }
        }
    }

    TRegExMatch regexMatch;
    if (!EmailValidationRegex.empty()) {
        regexMatch.Compile(EmailValidationRegex);
    }
    TSet<TString> subscribedEmails = GetEmails(server, subscribedIds, regexMatch);
    TSet<TString> unsubscribedEmails = GetEmails(server, unsubscribedIds, regexMatch);

    TMessagesCollector errors;
    if (auto sendr = server.GetSendrClient()) {
        for (auto&& list : GetSubscriptionLists()) {
            if (!unsubscribedEmails.empty() && !sendr->Unsubscribe(unsubscribedEmails, list, errors)) {
                return MakeUnexpected<TString>(errors.GetStringReport());
            }
            if (!subscribedEmails.empty() && !sendr->Subscribe(subscribedEmails, list, errors)) {
                return MakeUnexpected<TString>(errors.GetStringReport());
            }
        }
    } else {
        return MakeUnexpected<TString>("sendr client not configured");
    }

    result->SetLastEventId(events->back().GetHistoryEventId());
    return result;
}

NDrive::TScheme TRTMailSubscriptionWatcher::DoGetScheme(const IServerBase& server) const {
    NDrive::TScheme scheme = TBase::DoGetScheme(server);
    scheme.Add<TFSArray>("subscription_lists", "Списки отписки").SetElement<TFSString>();
    scheme.Add<TFSString>("tag_name", "Имя тега с данными");
    scheme.Add<TFSString>("field_name", "Поле тега с состоянием подписки");
    scheme.Add<TFSNumeric>("limit", "Лимит количества обрабатываемых событий");
    scheme.Add<TFSNumeric>("start_event_id", "Начать обработку с события");
    scheme.Add<TFSString>("validation_regex", "Валидация адресов").SetReadOnly(true);
    return scheme;
}

bool TRTMailSubscriptionWatcher::DoDeserializeFromJson(const NJson::TJsonValue& jsonInfo) {
    if (!NJson::ParseField(jsonInfo, "validation_regex", EmailValidationRegex, false)) {
        return false;
    }
    if (!EmailValidationRegex.empty()) {
        TRegExMatch regexMatch;
        regexMatch.Compile(EmailValidationRegex);
        if (!regexMatch.IsCompiled()) {
            return false;
        }
    }
    return
        NJson::ParseField(jsonInfo, "subscription_lists", SubscriptionLists, false) &&
        NJson::ParseField(jsonInfo, "tag_name", TagName, true) &&
        NJson::ParseField(jsonInfo, "field_name", FieldName, true) &&
        NJson::ParseField(jsonInfo, "limit", Limit, false) &&
        NJson::ParseField(jsonInfo, "start_event_id", StartEventId, false) &&
        TBase::DoDeserializeFromJson(jsonInfo);
}

NJson::TJsonValue TRTMailSubscriptionWatcher::DoSerializeToJson() const {
    NJson::TJsonValue result = TBase::DoSerializeToJson();
    NJson::InsertField(result, "subscription_lists", SubscriptionLists);
    NJson::InsertField(result, "tag_name", TagName);
    NJson::InsertField(result, "field_name", FieldName);
    NJson::InsertField(result, "limit", Limit);
    NJson::InsertField(result, "start_event_id", StartEventId);
    NJson::InsertField(result, "validation_regex", EmailValidationRegex);
    return result;
}
