#include "config.h"

#include <drive/backend/cars/car_model.h>
#include <drive/backend/cars/hardware.h>
#include <drive/backend/cars/status/state_filters.h>
#include <drive/backend/data/device_tags.h>

#include <rtline/util/logging/tskv_log.h>
#include <rtline/util/types/uuid.h>

TRTMajorMaintenanceWatcher::TFactory::TRegistrator<TRTMajorMaintenanceWatcher> TRTMajorMaintenanceWatcher::Registrator(TRTMajorMaintenanceWatcher::GetTypeName());
TRTMajorTOSynchronizerWatcher::TFactory::TRegistrator<TRTMajorTOSynchronizerWatcher> TRTMajorTOSynchronizerWatcher::Registrator(TRTMajorTOSynchronizerWatcher::GetTypeName());
TRTMajorNewCarProcess::TFactory::TRegistrator<TRTMajorNewCarProcess> TRTMajorNewCarProcess::Registrator(TRTMajorNewCarProcess::GetTypeName());
TRTMajorCheckCarsProcess::TFactory::TRegistrator<TRTMajorCheckCarsProcess> TRTMajorCheckCarsProcess::Registrator(TRTMajorCheckCarsProcess::GetTypeName());

const TVector<ui32> TRTMajorMaintenanceWatcher::DefaultMaintenancePoints = {15000, 30000, 45000, 60000, 75000, 90000, 105000};

TMaybe<TMajorWatcherContext::ENeedMaintenance> TMajorWatcherContext::CheckNeedMaintenance(const double currentMileage, ui32& maintenanceNumber, const TMaintenanceInfo* info) const {
    i32 idxCurrent = -1;
    for (ui32 i = 0; i < GetMaintenancePoints().size(); ++i) {
        if (std::abs(currentMileage - GetMaintenancePoints()[i]) < GetMaintenancePrecisionNew()) {
            idxCurrent = i;
        }
    }
    if (idxCurrent == -1) {
        for (ui32 i = 0; i < GetMaintenancePoints().size(); ++i) {
            if (currentMileage > GetMaintenancePoints()[i]) {
                idxCurrent = i;
            }
        }
    }
    if (idxCurrent == -1) {
        maintenanceNumber = 0;
        return ENeedMaintenance::Ready;
    }

    if (!info) {
        maintenanceNumber = idxCurrent + 1;
        return ENeedMaintenance::Need;
    }
    if (!info->HasReadyDate()) {
        maintenanceNumber = 0;
        return ENeedMaintenance::InProgress;
    }
    i32 idxPred = -1;
    if (info->HasMileage()) {
        for (ui32 i = 0; i < GetMaintenancePoints().size(); ++i) {
            if (std::abs(info->GetMileageUnsafe() - GetMaintenancePoints()[i]) <GetMaintenancePrecisionHistory()) {
                idxPred = i;
            }
        }

        if (idxPred == -1) {
            for (ui32 i = 0; i < GetMaintenancePoints().size(); ++i) {
                if (info->GetMileageUnsafe() > GetMaintenancePoints()[i]) {
                    idxPred = i;
                }
            }
        }
    }

    if (idxCurrent > idxPred) {
        maintenanceNumber = idxCurrent + 1;
        return ENeedMaintenance::Need;
    }
    return ENeedMaintenance::Ready;
}

bool TMajorWatcherContext::Init(const NDrive::IServer& server) {
    auto session = server.GetDriveAPI()->GetMaintenanceDB().BuildSession(/* readOnly = */ true);
    TSet<TString> vins;
    const auto& cars = RTContext.GetFetchedCarsData();
    Transform(cars.begin(), cars.end(), std::inserter(vins, vins.begin()), [](const auto& it) { return it.second.GetVin(); });
    auto infos = server.GetDriveAPI()->GetMaintenanceDB().GetObjects(vins, session);
    if (!infos) {
        NDrive::TEventLog::Log("MaintenanceError", NJson::TMapBuilder
            ("error", session.GetReport())
        );
        return false;
    }
    Transform(infos->begin(), infos->end(), std::inserter(MaintenanceInfo, MaintenanceInfo.begin()), [] (auto&& info) -> std::pair<TString, TMaintenanceInfo> { return { info.GetVIN(), info }; });
    return true;
}

TMaybe<TRTMajorMaintenanceWatcher::ESpecialAction> TMajorWatcherContext::CheckDevice(const TString& carId, const double value) const {
    const TDriveCarInfo* info = RTContext.GetFetchedCarsData(carId);
    if (!info) {
        return IRTSensorToTagsWatcher::ESpecialAction::Remove;
    }
    ui32 idx;
    auto maintenanceStatus = CheckNeedMaintenance(value, idx, MaintenanceInfo.FindPtr(info->GetVin()));
    NJson::TJsonValue ev = NJson::TMapBuilder
        ("car_id", carId)
        ("mileage", value)
        ("status", ToString(*maintenanceStatus))
        ("idx", idx)
    ;
    switch (*maintenanceStatus) {
        case ENeedMaintenance::InProgress:
            NDrive::TEventLog::Log("MaintenanceInProgress", ev);
            return IRTSensorToTagsWatcher::ESpecialAction::Ignore;
        case ENeedMaintenance::Need:
            if ((!MaintenancePoint) || (MaintenancePoints[idx - 1] == MaintenancePoint)) {
                NDrive::TEventLog::Log("MaintenanceAdd", ev);
                return IRTSensorToTagsWatcher::ESpecialAction::Add;
            } else {
                NDrive::TEventLog::Log("MaintenanceRemove", ev);
                return IRTSensorToTagsWatcher::ESpecialAction::Remove;
            }
        case ENeedMaintenance::Ready:
            NDrive::TEventLog::Log("MaintenanceRemove", ev);
            return IRTSensorToTagsWatcher::ESpecialAction::Remove;
    }
}

NDrive::TScheme TRTMajorMaintenanceWatcher::DoGetScheme(const IServerBase& server) const {
    NDrive::TScheme scheme = TBase::DoGetScheme(server);
    scheme.Add<TFSNumeric>("maintenance_precision", "Точность для определения архивного ТО(км)").SetMin(0).SetDefault(3000);
    scheme.Add<TFSNumeric>("maintenance_precision_new", "Точность для определения нового ТО(км)").SetMin(0).SetDefault(1500);
    scheme.Add<TFSNumeric>("maintenance_point_int", "Целевое ТО (0 - не важно)").SetMin(0).SetMax(1000000).SetDefault(0);
    scheme.Add<TFSString>("maintenance_points", "Схема ТО").SetDefault(JoinSeq(",", DefaultMaintenancePoints));
    return scheme;
}

bool TRTMajorMaintenanceWatcher::DoDeserializeFromJson(const NJson::TJsonValue& jsonInfo) {
    if (!TBase::DoDeserializeFromJson(jsonInfo)) {
        return false;
    }
    if (jsonInfo.Has("maintenance_point")) {
        TString maintenancePoint;
        JREAD_STRING_OPT(jsonInfo, "maintenance_point", maintenancePoint);
        if (!TryFromString(maintenancePoint, MaintenancePoint)) {
            MaintenancePoint = 0;
        }
    } else {
        JREAD_INT_OPT(jsonInfo, "maintenance_point_int", MaintenancePoint);
    }
    MaintenancePoints.clear();
    JREAD_CONTAINER_OPT(jsonInfo, "maintenance_points", MaintenancePoints);
    if (MaintenancePoint) {
        bool found = false;
        for (auto&& i : MaintenancePoints) {
            if (i == MaintenancePoint) {
                found = true;
                break;
            }
        }
        if (!found) {
            return false;
        }
    }
    if (MaintenancePoints.empty()) {
        return false;
    }
    JREAD_DOUBLE_OPT(jsonInfo, "maintenance_precision", MaintenancePrecisionHistory);
    JREAD_DOUBLE_OPT(jsonInfo, "maintenance_precision_new", MaintenancePrecisionNew);
    return true;
}

NJson::TJsonValue TRTMajorMaintenanceWatcher::DoSerializeToJson() const {
    NJson::TJsonValue result = TBase::DoSerializeToJson();
    JWRITE(result, "maintenance_precision", MaintenancePrecisionHistory);
    JWRITE(result, "maintenance_precision_new", MaintenancePrecisionNew);
    JWRITE(result, "maintenance_point_int", MaintenancePoint);
    TJsonProcessor::WriteContainerArray(result, "maintenance_points", MaintenancePoints);
    return result;
}

TExpectedState TRTMajorTOSynchronizerWatcher::DoExecute(TAtomicSharedPtr<IRTBackgroundProcessState> /*state*/, const TExecutionContext& context) const {
    const NDrive::IServer& frServer = context.GetServerAs<NDrive::IServer>();
    TVector<TMaintenanceInfo> infos;
    TMessagesCollector errors;
    const TMajorClient& client = frServer.GetDriveAPI()->GetMajorClient();
    if (!client.GetMaintenanceInfo(infos, errors)) {
        return MakeUnexpected(errors.GetStringReport());
    }
    NDrive::TEntitySession session = frServer.GetDriveAPI()->template BuildTx<NSQL::Writable>();
    for (auto&& i : infos) {
        i.SetSource("major");
        if (!frServer.GetDriveAPI()->GetMaintenanceDB().UpsertMaintenanceInfo(i, GetRobotUserId(), true, session)) {
            return MakeUnexpected(session.GetStringReport());
        }
    }
    if (!session.Commit()) {
        return MakeUnexpected(session.GetStringReport());
    }
    return new IRTBackgroundProcessState;
}

class TSpecifications {
public:
    template <class TAction>
    bool Compare(TAction&& action) const {
        for (const auto& specification : Impl) {
            if (action(specification)) {
                return true;
            }
        }
        return false;
    }

    bool Parse(const NJson::TJsonValue& json) {
        if (!json.IsArray()) {
            return false;
        }
        const auto& jsonArray = json.GetArray();
        if (jsonArray.size() < 1) {
            return false;
        }
        for (const auto& elem : jsonArray) {
            NMajorClient::TCarInfo::TConfiguration configuration;
            if (!configuration.SimpleParse(elem)) {
                return false;
            }
            Impl.emplace_back(std::move(configuration));
        }
        return true;
    }

private:
    TVector<NMajorClient::TCarInfo::TConfiguration> Impl;
};

template <>
bool NJson::TryFromJson<TSpecifications>(const NJson::TJsonValue& value, TSpecifications& result) {
    return result.Parse(value);
}

class TCarModelSpecification {
public:
    class TAdditionalTags {
    public:
        R_READONLY(TSpecifications, Specifications);
        R_READONLY(TVector<TString>, Tags);

    public:
        bool Parse(const NJson::TJsonValue& json) {
            return NJson::ParseField(json, "specifications", Specifications, true) && NJson::ParseField(json, "tags", Tags, true);
        }
    };

    R_READONLY(TSpecifications, Specifications);
    R_READONLY(TVector<TAdditionalTags>, AdditionalTags);
    R_READONLY(bool, Deprecated, false);

public:
    bool Parse(const NJson::TJsonValue& json) {
        return NJson::ParseField(json, "deprecated", Deprecated, false) && NJson::ParseField(json, "specifications", Specifications, true) && NJson::ParseField(json, "additional_tags", AdditionalTags);
    }
};

template <>
bool NJson::TryFromJson<TCarModelSpecification::TAdditionalTags>(const NJson::TJsonValue& value, TCarModelSpecification::TAdditionalTags& result) {
    return result.Parse(value);
}

bool InitModelSpecifications(const TString& source, const NDrive::IServer& server, TMap<TString, TCarModelSpecification>& models, TString& error) {
    TString modelsStr;
    NJson::TJsonValue modelsJson;
    NJson::TJsonValue::TMapType modelsJsonMap;
    if (!server.GetSettings().GetValue(source, modelsStr) || !NJson::ReadJsonTree(modelsStr, &modelsJson) || !modelsJson.GetMap(&modelsJsonMap)) {
        error = "Некорректная спецификация моделей";
        return false;
    }

    for (const auto& model : modelsJsonMap) {
        TCarModelSpecification configuration;
        if (!configuration.Parse(model.second)) {
            error = "Некорректная спецификация модели: " + model.second.GetStringRobust();
            return false;
        }
        models.emplace(model.first, std::move(configuration));
    }
    return true;
}

NDrive::TScheme TRTMajorNewCarProcess::DoGetScheme(const IServerBase& server) const {
    NDrive::TScheme scheme = TBase::DoGetScheme(server);
    scheme.Add<TFSVariants>("notifier", "Способ нотификации").SetVariants(server.GetNotifierNames());
    scheme.Add<TFSBoolean>("notification_only", "Только нотификация").SetDefault(true);
    return scheme;
}

bool TRTMajorNewCarProcess::DoDeserializeFromJson(const NJson::TJsonValue& jsonInfo) {
    JREAD_STRING(jsonInfo, "notifier", NotifierName);
    JREAD_BOOL(jsonInfo, "notification_only", NotificationOnly);
    return TBase::DoDeserializeFromJson(jsonInfo);
}

NJson::TJsonValue TRTMajorNewCarProcess::DoSerializeToJson() const {
    NJson::TJsonValue result = TBase::DoSerializeToJson();
    TJsonProcessor::Write(result, "notifier", NotifierName);
    TJsonProcessor::Write(result, "notification_only", NotificationOnly);
    return result;
}

TExpectedState TRTMajorNewCarProcess::DoExecute(TAtomicSharedPtr<IRTBackgroundProcessState> /*state*/, const TExecutionContext& context) const {
    const NDrive::IServer& frServer = context.GetServerAs<NDrive::IServer>();
    if (!frServer.GetDriveAPI()->HasMajorClient()) {
        return MakeUnexpected<TString>({});
    }

    TMap<TString, TCarModelSpecification> modelSpecifications;
    {
        TString error;
        if (!InitModelSpecifications("models_specialization", frServer, modelSpecifications, error)) {
            NDrive::INotifier::Notify(frServer.GetNotifier(GetNotifierName()), error);
            return MakeUnexpected<TString>({});
        }
    }

    TMessagesCollector errors;
    TVector<NMajorClient::TCarInfo> infos;
    if (!frServer.GetDriveAPI()->GetMajorClient().GetReservedInfo(infos, errors)) {
        NDrive::INotifier::Notify(frServer.GetNotifier(GetNotifierName()), errors.GetStringReport());
        return MakeUnexpected<TString>({});
    }

    NDrive::TEntitySession session = frServer.GetDriveAPI()->template BuildTx<NSQL::Writable>();
    auto allVins = frServer.GetDriveAPI()->GetCarsData()->GetAllVins(session);
    auto gModelsData = frServer.GetDriveAPI()->GetModelsData()->FetchInfo(session);

    TVector<TString> messages;
    for (const auto& car : infos) {
        if (!car.GetVin()) {
            continue;
        }
        auto vin = ToUpperUTF8(StripString(car.GetVin()));
        if (allVins.contains(vin)) {
            continue;
        }
        const auto& configuration = car.GetConfiguration();
        TString modelCode;
        TCarModelSpecification modelSpecification;
        bool doubleSpecification = false;
        for (const auto& specification : modelSpecifications) {
            if (specification.second.GetSpecifications().Compare([&configuration](const NMajorClient::TCarInfo::TConfiguration& modelConfiguration) { return modelConfiguration.Like(configuration); })) {
                if (specification.second.GetDeprecated()) {
                    continue;
                }
                if (modelCode) {
                    messages.push_back(vin + ": Обнаружено несколько подходящих моделей для " + car.GetConfiguration().ToJson().GetStringRobust() + " (" + modelCode + "-" + specification.first + ")");
                    doubleSpecification = true;
                    break;
                } else {
                    modelCode = specification.first;
                    modelSpecification = specification.second;
                }
            }
        }
        if (doubleSpecification) {
            continue;
        }
        if (!modelCode) {
            messages.push_back(vin + ": Не обнаружено модели для " + car.GetConfiguration().ToJson().GetStringRobust());
            continue;
        }

        auto modelPtr = gModelsData.GetResultPtr(modelCode);
        if (!modelPtr) {
            messages.push_back(vin + ": Некорректная модель " + modelCode);
            continue;
        }

        messages.push_back("Новое авто: " + vin + " " + modelCode);
        if (NotificationOnly) {
            continue;
        }

        NJson::TJsonValue minCarJson;
        TString carId = NUtil::CreateUUID();

        minCarJson["id"] = carId;
        minCarJson["vin"] = vin;
        minCarJson["model_code"] = modelCode;
        if (car.GetNumber()) {
            minCarJson["number"] = ToLowerUTF8(car.GetNumber());
        }

        if (!frServer.GetDriveAPI()->GetCarsData()->UpdateCarFromJSON(minCarJson, carId, GetRobotUserId(), session)) {
            NDrive::INotifier::Notify(frServer.GetNotifier(GetNotifierName()), vin + ": " + session.GetStringReport());
            return MakeUnexpected<TString>({});
        }

        for (auto&& defaultTag : modelPtr->GetDefaultTags()) {
            ITag::TPtr tagData = frServer.GetDriveAPI()->GetTagsManager().GetTagsMeta().CreateTag(defaultTag.TagName, GetRTProcessName());
            tagData->SetTagPriority(defaultTag.Priority);
            if (!frServer.GetDriveAPI()->GetTagsManager().GetDeviceTags().AddTag(tagData, GetRobotUserId(), carId, &frServer, session)) {
                NDrive::INotifier::Notify(frServer.GetNotifier(GetNotifierName()), vin + ": " + session.GetStringReport());
                return MakeUnexpected<TString>({});
            }
        }

        for (const auto& defaultTags : modelSpecification.GetAdditionalTags()) {
            if (defaultTags.GetSpecifications().Compare([&configuration](const NMajorClient::TCarInfo::TConfiguration& modelConfiguration) { return modelConfiguration.SimpleLike(configuration); })) {
                for (const auto& tag : defaultTags.GetTags()) {
                    ITag::TPtr tagData = frServer.GetDriveAPI()->GetTagsManager().GetTagsMeta().CreateTag(tag, GetRTProcessName());
                    if (!frServer.GetDriveAPI()->GetTagsManager().GetDeviceTags().AddTag(tagData, GetRobotUserId(), carId, &frServer, session)) {
                        NDrive::INotifier::Notify(frServer.GetNotifier(GetNotifierName()), vin + ": " + session.GetStringReport());
                        return MakeUnexpected<TString>({});
                    }
                }
            }
        }

        THolder<TCarRegistryDocument> regDocument(new TCarRegistryDocument());
        if (regDocument->PatchWithCarJSON(minCarJson, frServer.GetDriveAPI()->GetModelsData())) {
            TCarGenericAttachment attachment(regDocument.Release());
            if (!frServer.GetDriveAPI()->GetCarAttachmentAssignments().Attach(attachment, carId, GetRobotUserId(), session, &frServer)) {
                NDrive::INotifier::Notify(frServer.GetNotifier(GetNotifierName()), vin + ": " + session.GetStringReport());
                return MakeUnexpected<TString>({});
            }
        } else {
            messages.push_back(vin + ": Не удалось применить изменения");
            continue;
        }
    }

    if (!session.Commit()) {
        NDrive::INotifier::Notify(frServer.GetNotifier(GetNotifierName()), session.GetStringReport());
        return MakeUnexpected<TString>({});
    }

    NDrive::INotifier::MultiLinesNotify(frServer.GetNotifier(GetNotifierName()), "Обнаружены новые авто " + ToString(messages.size()), messages);
    return new IRTBackgroundProcessState;
}

NDrive::TScheme TRTMajorCheckCarsProcess::DoGetScheme(const IServerBase& server) const {
    NDrive::TScheme scheme = TBase::DoGetScheme(server);
    scheme.Add<TFSVariants>("notifier", "Способ нотификации").SetVariants(server.GetNotifierNames());
    return scheme;
}

bool TRTMajorCheckCarsProcess::DoDeserializeFromJson(const NJson::TJsonValue& jsonInfo) {
    JREAD_STRING(jsonInfo, "notifier", NotifierName);
    return TBase::DoDeserializeFromJson(jsonInfo);
}

NJson::TJsonValue TRTMajorCheckCarsProcess::DoSerializeToJson() const {
    NJson::TJsonValue result = TBase::DoSerializeToJson();
    TJsonProcessor::Write(result, "notifier", NotifierName);
    return result;
}

bool CheckCar(const NMajorClient::TCarInfo& majorCar, const TDriveCarInfo& carInfo, const TMap<TString, TCarModelSpecification>& modelSpecifications, TString& error) {
    TSet<TString> majorModelCode;
    bool deprecated = true;
    const auto& configuration = majorCar.GetConfiguration();
    for (const auto& specification : modelSpecifications) {
        if (specification.second.GetSpecifications().Compare([&configuration](const NMajorClient::TCarInfo::TConfiguration& modelConfiguration) { return modelConfiguration.Like(configuration); })) {
            if (!deprecated && !specification.second.GetDeprecated()) {
                error = "Обнаружено несколько подходящих моделей для " + majorCar.GetConfiguration().ToJson().GetStringRobust() + " (" + JoinSeq(",", majorModelCode) + "-" + specification.first + ")";
                return false;
            } else {
                deprecated &= specification.second.GetDeprecated();
            }
            majorModelCode.emplace(specification.first);
        }
    }
    if (majorModelCode.empty()) {
        error = "Не обнаружено модели для " + majorCar.GetConfiguration().ToJson().GetStringRobust();
        return false;
    }
    if (!majorModelCode.contains(carInfo.GetModel())) {
        error = "Несоответстие модели " + JoinSeq(",", majorModelCode) + "-" + carInfo.GetModel();
        return false;
    }
    TString majorNumber = ToLowerUTF8(StripString(majorCar.GetNumber()));
    if (carInfo.GetNumber() && majorNumber != carInfo.GetNumber()) {
        error = "Несоответстие грз " + majorNumber + "-" + carInfo.GetNumber();
        return false;
    }
    return true;
}

TExpectedState TRTMajorCheckCarsProcess::DoExecuteFiltered(TAtomicSharedPtr<IRTBackgroundProcessState> /*state*/, const NDrive::IServer& server, TTagsModificationContext& context) const {
    if (!server.GetDriveAPI()->HasMajorClient()) {
        return MakeUnexpected<TString>({});
    }

    TMap<TString, TCarModelSpecification> modelSpecifications;
    {
        TString error;
        if (!InitModelSpecifications("models_specialization", server, modelSpecifications, error)) {
            NDrive::INotifier::Notify(server.GetNotifier(GetNotifierName()), error);
            return MakeUnexpected<TString>({});
        }
    }

    TMessagesCollector errors;
    TVector<NMajorClient::TCarInfo> reservedInfos;
    if (!server.GetDriveAPI()->GetMajorClient().GetReservedInfo(reservedInfos, errors)) {
        NDrive::INotifier::Notify(server.GetNotifier(GetNotifierName()), errors.GetStringReport());
        return MakeUnexpected<TString>({});
    }
    TVector<NMajorClient::TIssuedCarInfo> issuedInfos;
    if (!server.GetDriveAPI()->GetMajorClient().GetAllCarInfo(issuedInfos, errors)) {
        NDrive::INotifier::Notify(server.GetNotifier(GetNotifierName()), errors.GetStringReport());
        return MakeUnexpected<TString>({});
    }


    auto carIds = context.GetFilteredCarIds();
    auto session = server.GetDriveAPI()->template BuildTx<NSQL::ReadOnly>();
    auto vins = server.GetDriveAPI()->GetCarsData()->GetAllVins(session);
    auto gModelsData = server.GetDriveAPI()->GetModelsData()->FetchInfo(session);

    TVector<TString> unknownMajorCars;
    TSet<TString> issuedCars;
    TSet<TString> newCars;
    for (const auto& majorCar : issuedInfos) {
        TString majorVin = ToUpperUTF8(StripString(majorCar.GetVin()));
        auto it = vins.find(majorVin);
        if (it == vins.end() && majorCar.GetIsIssued()) {
            unknownMajorCars.emplace_back(majorVin + " неизвестный выданный авто");
            continue;
        }
        if (!carIds.contains(it->second)) {
            unknownMajorCars.emplace_back(majorVin + " неверно отфильтрован или повторяется");
            continue;
        }
        carIds.erase(it->second);
        if (majorCar.GetIsIssued()) {
            issuedCars.emplace(it->second);
        } else {
            newCars.emplace(it->second);
        }

        auto carPtr = context.GetFetchedCarsData(it->second);
        if (!carPtr) {
            unknownMajorCars.emplace_back(majorVin + ": Внутренняя неконсистентность");
            continue;
        }
        TString error;
        if (!CheckCar(majorCar, *carPtr, modelSpecifications, error)) {
            unknownMajorCars.emplace_back(majorVin + "(" + carPtr->GetHRReport() + "): " + error);
        }
    }
    NDrive::INotifier::MultiLinesNotify(server.GetNotifier(GetNotifierName()), "Несоответствие данный Major по отданным машинам " + ToString(unknownMajorCars.size()), unknownMajorCars);

    TVector<TString> reservedErrors;
    for (const auto& majorCar : reservedInfos) {
        TString majorVin = ToUpperUTF8(StripString(majorCar.GetVin()));
        auto it = vins.find(majorVin);
        if (it == vins.end()) {
            continue;
        }
        if (issuedCars.contains(it->second)) {
            reservedErrors.emplace_back(majorVin + ": найден в списке выданных");
        } else {
            if (!newCars.contains(it->second)) {
                reservedErrors.emplace_back(majorVin + ": не найден в списке выданных");
            } else {
                newCars.erase(it->second);
            }
        }
    }

    if (newCars.size()) {
        TVector<TString> carInfos;
        for (const auto& id : newCars) {
            auto carPtr = context.GetFetchedCarsData(id);
            if (!carPtr) {
                reservedErrors.emplace_back(id + ": Внутренняя неконсистентность");
                continue;
            }
            carInfos.emplace_back(carPtr->GetHRReport());
        }
        NDrive::INotifier::MultiLinesNotify(server.GetNotifier(GetNotifierName()), "Найдены в полном списке машин, но не найдены в списке на выдачу " + ToString(newCars.size()), carInfos);
    }

    if (carIds.size()) {
        TVector<TString> carInfos;
        auto states = server.GetDriveAPI()->GetStateFiltersDB()->GetObjectStates();
        auto itState = states.begin();
        for (const auto& id : carIds) {
            auto carPtr = context.GetFetchedCarsData(id);
            if (!carPtr) {
                reservedErrors.emplace_back(id + ": Внутренняя неконсистентность");
                continue;
            }
            TString status = "undefined";
            if (Advance(itState, states.end(), id)) {
                status = itState->second;
            }
            carInfos.emplace_back(carPtr->GetHRReport() + " " + status);
        }
        NDrive::INotifier::MultiLinesNotify(server.GetNotifier(GetNotifierName()), "Неизвестные авто " + ToString(carIds.size()), carInfos);
    }

    NDrive::INotifier::MultiLinesNotify(server.GetNotifier(GetNotifierName()), "Несоответствие данный Major по новым машинам " + ToString(reservedErrors.size()), reservedErrors);

    return new IRTBackgroundProcessState;
}
