#include "manager.h"

#include <drive/backend/logging/events.h>

#include <rtline/library/json/exception.h>
#include <rtline/library/storage/abstract.h>
#include <rtline/library/unistat/cache.h>
#include <rtline/library/unistat/signals.h>
#include <rtline/util/types/expected.h>

#include <util/system/hostname.h>

namespace {
    const TString ExceptionStatus = "EXCEPTION";
    const TString FailureStatus = "FAILURE";
    const TString SleepingStatus = "SLEEPING";
}

bool TExecutionAgent::FlushState(const TRTBackgroundProcessStateContainer& stateContainer, bool withRetries) const {
    auto table = Database->GetTable("rt_background_state");
    auto start = Now();
    auto current = Now();
    do {
        current = Now();
        auto transaction = Database->CreateTransaction(false);
        NStorage::TTableRecord trUpdate = stateContainer.SerializeToTableRecord();

        NStorage::TTableRecord trCondition;
        trCondition.Set("bp_name", ProcessSettings.GetName());

        auto result = table->Upsert(trUpdate, transaction, trCondition);
        if (!!result && result->IsSucceed() && transaction->Commit()) {
            return true;
        } else if (current - start > TDuration::Minutes(1)) {
            ALERT_LOG << ProcessSettings.GetName() << ": cannot flush state: " << transaction->GetErrors().GetStringReport() << Endl;
        } else {
            ERROR_LOG << "Problem on FlushState: " << transaction->GetErrors().GetStringReport() << Endl;
        }
        Sleep(Config.GetFinishAttemptionsPause());
    } while (withRetries && (current - start < Config.GetFinishAttemptionsTimeout()));
    return false;
}

namespace {
    TMutex MutexUnlockableTasks;
    TMap<TString, TInstant> InstantStartUnlockableTasks;
}

bool TExecutionAgent::FinishState(IRTBackgroundProcessState::TPtr state, const TString& status, const TString& message) {
    ProcessState.SetMessage(message);
    ProcessState.SetStatus(status);
    ProcessState.SetHostName(GetFQDNHostName());
    const TInstant now = Now();
    if (ProcessSettings->IsSimultaneousProcess()) {
        TGuard<TMutex> g(MutexUnlockableTasks);
        InstantStartUnlockableTasks[ProcessSettings.GetName()] = now;
    }

    ProcessState.SetLastExecution(now);
    ProcessState.SetLastFlush(now);
    if (!!state) {
        ProcessState.SetProcessState(state);
    }
    return FlushState(ProcessState, /*withRetries=*/true);
}

void TExecutionAgent::FinishStateEx(IRTBackgroundProcessState::TPtr state, const TString& status, const TString& message) {
    if (!FinishState(state, status, message)) {
        NDrive::TEventLog::Log("FinishStateFailed", NJson::TMapBuilder
            ("state", state ? state->GetReport() : NJson::TJsonValue{})
            ("status", status)
            ("message", message)
        );
    }
}

bool TExecutionAgent::CheckState(const TRTBackgroundProcessStateContainer& processState) const {
    TInstant lastExecution;
    if (!ProcessSettings->GetTimeRestrictions().Empty() && !ProcessSettings->GetTimeRestrictions().IsActualNow(ModelingNow())) {
        DEBUG_LOG << "Too young start (time restrictions) for " << ProcessSettings.GetName() << Endl;
        return false;
    }
    if (ProcessSettings->IsSimultaneousProcess()) {
        TGuard<TMutex> g(MutexUnlockableTasks);
        auto it = InstantStartUnlockableTasks.find(ProcessSettings.GetName());
        lastExecution = (it == InstantStartUnlockableTasks.end()) ? TInstant::Zero() : it->second;
    } else {
        lastExecution = processState.GetLastExecution();
    }
    if (!!processState && ProcessSettings->GetNextStartInstant(lastExecution) > Now()) {
        DEBUG_LOG << "Too young start for " << ProcessSettings.GetName() << Endl;
        return false;
    }
    return true;
}

bool TExecutionAgent::StartExecution(const TRTBackgroundManager& owner) {
    if (!ProcessSettings->GetEnabled() || Config.GetBlockedProcesses().contains(ProcessSettings.GetName())) {
        DEBUG_LOG << "Disabled process " << ProcessSettings.GetName() << Endl;
        return false;
    }

    if (!CheckState(ProcessState)) {
        return false;
    }

    if (!ProcessSettings->IsSimultaneousProcess()) {
        ExternalLock = Checked(LockDatabase)->Lock("lock_process_" + ProcessSettings.GetName(), true, TDuration::Zero());
        ExternalLock2 = LockDatabase2 ? LockDatabase2->Lock("lock_process_" + ProcessSettings.GetName(), true, TDuration::Zero()) : nullptr;
    } else {
        ExternalLock = new NRTLine::TFakeLock("fake");
        ExternalLock2 = new NRTLine::TFakeLock("fake");
    }
    InternalLock = NNamedLock::TryAcquireLock("lock_process_" + ProcessSettings.GetName());
    if (!ExternalLock || (LockDatabase2 && !ExternalLock2) || !InternalLock) {
        INFO_LOG << "Cannot lock background process '" << ProcessSettings.GetName() << "'" << !!InternalLock << "/" << !!ExternalLock << "/" << !!ExternalLock2 << Endl;
        return false;
    }
    if (!owner.GetActualProcessState(ProcessSettings.GetName(), ProcessState) || !CheckState(ProcessState)) {
        return false;
    }
    ProcessState.SetStatus("ACTIVE");
    ProcessState.SetHostName(GetFQDNHostName());
    ProcessState.SetLastFlush(Now());
    return FlushState(ProcessState, /*withRetries=*/false);
}

void TExecutionAgent::Process(void* /*threadSpecificResource*/) {
    auto runId = ToString(MicroSeconds());
    NDrive::TEventLog::TReqIdGuard reqIdGuard(runId);
    NDrive::TEventLog::TSourceGuard sourceGuard(ProcessSettings.GetName());
    try {
        IRTBackgroundProcess::TExecutionContext context(Server);
        NDrive::TEventLog::Log("ProcessStarted", ProcessState.GetState() ? ProcessState.GetState()->GetReport() : "empty state");
        TExpected<IRTBackgroundProcessState::TPtr, TString> nextState = ProcessSettings->Execute(ProcessState.GetState(), context);
        if (!nextState) {
            ERROR_LOG << "Cannot execute process " << ProcessSettings.GetName() << ": " << nextState.GetError() << Endl;
            TUnistatSignalsCache::SignalAdd("rt-bg-" + ProcessSettings.GetName(), "fail", 1);
            NDrive::TEventLog::Log("ProcessFailed", NJson::ToJson(NJson::JsonString(nextState.GetError())));
            FinishStateEx(nullptr, FailureStatus, nextState.GetError());
        } else {
            TUnistatSignalsCache::SignalAdd("rt-bg-" + ProcessSettings.GetName(), "success", 1);
            NDrive::TEventLog::Log("ProcessSucceeded", nextState->Get() ? (*nextState)->GetReport() : NJson::TJsonValue());
            FinishStateEx(*nextState, SleepingStatus);
        }
    } catch (const std::exception& e) {
        ERROR_LOG << "Cannot execute process " << ProcessSettings.GetName() << ": " << FormatExc(e) << Endl;
        TUnistatSignalsCache::SignalAdd("rt-bg-" + ProcessSettings.GetName(), "exception", 1);
        NDrive::TEventLog::Log("ProcessException", CurrentExceptionInfo());
        FinishStateEx(nullptr, ExceptionStatus, FormatExc(e));
    }
}

TRTBackgroundManager::TRTBackgroundManager(const IServerBase& server, const TRTBackgroundManagerConfig& config)
    : TBase("rt_background_settings", MakeHolder<THistoryContext>(server.GetDatabase(config.GetDBName())), config.GetHistoryConfig())
    , Config(config)
    , TasksQueue(IThreadPool::TParams()
        .SetThreadName("rt_background_manager")
    )
    , Server(server)
    , Propositions(GetHistoryManager().GetContext(), Config.GetPropositionsConfig())
    , TasksExecutor(*this, Config)
{
    LockDb = Config.GetLockDbName() ? server.GetDatabase(Config.GetLockDbName()) : HistoryCacheDatabase;
    LockDb2 = Config.GetLockDb2Name() ? server.GetDatabase(Config.GetLockDb2Name()) : nullptr;
    AssertCorrectConfig(LockDb != nullptr, "LockDb %s not found", Config.GetLockDbName().c_str());
    AssertCorrectConfig(LockDb2 != nullptr || !Config.GetLockDb2Name(), "LockDb2 %s not found", Config.GetLockDb2Name().c_str());
}

TRTBackgroundManager::~TRTBackgroundManager() {
}

bool TRTBackgroundManager::GetActualProcessState(const TString& processName, TRTBackgroundProcessStateContainer& processState) const {
    if (!RefreshState(nullptr, States, processName)) {
        return false;
    }
    processState = GetProcessState(processName);
    return true;
}

TRTBackgroundProcessStateContainer TRTBackgroundManager::GetProcessState(const TString& processName) const {
    TRTBackgroundProcessStateContainer stateContainer;
    auto it = States.find(processName);
    if (it == States.end()) {
        stateContainer = TRTBackgroundProcessStateContainer(new IRTBackgroundProcessState());
        stateContainer.SetLastExecution(TInstant::Zero());
        stateContainer.SetProcessName(processName);
    } else {
        stateContainer = std::move(it->second);
    }
    return stateContainer;
}

bool TRTBackgroundManager::DoRunProcesses() {
    auto tasks = GetTasksCopySafe();

    if (!GetStatesInfo(nullptr, States)) {
        return false;
    }

    for (auto&& i : tasks) {
        if (!i.second->IsHostAvailable(Server)) {
            continue;
        }
        auto stateContainer = GetProcessState(i.first);
        auto ea = MakeHolder<TExecutionAgent>(
            std::move(stateContainer),
            i.second,
            Server,
            HistoryCacheDatabase,
            LockDb,
            LockDb2,
            Config
        );
        if (!ea->StartExecution(*this)) {
            continue;
        }
        TasksQueue.SafeAddAndOwn(std::move(ea));
    }

    return true;
}

bool TRTBackgroundManager::GetSettingsInfo(NDrive::TEntitySession* session, TMap<TString, TRTBackgroundProcessContainer>& settings, const NSQL::TQueryOptions& options) const {
    settings.clear();
    auto table = HistoryCacheDatabase->GetTable("rt_background_settings");
    NStorage::TObjectRecordsSet<TRTBackgroundProcessContainer> records;
    {
        auto transaction = session ? session->GetTransaction() : HistoryCacheDatabase->CreateTransaction(true);
        auto result = table->GetRows(options.PrintConditions(*transaction), records, transaction);
        if (!result->IsSucceed()) {
            ERROR_LOG << "Cannot refresh data for rt_background_manager" << Endl;
            return false;
        }
    }
    for (auto&& i : records) {
        settings.emplace(i.GetName(), std::move(i));
    }
    return true;
}

bool TRTBackgroundManager::GetSettingsInfo(NDrive::TEntitySession* session, TMap<TString, TRTBackgroundProcessContainer>& settings, const TSet<TString>& ids) const {
    NSQL::TQueryOptions options;
    if (!ids.empty()) {
        options.SetGenericCondition("bp_name", ids);
    }
    return GetSettingsInfo(session, settings, options);
}

bool TRTBackgroundManager::RefreshState(NDrive::TEntitySession* session, TMap<TString, TRTBackgroundProcessStateContainer>& states, const TString& processName) const {
    auto table = HistoryCacheDatabase->GetTable("rt_background_state");
    TRecordsSet records;
    {
        auto transaction = session ? session->GetTransaction() : HistoryCacheDatabase->CreateTransaction(false);
        auto result = table->GetRows("bp_name=" + transaction->Quote(processName), records, transaction);
        if (!result->IsSucceed()) {
            ERROR_LOG << "Cannot refresh data for rt_background_manager process " << processName << Endl;
            return false;
        }
    }
    for (auto&& i : records) {
        TRTBackgroundProcessStateContainer container;
        if (!container.DeserializeFromTableRecord(i)) {
            WARNING_LOG << "Cannot parse info from record: " << i.ToCSV(",") << Endl;
            continue;
        }
        const TString processName = container.GetProcessName();
        states[processName] = std::move(container);
    }
    return true;
}

bool TRTBackgroundManager::GetStatesInfo(NDrive::TEntitySession* session, TMap<TString, TRTBackgroundProcessStateContainer>& states, const TSet<TString>& ids) const {
    states.clear();
    auto table = HistoryCacheDatabase->GetTable("rt_background_state");
    TRecordsSet records;
    {
        auto transaction = session ? session->GetTransaction() : HistoryCacheDatabase->CreateTransaction(true);
        auto condition = ids.empty() ? TString() : NSQL::TQueryOptions::PrintCondition("bp_name", ids, *transaction);
        auto result = table->GetRows(condition, records, transaction);
        if (!result->IsSucceed()) {
            ERROR_LOG << "Cannot refresh data for rt_background_manager" << Endl;
            return false;
        }
    }
    for (auto&& i : records) {
        TRTBackgroundProcessStateContainer container;
        if (!container.DeserializeFromTableRecord(i)) {
            WARNING_LOG << "Cannot parse info from record: " << i.ToCSV(",") << Endl;
            continue;
        }
        states.emplace(container.GetProcessName(), std::move(container));
    }
    return true;
}

bool TRTBackgroundManager::RemoveBackground(const TVector<TString>& names, const TString& userId, NDrive::TEntitySession& session) const {
    auto transaction = session.GetTransaction();
    for (auto&& name : names) {
        NStorage::TTableRecord trCondition;
        trCondition.Set("bp_name", name);
        NStorage::TObjectRecordsSet<TRTBackgroundProcessContainer> records;
        auto table = HistoryCacheDatabase->GetTable("rt_background_settings");
        auto result = table->RemoveRow(trCondition, transaction, &records);
        if (!result || !result->IsSucceed() || records.size() != 1) {
            session.SetErrorInfo("cannot remove background settings", "remove", EDriveSessionResult::TransactionProblem);
            return false;
        }
        if (!HistoryManager->AddHistory(*records.begin(), userId, EObjectHistoryAction::Remove, session)) {
            return false;
        }
    }

    return true;
}

bool TRTBackgroundManager::UpsertObject(const TRTBackgroundProcessContainer& process, const TString& userId, NDrive::TEntitySession& session) const {
    if (!process) {
        session.SetErrorInfo("incorrect user data", "upsert background settings", EDriveSessionResult::DataCorrupted);
        return false;
    }

    NStorage::TTableRecord trUpdate = process.SerializeToTableRecord();
    NStorage::TTableRecord trCondition;
    trCondition.Set("bp_name", process.GetName());

    auto table = HistoryCacheDatabase->GetTable("rt_background_settings");

    NStorage::TObjectRecordsSet<TRTBackgroundProcessContainer> records;
    bool isUpdate = false;
    switch (table->UpsertWithRevision(trUpdate, session.GetTransaction(), trCondition, process.OptionalRevision(), "bp_revision", &records)) {
        case NStorage::ITableAccessor::EUpdateWithRevisionResult::IncorrectRevision:
            session.SetErrorInfo("upsert_bp_settings", "incorect_revision", EDriveSessionResult::InconsistencyUser);
            return false;
        case NStorage::ITableAccessor::EUpdateWithRevisionResult::Failed:
            session.SetErrorInfo("upsert_bp_settings", "UpsertWithRevision failure", EDriveSessionResult::TransactionProblem);
            return false;
        case NStorage::ITableAccessor::EUpdateWithRevisionResult::Updated:
            isUpdate = true;
        case NStorage::ITableAccessor::EUpdateWithRevisionResult::Inserted:
            break;
    }

    if (!HistoryManager->AddHistory(records.GetObjects(), userId, isUpdate ? EObjectHistoryAction::UpdateData : EObjectHistoryAction::Add, session)) {
        return false;
    }

    return true;
}

bool TRTBackgroundManager::ForceUpsertObject(const TRTBackgroundProcessContainer& process, const TString& userId, NDrive::TEntitySession& session, NStorage::TObjectRecordsSet<TRTBackgroundProcessContainer>* containerExt /* = nullptr */) const {
    if (!process) {
        session.SetErrorInfo("incorrect user data", "upsert background settings", EDriveSessionResult::DataCorrupted);
        return false;
    }

    NStorage::TTableRecord trUpdate = process.SerializeToTableRecord();
    auto transaction = session.GetTransaction();
    NStorage::TTableRecord trCondition;
    trCondition.Set("bp_name", process.GetName());
    NStorage::TObjectRecordsSet<TRTBackgroundProcessContainer> containerLocal;
    NStorage::TObjectRecordsSet<TRTBackgroundProcessContainer>& records = containerExt ? *containerExt : containerLocal;
    auto table = HistoryCacheDatabase->GetTable("rt_background_settings");
    NStorage::IQueryResult::TPtr result;
    bool isUpdate;
    trUpdate.ForceSet("bp_revision", "nextval('rt_background_settings_bp_revision_seq')");
    result = table->Upsert(trUpdate, transaction, trCondition, &isUpdate, &records);
    if (!result || !result->IsSucceed() || records.size() != 1) {
        session.SetErrorInfo("cannot upsert background settings (may be incorrect revision)", session.GetStringReport(), EDriveSessionResult::InconsistencyUser);
        return false;
    }
    if (!HistoryManager->AddHistory(*records.begin(), userId, isUpdate ? EObjectHistoryAction::UpdateData : EObjectHistoryAction::Add, session)) {
        return false;
    }

    return true;
}

bool TRTBackgroundManager::RemoveObject(const TSet<typename TRTBackgroundProcessContainer::TId>& ids, const TString& userId, NDrive::TEntitySession& session) const {
    TMap<TString, TRTBackgroundProcessContainer> settings;
    if (!GetSettingsInfo(&session, settings, ids)) {
        return false;
    }
    return RemoveBackground(MakeVector(NContainer::Keys(settings)), userId, session);
}

const IPropositionsManager<TRTBackgroundProcessContainer>* TRTBackgroundManager::GetPropositions() const {
    return &Propositions;
}

bool TRTBackgroundManager::GetObjects(TMap<typename TRTBackgroundProcessContainer::TId, TRTBackgroundProcessContainer>& /* objects */, const TInstant /* reqActuality = TInstant::Zero() */) const {
    return false;
}

bool TRTBackgroundManager::AddObjects(const TVector<TRTBackgroundProcessContainer>& /* objects */, const TString& /* userId */, NDrive::TEntitySession& /* session */, NStorage::TObjectRecordsSet<TRTBackgroundProcessContainer>* /* containerExt = nullptr */) const {
    return false;
}
