#pragma once

#include "config.h"
#include "settings.h"
#include "state.h"

#include <drive/backend/database/history/cache.h>
#include <drive/backend/database/history/manager.h>
#include <drive/backend/database/history/db_entities.h>

#include <library/cpp/threading/named_lock/named_lock.h>

#include <rtline/library/storage/abstract.h>

#include <util/thread/pool.h>

class TRTBackgroundManager;

class TExecutionAgent: public IObjectInQueue {
private:
    TRTBackgroundProcessStateContainer ProcessState;
    TRTBackgroundProcessContainer ProcessSettings;
    NRTLine::TAbstractLock::TPtr ExternalLock;
    NRTLine::TAbstractLock::TPtr ExternalLock2;
    NNamedLock::TNamedLockPtr InternalLock;
    const IServerBase& Server;
    NStorage::IDatabase::TPtr Database;
    NStorage::IDatabase::TPtr LockDatabase;
    NStorage::IDatabase::TPtr LockDatabase2;
    const TRTBackgroundManagerConfig& Config;

private:
    bool FlushState(const TRTBackgroundProcessStateContainer& stateContainer, bool withRetries) const;
    bool FinishState(IRTBackgroundProcessState::TPtr state, const TString& status, const TString& message = {});
    void FinishStateEx(IRTBackgroundProcessState::TPtr state, const TString& status, const TString& message = {});

protected:
    bool CheckState(const TRTBackgroundProcessStateContainer& processState) const;

public:
    TExecutionAgent(
        TRTBackgroundProcessStateContainer&& processState,
        const TRTBackgroundProcessContainer& processSettings,
        const IServerBase& server,
        NStorage::IDatabase::TPtr db,
        NStorage::IDatabase::TPtr lockDb,
        NStorage::IDatabase::TPtr lockDb2,
        const TRTBackgroundManagerConfig& config
    )
        : ProcessState(std::move(processState))
        , ProcessSettings(processSettings)
        , Server(server)
        , Database(db)
        , LockDatabase(lockDb)
        , LockDatabase2(lockDb2)
        , Config(config)
    {
    }

    bool StartExecution(const TRTBackgroundManager& owner);

    virtual void Process(void* /*threadSpecificResource*/) override;
};

class TBackgroundPropositions: public TPropositionsManager<TRTBackgroundProcessContainer> {
private:
    using TBase = TPropositionsManager<TRTBackgroundProcessContainer>;

public:
    TBackgroundPropositions(const IHistoryContext& context, const TPropositionsManagerConfig& config)
        : TBase(context, "rt_background_settings_propositions", config)
    {
    }
};

class TBackgroundHistoryManager: public TIndexedAbstractHistoryManager<TRTBackgroundProcessContainer> {
private:
    using TBase = TIndexedAbstractHistoryManager<TRTBackgroundProcessContainer>;
protected:
    using TBase::TBase;

public:
    TBackgroundHistoryManager(const IHistoryContext& context, const THistoryConfig& config)
        : TBase(context, "rt_background_settings_history", config)
    {
    }
};

class TRTBackgroundManager
    : public TDBCacheWithHistoryOwner<TBackgroundHistoryManager, TRTBackgroundProcessContainer>
    , public IDBEntitiesWithPropositionsManager<TRTBackgroundProcessContainer>
    , public TNonCopyable
{
private:
    using TBase = TDBCacheWithHistoryOwner<TBackgroundHistoryManager, TRTBackgroundProcessContainer>;

public:
    R_CONST(TRTBackgroundManagerConfig, Config);

private:
    TThreadPool TasksQueue;
    const IServerBase& Server;
    TBackgroundPropositions Propositions;

    NStorage::IDatabase::TPtr LockDb;
    NStorage::IDatabase::TPtr LockDb2;

    mutable TMap<TString, TRTBackgroundProcessStateContainer> States;

private:
    class TTasksExecutor: public IAutoActualization {
    private:
        TRTBackgroundManager& Owner;

    public:
        virtual bool Refresh() override {
            return Owner.DoRunProcesses();
        }

        TTasksExecutor(TRTBackgroundManager& owner, const TRTBackgroundManagerConfig& config)
            : IAutoActualization("TRTBackgroundManager::TTasksExecutor", config.GetPingPeriod())
            , Owner(owner)
        {
        }
    };

private:
    TTasksExecutor TasksExecutor;

private:
    virtual bool DoRebuildCacheUnsafe() const override {
        if (!GetSettingsInfo(nullptr, Objects)) {
            return false;
        }
        for (auto&& i : Objects) {
            i.second->Start(i.second);
        }
        return true;
    }

    virtual TStringBuf GetEventObjectId(const TObjectEvent<TRTBackgroundProcessContainer>& ev) const override {
        return ev.GetName();
    }

    virtual void AcceptHistoryEventUnsafe(const TObjectEvent<TRTBackgroundProcessContainer>& ev) const override {
        if (ev.GetHistoryAction() == EObjectHistoryAction::Remove) {
            Objects.erase(ev.GetName());
        } else {
            auto container = ev.Clone();
            if (!container) {
                Objects.erase(ev.GetName());
            } else {
                Objects[ev.GetName()] = std::move(container);
                Objects[ev.GetName()]->Start(Objects[ev.GetName()]);
            }
        }
    }

    bool DoRunProcesses();

    bool DoStart() override {
        TasksQueue.Start(Config.GetThreadsCount());
        if (!TBase::DoStart()) {
            return false;
        }
        return TasksExecutor.Start();
    }

    bool DoStop() override {
        if (!TasksExecutor.Stop()) {
            return false;
        }
        if (!TBase::DoStop()) {
            return false;
        }
        TasksQueue.Stop();
        return true;
    }

    const TObjects GetTasksCopySafe() const {
        auto rg = MakeObjectReadGuard();
        return Objects;
    }

    TRTBackgroundProcessStateContainer GetProcessState(const TString& processName) const;

public:
    TRTBackgroundManager(const IServerBase& server, const TRTBackgroundManagerConfig& config);
    ~TRTBackgroundManager();

    bool GetActualProcessState(const TString& processName, TRTBackgroundProcessStateContainer& processState) const;

    bool GetSettingsInfo(NDrive::TEntitySession* session, TMap<TString, TRTBackgroundProcessContainer>& settings, const NSQL::TQueryOptions& options) const;
    bool GetSettingsInfo(NDrive::TEntitySession* session, TMap<TString, TRTBackgroundProcessContainer>& settings, const TSet<TString>& ids = {}) const;
    bool GetStatesInfo(NDrive::TEntitySession* session, TMap<TString, TRTBackgroundProcessStateContainer>& states, const TSet<TString>& ids = {}) const;
    bool RefreshState(NDrive::TEntitySession* session, TMap<TString, TRTBackgroundProcessStateContainer>& states, const TString& processName) const;

    bool RemoveBackground(const TVector<TString>& names, const TString& userId, NDrive::TEntitySession& session) const;

public:
    bool RemoveObject(const TSet<typename TRTBackgroundProcessContainer::TId>& ids, const TString& userId, NDrive::TEntitySession& session) const override;
    bool UpsertObject(const TRTBackgroundProcessContainer& object, const TString& userId, NDrive::TEntitySession& session) const override;
    bool ForceUpsertObject(const TRTBackgroundProcessContainer& object, const TString& userId, NDrive::TEntitySession& session, NStorage::TObjectRecordsSet<TRTBackgroundProcessContainer>* containerExt = nullptr) const override;
    const IPropositionsManager<TRTBackgroundProcessContainer>* GetPropositions() const override;

private:
    bool AddObjects(const TVector<TRTBackgroundProcessContainer>& objects, const TString& userId, NDrive::TEntitySession& session, NStorage::TObjectRecordsSet<TRTBackgroundProcessContainer>* containerExt = nullptr) const override;
    bool GetObjects(TMap<typename TRTBackgroundProcessContainer::TId, TRTBackgroundProcessContainer>& objects, const TInstant reqActuality = TInstant::Zero()) const override;
};
