#include "settings.h"

#include "state.h"

#include <drive/backend/logging/events.h>

#include <drive/library/cpp/scheme/scheme.h>

#include <kernel/daemon/common/time_guard.h>

#include <rtline/library/unistat/cache.h>
#include <util/generic/guid.h>

bool TRTBackgroundProcessContainer::DeserializeWithDecoder(const TDecoder& decoder, const TConstArrayRef<TStringBuf>& values, const IHistoryContext* /*hContext*/) {
    READ_DECODER_VALUE(decoder, values, Id);
    READ_DECODER_VALUE_DEF(decoder, values, Revision, Max<ui64>());
    READ_DECODER_VALUE(decoder, values, Name);
    if (!Name || (!Id && decoder.GetStrict() && Revision != Max<ui64>())) {
        return false;
    }
    TString typeName;
    READ_DECODER_VALUE_TEMP(decoder, values, typeName, Type);

    ProcessSettings = IRTBackgroundProcess::TFactory::Construct(typeName);
    if (!ProcessSettings) {
        return false;
    }
    ProcessSettings->SetRTProcessName(Name);
    NJson::TJsonValue jsonMeta;
    READ_DECODER_VALUE_JSON(decoder, values, jsonMeta, Meta);
    if (!ProcessSettings->DeserializeFromJson(jsonMeta)) {
        return false;
    }
    return true;
}

TRTBackgroundProcessContainer TRTBackgroundProcessContainer::Clone() const {
    TRTBackgroundProcessContainer result;
    if (TBaseDecoder::DeserializeFromJson(result, SerializeToTableRecord().SerializeToJson())) {
        return result;
    }
    Y_ASSERT(false);
    ALERT_LOG << "Failed deep copy for " << GetId() << Endl;
    return TRTBackgroundProcessContainer();
}

NJson::TJsonValue TRTBackgroundProcessContainer::GetReport(bool compact) const {
    Y_ENSURE_BT(ProcessSettings);
    NJson::TJsonValue result(NJson::JSON_MAP);
    NJson::InsertNonNull(result, "bp_id", Id);
    NJson::InsertField(result, "bp_name", Name);
    NJson::InsertField(result, "bp_settings", ProcessSettings->GetReport(compact));
    NJson::InsertField(result, "bp_type", ProcessSettings->GetType());
    if (HasRevision()) {
        NJson::InsertField(result, "bp_revision", GetRevision());
    }
    return result;
}

NDrive::TScheme TRTBackgroundProcessContainer::GetScheme(const IServerBase& server) const {
    CHECK_WITH_LOG(ProcessSettings);
    NDrive::TScheme result;
    result.Add<TFSIgnore>("bp_id");
    result.Add<TFSNumeric>("bp_revision", "Версия конфигурации").SetReadOnly(true);
    result.Add<TFSString>("bp_name", "Идентификатор процесса");
    result.Add<TFSStructure>("bp_settings", "Настройки").SetStructure(ProcessSettings->GetScheme(server));
    return result;
}

NStorage::TTableRecord TRTBackgroundProcessContainer::SerializeToTableRecord() const {
    CHECK_WITH_LOG(!!ProcessSettings);
    NStorage::TTableRecord result;
    result.Set("bp_name", Name);
    if (!!Id) {
        result.Set("bp_id", Id);
    }
    result.Set("bp_type", ProcessSettings->GetType());
    result.Set("bp_settings", ProcessSettings->SerializeToJson());
    if (HasRevision()) {
        result.Set("bp_revision", GetRevision());
    }
    return result;
}

bool TRTBackgroundProcessContainer::CheckOwner(const TString& userId) const {
    return ProcessSettings->GetOwners().empty() || ProcessSettings->GetOwners().contains(userId);
}

NDrive::TScheme IRTRegularBackgroundProcess::DoGetScheme(const IServerBase& server) const {
    NDrive::TScheme result = TBase::DoGetScheme(server);
    auto gTab = result.StartTabGuard("execution");
    result.Add<TFSString>("timetable", "Расписание hhmm,hhmm,hhmm... запуска процесса", 90001);
    result.Add<TFSDuration>("period", "Период запуска процесса (если не указано расписание)", 90000).SetDefault(TDuration::Minutes(5));
    result.Add<TFSString>("robot_user_id", "Пользователь", 90000);
    return result;
}

NJson::TJsonValue IRTRegularBackgroundProcess::DoSerializeToJson() const {
    NJson::TJsonValue result = TBase::DoSerializeToJson();
    if (Period != TDuration::Max()) {
        TJsonProcessor::WriteDurationString(result, "period", Period);
    }
    NJson::InsertNonNull(result, "robot_user_id", RobotUserId);
    if (Timetable.size()) {
        JWRITE_DEF(result, "timetable", JoinSeq(",", Timetable), "");
    }
    return result;
}

bool IRTRegularBackgroundProcess::DoDeserializeFromJson(const NJson::TJsonValue& jsonInfo) {
    JREAD_DURATION_OPT(jsonInfo, "period", Period);
    if (!NJson::ParseField(jsonInfo["robot_user_id"], RobotUserId)) {
        return false;
    }
    if (RobotUserId && GetUuid(*RobotUserId).IsEmpty()) {
        return false;
    }
    TString timetableStr;
    JREAD_STRING_OPT(jsonInfo, "timetable", timetableStr);
    TSet<TString> timetableStrings;
    StringSplitter(timetableStr).SplitBySet(", ").SkipEmpty().Collect(&timetableStrings);
    for (auto&& i : timetableStrings) {
        ui16 tt;
        if (!TryFromString(i, tt)) {
            return false;
        }
        Timetable.emplace(tt);
    }
    if (Timetable.empty() && Period == TDuration::Zero()) {
        return false;
    }
    return TBase::DoDeserializeFromJson(jsonInfo);
}

TInstant IRTRegularBackgroundProcess::GetNextStartInstant(const TInstant lastCallInstant) const {
    if (Timetable.size()) {
        ui16 time = lastCallInstant.Minutes() % 60;
        time += (lastCallInstant.Hours() % 24) * 100;
        DEBUG_LOG << time << Endl;
        TMaybe<TInstant> result;
        for (auto&& i : Timetable) {
            if (i > time) {
                result = TInstant::Days(lastCallInstant.Days()) + TDuration::Hours(i / 100) + TDuration::Minutes(i % 100);
                break;
            }
        }
        if (!result) {
            result = TInstant::Days(lastCallInstant.Days() + 1) + TDuration::Hours(*Timetable.begin() / 100) + TDuration::Minutes(*Timetable.begin() % 100);
        }
        DEBUG_LOG << result->Seconds() << Endl;
        return Max(Now(), *result);
    } else {
        return lastCallInstant + GetPeriod();
    }
}

bool IRTBackgroundProcess::DoStart(const TRTBackgroundProcessContainer& container) {
    RTProcessName = container.GetName();
    return true;
}

bool IRTBackgroundProcess::DeserializeFromJson(const NJson::TJsonValue& jsonSettings) {
    JREAD_STRING_OPT(jsonSettings, "bp_description", Description);
    JREAD_BOOL_OPT(jsonSettings, "bp_enabled", Enabled);
    JREAD_DURATION_OPT(jsonSettings, "freshness", Freshness);
    if (jsonSettings.Has("host_filter") && !HostFilter.DeserializeFromJson(jsonSettings["host_filter"])) {
        return false;
    }
    if (jsonSettings.Has("bp_owners")) {
        if (!jsonSettings["bp_owners"].IsArray()) {
            return false;
        }
        for (auto&& i : jsonSettings["bp_owners"].GetArraySafe()) {
            TGUID tester;
            if (!GetUuid(i.GetString(), tester)) {
                return false;
            }
            Owners.emplace(i.GetString());
        }
    }
     if (jsonSettings.Has("time_restrictions")) {
        if (!TimeRestrictions.DeserializeFromJson(jsonSettings["time_restrictions"])) {
            return false;
        }
    }
    return TAttributedEntity::DeserializeAttributes(jsonSettings) && DoDeserializeFromJson(jsonSettings);
}

NJson::TJsonValue IRTBackgroundProcess::SerializeToJson() const {
    NJson::TJsonValue result = DoSerializeToJson();
    JWRITE(result, "bp_description", Description);
    JWRITE(result, "bp_enabled", Enabled);
    TJsonProcessor::WriteContainerArray(result, "bp_owners", Owners, false);
    TJsonProcessor::WriteDurationString(result, "freshness", Freshness);
    if (!TimeRestrictions.Empty()) {
        JWRITE(result, "time_restrictions", TimeRestrictions.SerializeToJson());
    }
    JWRITE(result, "host_filter", HostFilter.SerializeToJson());
    TAttributedEntity::SerializeAttributes(result);
    return result;
}

NJson::TJsonValue IRTBackgroundProcess::GetReport(bool compact) const {
    if (compact) {
        NJson::TJsonValue result;
        result.InsertValue("bp_enabled", Enabled);
        result.InsertValue("host_filter", HostFilter.SerializeToJson());
        return result;
    } else {
        return SerializeToJson();
    }
}

NDrive::TScheme IRTBackgroundProcess::GetScheme(const IServerBase& server) const {
    NDrive::TScheme result = DoGetScheme(server);
    auto gTab = result.StartTabGuard("execution");
    result.Add<TFSText>("bp_description", "Описание процесса", 100000).SetRequired(true);
    result.Add<TFSBoolean>("bp_enabled", "Активность", 0).SetDefault(false);
    result.Add<TFSStructure>("time_restrictions", "Ограничения времени запуска").SetStructure(TTimeRestrictionsPool<class TTimeRestriction>::GetScheme());
    const TString ctype = server.GetSettings().GetValueDef("rt_backgrounds.ctype_alias." + server.GetCType(), server.GetCType());
    result.Add<TFSStructure>("host_filter", "Фильтр хостов для исполнения процесса", 100001).SetStructure(HostFilter.GetScheme(server, ctype));
    result.Add<TFSDuration>("freshness", "Максимально допустимый возраст данных", 100002).SetDefault(TDuration::Minutes(1));
    result.Add<TFSArray>("bp_owners", "Владельцы").SetElement<TFSString>();
    TAttributedEntity::FillAttributesScheme(server, "rt_background", result);
    return result;
}

TExpectedState IRTBackgroundProcess::Execute(TAtomicSharedPtr<IRTBackgroundProcessState> state, const TExecutionContext& context) const {
    TSignalGuardLast executionSignalGuard(GetRTProcessName(), "execution", 1, 0);
    StartInstant = Now();
    DataActuality = StartInstant - GetFreshness();
    return DoExecute(state, context);
}

TExpectedState IRTRegularBackgroundProcess::Execute(TAtomicSharedPtr<IRTBackgroundProcessState> state, const TExecutionContext& context) const {
    NDrive::TEventLog::TUserIdGuard userIdGuard(GetRobotUserId());
    return TBase::Execute(state, context);
}

TString IRTRegularBackgroundProcess::GetRobotId() const {
    return "rt(" + GetRTProcessName() + ")";
}

TString IRTRegularBackgroundProcess::GetRobotUserId() const {
    if (RobotUserId) {
        return *RobotUserId;
    }
    if (CachedRobotUserId) {
        return *CachedRobotUserId;
    }
    if (NDrive::HasServer()) {
        const auto& server = NDrive::GetServer();
        const auto& settings = server.GetSettings();
        auto optionalGenerateRobotUserId = settings.GetValue<bool>("rt_backgrounds." + GetType() + ".generate_robot_user_id");
        if (!optionalGenerateRobotUserId) {
            optionalGenerateRobotUserId = settings.GetValue<bool>("rt_backgrounds.default.generate_robot_user_id");
        }
        auto generateRobotUserId = optionalGenerateRobotUserId.GetOrElse(false);
        auto robotUserId = generateRobotUserId ? server.GetRtBackgroundRobotUserId(GetRTProcessName()) : Nothing();
        if (robotUserId) {
            CachedRobotUserId = robotUserId;
            return *robotUserId;
        }
    }
    return "ec65f4f0-8fa8-4887-bfdc-ca01c9906696";
}
